/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU Lesser General Public License version
 *  2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
 
#ifndef MICROFEEDSUBSCRIBER_H
#define MICROFEEDSUBSCRIBER_H

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>

#include <time.h>

/**
 * @addtogroup subscriber libmicrofeed-subscriber - Modules that are used in the subscriber side
 * @{
 * @addtogroup MicrofeedSubscriber
 *
 * Subscriber's connection to the DBus system.
 *
 * @{
 */

/**
 * Opaque data type representing Microfeed subscriber.
 */
typedef struct _MicrofeedSubscriber MicrofeedSubscriber;


/** @} */

#include <microfeed-common/microfeedstore.h>
#include <microfeed-common/microfeeditem.h>
#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-common/microfeedprotocol.h>

/**
 * @addtogroup MicrofeedSubscriber
 * @{
 */

#define MICROFEED_DBUS_OBJECT_PATH_PREFIX_SUBSCRIBER MICROFEED_DBUS_OBJECT_PATH_PREFIX "subscriber/"
#define MICROFEED_SUBSCRIBER_DBUS_METHOD_CALL_TIMEOUT 600000

typedef void (*MicrofeedSubscriberReplyCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data);
typedef void (*MicrofeedSubscriberErrorCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const char* error_name, const char* error_message, void* user_data);
typedef void (*MicrofeedSubscriberFeedCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, void* user_data);
typedef void (*MicrofeedSubscriberItemCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, void* user_data);
typedef void (*MicrofeedSubscriberItemRemovedCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, void* user_data);
typedef void (*MicrofeedSubscriberStatusCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, const MicrofeedItemStatus status, void* user_data);
typedef void (*MicrofeedSubscriberDataCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* url, const char* path, void* user_data);

typedef struct {
	MicrofeedSubscriberErrorCallback error_occured;
	MicrofeedSubscriberFeedCallback feed_update_started;
	MicrofeedSubscriberFeedCallback feed_update_ended;
	MicrofeedSubscriberFeedCallback feed_republishing_started;
	MicrofeedSubscriberFeedCallback feed_republishing_ended;	
	MicrofeedSubscriberItemCallback item_added;
	MicrofeedSubscriberItemCallback item_changed;
	MicrofeedSubscriberItemCallback item_republished;
	MicrofeedSubscriberItemRemovedCallback item_removed;
	MicrofeedSubscriberStatusCallback item_status_changed;
} MicrofeedSubscriberCallbacks;

typedef void (*MicrofeedSubscriberConfiguredSubscribeCallback)(MicrofeedSubscriber* subscriber, const char* publisher, void* callback_data);
typedef void (*MicrofeedSubscriberConfiguredUnsubscribeCallback)(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedSubscriberCallbacks* callbacks, void* callbacks_user_data, void* user_data);

MicrofeedSubscriber* microfeed_subscriber_new(const char* identifier, DBusConnection* connection);
void microfeed_subscriber_free(MicrofeedSubscriber* subscriber);

const char* microfeed_subscriber_get_identifier(MicrofeedSubscriber* subscriber);

int microfeed_subscriber_subscribe_feed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedSubscriberCallbacks* callbacks, void* callbacks_user_data, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_unsubscribe_feed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedSubscriberCallbacks* callbacks, void* callbacks_user_data, MicrofeedSubscriberReplyCallback callback, void* user_data);

int microfeed_subscriber_add_item(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_modify_item(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedItem* item, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_remove_item(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* uid, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_update_feed(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_republish_items(MicrofeedSubscriber* subscriber, const char* publisher, const char* uri, const char* start_uid, const char* end_uid, unsigned int max_count, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_get_item_data(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_mark_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_unmark_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_read_item(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* uri, const char* uid, MicrofeedSubscriberReplyCallback callback, void* user_data);

int microfeed_subscriber_store_data(MicrofeedSubscriber* subscriber, const char* publisher_identifier, const char* url, MicrofeedSubscriberErrorCallback callback, void* user_data);
void microfeed_subscriber_add_data_stored_callback(MicrofeedSubscriber* subscriber, const char* publisher_identifier, MicrofeedSubscriberDataCallback callback, void* user_data);
void microfeed_subscriber_remove_data_stored_callback(MicrofeedSubscriber* subscriber, const char* publisher_identifier, MicrofeedSubscriberDataCallback callback, void* user_data);

int microfeed_subscriber_create_publisher(MicrofeedSubscriber* subscriber, const char* publisher, MicrofeedSubscriberReplyCallback callback, void* user_data);
int microfeed_subscriber_destroy_publisher(MicrofeedSubscriber* subscriber, const char* publisher, MicrofeedSubscriberReplyCallback callback, void* user_data);

int microfeed_subscriber_handle_configured_subscriptions(MicrofeedSubscriber* subscriber, MicrofeedSubscriberConfiguredSubscribeCallback configured_subscribe, MicrofeedSubscriberConfiguredUnsubscribeCallback configured_unsubscribe, void* user_data);

/**
 * @}
 * @}
 */

#endif
