
/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-common/microfeedobject.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-common/microfeedmisc.h>
#include <stdio.h>
#include <stdlib.h>
#include <execinfo.h>

int microfeed_debug_references = 0;
int microfeed_debug_locking = 0;

struct _MicrofeedObjectImpl  {
	MicrofeedObject* object;
	MicrofeedClass* class;
	MicrofeedMutex* mutex;
	unsigned long weak_reference_count;
};

static void destroy(MicrofeedObjectImpl* object_impl);
static void print_caller(void);

MicrofeedObject* microfeed_object_new_impl(MicrofeedClass* class, size_t size, const char* class_name) {
	MicrofeedObject* object;

	object = (MicrofeedObject*)microfeed_memory_allocate_bytes(size);
	object->object_impl = microfeed_memory_allocate(MicrofeedObjectImpl);
	object->object_impl->object = object;
	object->object_impl->class = class;
	object->object_impl->mutex = microfeed_mutex_new();
	object->mutex = microfeed_mutex_new();
	object->reference_count = 1;

	if (microfeed_debug_references) {
		printf("%p NEW     %s (thread %p)\n", object, class_name, microfeed_thread_get_current());
	}

	return object;
}

MicrofeedObject* microfeed_object_ref_impl(MicrofeedObject* object, const char* class_name) {
	microfeed_assert(object->object_impl);
	microfeed_assert(!strcmp(class_name, object->object_impl->class->name));
	microfeed_assert(object->object_impl->object == object);

	microfeed_mutex_lock(object->object_impl->mutex);
	
	object->reference_count++;

	if (microfeed_debug_references) {
		printf("%p REF     %s (thread %p) %d\n", object, class_name, microfeed_thread_get_current(), object->reference_count);
	}
	
	microfeed_mutex_unlock(object->object_impl->mutex);
	
	return object;
}

MicrofeedObject* microfeed_object_ref_generic(MicrofeedObject* object) {
	microfeed_assert(object->object_impl);
	microfeed_assert(object->object_impl->object == object);

	microfeed_mutex_lock(object->object_impl->mutex);
	
	object->reference_count++;

	if (microfeed_debug_references) {
		printf("%p REF     %s (thread %p) %d\n", object, object->object_impl->class->name, microfeed_thread_get_current(), object->reference_count);
	}
	
	microfeed_mutex_unlock(object->object_impl->mutex);
	
	return object;
}


void microfeed_object_unref_impl(MicrofeedObject* object, const char* class_name) {
	MicrofeedObjectImpl* object_impl;
	MicrofeedObjectFreeFunction free_function;
	
	microfeed_assert(object->object_impl);
	microfeed_assert(!strcmp(class_name, object->object_impl->class->name));
	microfeed_assert(object->object_impl->object == object);

	microfeed_mutex_lock(object->object_impl->mutex);

	microfeed_assert(object->reference_count > 0);
	
	object->reference_count--;

	if (microfeed_debug_references) {
		printf("%p UNREF   %s (thread %p) %d\n", object, class_name, microfeed_thread_get_current(), object->reference_count);
	}

	if (object->reference_count == 0) {
		if (microfeed_debug_references) {
			printf("%p FREE    %s (thread %p)\n", object, class_name, microfeed_thread_get_current());
		}
		
		object_impl = object->object_impl;
		free_function = object_impl->class->free_function;
		object->object_impl = NULL;
		object_impl->object = NULL;
		if (object_impl->weak_reference_count == 0) {
			destroy(object_impl);
		} else {

			microfeed_mutex_unlock(object_impl->mutex);
		}
		free_function(object);
		microfeed_mutex_free(object->mutex);
		microfeed_memory_free(object);
	} else {

		microfeed_mutex_unlock(object->object_impl->mutex);
	}
}

void microfeed_object_unref_generic(MicrofeedObject* object) {
	MicrofeedObjectImpl* object_impl;
	MicrofeedObjectFreeFunction free_function;
	
	microfeed_assert(object->object_impl);
	microfeed_assert(object->object_impl->object == object);

	microfeed_mutex_lock(object->object_impl->mutex);

	microfeed_assert(object->reference_count > 0);
	
	object->reference_count--;

	if (microfeed_debug_references) {
		printf("%p UNREF   %s (thread %p) %d\n", object, object->object_impl->class->name, microfeed_thread_get_current(), object->reference_count);
	}

	if (object->reference_count == 0) {
		if (microfeed_debug_references) {
			printf("%p FREE    %s (thread %p)\n", object, object->object_impl->class->name, microfeed_thread_get_current());
		}

		object_impl = object->object_impl;
		free_function = object_impl->class->free_function;
		object->object_impl = NULL;
		object_impl->object = NULL;
		if (object_impl->weak_reference_count == 0) {
			destroy(object_impl);
		} else {

			microfeed_mutex_unlock(object_impl->mutex);
		}
		free_function(object);
		microfeed_mutex_free(object->mutex);
		microfeed_memory_free(object);
	} else {

		microfeed_mutex_unlock(object->object_impl->mutex);
	}
}

MicrofeedWeakReference* microfeed_object_get_weak_reference_impl(MicrofeedObject* object, const char* class_name) {
	microfeed_assert(object->object_impl);
	
	microfeed_assert(!strcmp(class_name, object->object_impl->class->name));
	microfeed_assert(object->object_impl->object == object);

	microfeed_mutex_lock(object->object_impl->mutex);
	
	object->object_impl->weak_reference_count++;
	
	if (microfeed_debug_references) {
		printf("%p WEAK    %s (thread %p) %p %d\n", object->object_impl, object->object_impl->class->name, microfeed_thread_get_current(), object, object->object_impl->weak_reference_count);
	}

	microfeed_mutex_unlock(object->object_impl->mutex);

	return (MicrofeedWeakReference*)object->object_impl;
}

MicrofeedWeakReference* microfeed_object_get_weak_reference_generic(MicrofeedObject* object) {
	microfeed_assert(object->object_impl);
	
	microfeed_assert(object->object_impl->object == object);

	microfeed_mutex_lock(object->object_impl->mutex);
	
	object->object_impl->weak_reference_count++;

	if (microfeed_debug_references) {
		printf("%p WEAK    %s (thread %p) %p %d\n", object->object_impl, object->object_impl->class->name, microfeed_thread_get_current(), object, object->object_impl->weak_reference_count);
	}
		
	microfeed_mutex_unlock(object->object_impl->mutex);

	return (MicrofeedWeakReference*)object->object_impl;
}

void microfeed_object_lock_impl(MicrofeedObject* object, const char* class_name) {
	microfeed_assert(object->object_impl);
	microfeed_assert(!strcmp(class_name, object->object_impl->class->name));
	microfeed_assert(object->object_impl->object == object);

	if (microfeed_debug_locking) {
		printf("%p LOCK    %s (thread %p)\n", object, class_name, microfeed_thread_get_current());
	}

	microfeed_mutex_lock(object->mutex);
}

void microfeed_object_unlock_impl(MicrofeedObject* object, const char* class_name) {
	microfeed_assert(object->object_impl);
	microfeed_assert(!strcmp(class_name, object->object_impl->class->name));
	microfeed_assert(object->object_impl->object == object);

	if (microfeed_debug_locking) {
		printf("%p UNLOCK  %s (thread %p)\n", object, class_name, microfeed_thread_get_current());
	}

	microfeed_mutex_unlock(object->mutex);
}

void microfeed_weak_reference_unref(MicrofeedWeakReference* weak_reference) {
	MicrofeedObjectImpl* object_impl;
	
	object_impl = (MicrofeedObjectImpl*)weak_reference;
	
	microfeed_mutex_lock(object_impl->mutex);
	
	microfeed_assert(object_impl->weak_reference_count > 0);
	
	object_impl->weak_reference_count--;
	
	if (microfeed_debug_references) {
		printf("%p UNWEAK  %s (thread %p) %p %d\n", object_impl, object_impl->class->name, microfeed_thread_get_current(), object_impl->object, object_impl->weak_reference_count);
	}
	
	if (!object_impl->object && object_impl->weak_reference_count == 0) {
		destroy(object_impl);
	} else {
	
		microfeed_mutex_unlock(object_impl->mutex);
	}	
}

MicrofeedObject* microfeed_weak_reference_get_object_impl(MicrofeedWeakReference* weak_reference, const char* class_name) {
	MicrofeedObject* object;
	MicrofeedObjectImpl* object_impl;
	
	object_impl = (MicrofeedObjectImpl*)weak_reference;
	
	microfeed_assert(!strcmp(class_name, object_impl->class->name));

	microfeed_mutex_lock(object_impl->mutex);

	object = object_impl->object;
	if (object) {
		object->reference_count++;

		if (microfeed_debug_references) {
			printf("%p REF     %s (thread %p) %d [from weak]\n", object, class_name, microfeed_thread_get_current(), object->reference_count);
		}
	}
	
	microfeed_mutex_unlock(object_impl->mutex);
	
	return object;
}

MicrofeedObject* microfeed_weak_reference_get_object_generic(MicrofeedWeakReference* weak_reference) {
	MicrofeedObject* object;
	MicrofeedObjectImpl* object_impl;
	
	object_impl = (MicrofeedObjectImpl*)weak_reference;

	microfeed_mutex_lock(object_impl->mutex);

	object = object_impl->object;
	if (object) {
		object->reference_count++;

		if (microfeed_debug_references) {
			printf("%p REF     %s (thread %p) %d [from weak]\n", object, object->object_impl->class->name, microfeed_thread_get_current(), object->reference_count);
		}
	}
	
	microfeed_mutex_unlock(object_impl->mutex);
	
	return object;
}

static void destroy(MicrofeedObjectImpl* object_impl) {
	microfeed_mutex_free(object_impl->mutex);

	object_impl->class = NULL;
	object_impl->mutex = NULL;
	
	microfeed_memory_free(object_impl);
}

static void print_caller(void) {
	void* array[3];
	size_t size;
	char** strings;
	
	size = backtrace(array, 3);
	if (size >= 3) {
		strings = backtrace_symbols(array, size);
		printf("    %s\n", strings[2]);
		free(strings);
	}
}
