/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-common/microfeeditem.h>
#include <microfeed-common/microfeedmisc.h>

#include <string.h>
#include <stdint.h>
#include <ctype.h>
#include <inttypes.h>
#include <time.h>
#include <stdio.h>
#include <arpa/inet.h>

typedef uint16_t property_size_t;

struct _MicrofeedItem {
	size_t* properties; /* Offsets from the start of the buffer */
	unsigned int properties_length;
	unsigned int properties_reserved;
	char* buffer;
	size_t buffer_size;
	MicrofeedItemIterator* iterators;
};

struct _MicrofeedItemIterator {
	MicrofeedItemIterator* next;
	MicrofeedItemIterator* previous;
	
	MicrofeedItem* item;
	unsigned int index;
};

typedef struct {
	char status;
	property_size_t uid_length;
	uint32_t timestamp;
	char uid[1];
} Buffer;

static int get_index(const MicrofeedItem* item, const char* key, size_t length, unsigned int* index);

static MicrofeedItem* item_new(const char* uid) {
	MicrofeedItem* item;
	size_t length;
	
	length = strlen(uid);
	item = microfeed_memory_allocate(MicrofeedItem);
	item->buffer_size = sizeof(Buffer) + length;
	item->buffer = (char*)microfeed_memory_allocate_bytes(item->buffer_size);
	((Buffer*)item->buffer)->uid_length = length;
	memcpy(((Buffer*)item->buffer)->uid, uid, length + 1);

	return item;
}
	
MicrofeedItem* microfeed_item_new(const char* uid, time_t timestamp) {
	MicrofeedItem* item;
	
	item = item_new(uid);
	microfeed_item_set_timestamp(item, timestamp);

	return item;
}

MicrofeedItem* microfeed_item_new_temporary(void) {
	MicrofeedItem* item;

	item = item_new(microfeed_util_create_unique_identifier());
	microfeed_item_set_timestamp(item, time(NULL));

	return item;
}

MicrofeedItem* microfeed_item_new_with_status(const char* uid, time_t timestamp, MicrofeedItemStatus status) {
	MicrofeedItem* item;
	
	item = item_new(uid);
	microfeed_item_set_timestamp(item, timestamp);
	microfeed_item_set_status(item, status);
	
	return item;
}

MicrofeedItem* microfeed_item_new_from_marshalled(const void* data, size_t size) {
	MicrofeedItem* item = NULL;
	size_t offset;

	if (size > sizeof(Buffer)) {
		item = microfeed_memory_allocate(MicrofeedItem);
		item->buffer = (char*)malloc(size);
		memcpy(item->buffer, data, size);
		item->buffer_size = size;

		for (offset = sizeof(Buffer) + ((Buffer*)item->buffer)->uid_length; offset < size; offset += *((property_size_t*)(item->buffer + offset))) {
			if (item->properties_length == item->properties_reserved) {
				item->properties_reserved += 8 * sizeof(size_t);
				item->properties = (size_t*)realloc(item->properties, item->properties_reserved * sizeof(size_t));
			}
			item->properties[item->properties_length++] = offset;
		}

		if (offset > size) {
			free(item->buffer);
			item->buffer = NULL;
			item->buffer_size = 0;
			free(item->properties);
			item->properties = NULL;
			item->properties_length = 0;
			item->properties_reserved = 0;
			microfeed_memory_free(item);
			item = NULL;;
			/* TODO: Remove the next. */
			abort();
		}
	}

	return item;
}

void microfeed_item_free(MicrofeedItem* item) {
	if (item) {
		free(item->properties);
		free(item->buffer);
		item->properties = NULL;
		item->buffer = NULL;
		microfeed_memory_free(item);
	}
}

MicrofeedItem* microfeed_item_duplicate(MicrofeedItem* item) {
	MicrofeedItem* duplicate;
	
	duplicate = microfeed_memory_allocate(MicrofeedItem);
	duplicate->properties = (size_t*)malloc(item->properties_length * sizeof(size_t));
	memcpy(duplicate->properties, item->properties, item->properties_length * sizeof(size_t));
	duplicate->properties_length = duplicate->properties_reserved = item->properties_length;
	duplicate->buffer = (char*)malloc(item->buffer_size);
	memcpy(duplicate->buffer, item->buffer, item->buffer_size);
	duplicate->buffer_size = item->buffer_size;

	return duplicate;	
}

const char* microfeed_item_get_uid(MicrofeedItem* item) {

	return ((Buffer*)item->buffer)->uid;
}

void microfeed_item_marshalled_get_uid(const void* data, size_t data_size, const char** uid_return, size_t* uid_size_return) {
	*uid_return = ((Buffer*)data)->uid;
	*uid_size_return = (size_t)((Buffer*)data)->uid_length + 1;
}

time_t microfeed_item_get_timestamp(MicrofeedItem* item) {
	
	return (time_t)ntohl(((Buffer*)item->buffer)->timestamp);
}

void microfeed_item_marshalled_get_timestamp_uid(const void* data, size_t data_size, const void** timestamp_uid_return, size_t* timestamp_uid_size_return) {
	*timestamp_uid_return = &((Buffer*)data)->timestamp;
	*timestamp_uid_size_return = sizeof(uint32_t) + ((Buffer*)data)->uid_length;
}

void microfeed_item_set_timestamp(MicrofeedItem* item, time_t timestamp) {
	((Buffer*)item->buffer)->timestamp = htonl((uint32_t)timestamp);
}

MicrofeedItemStatus microfeed_item_get_status(MicrofeedItem* item) {
	
	return (MicrofeedItemStatus)(((Buffer*)item->buffer)->status);
}

MicrofeedItemStatus microfeed_item_marshalled_get_status(const void* data, size_t data_size) {

	return (MicrofeedItemStatus)(((Buffer*)data)->status);
}

void microfeed_item_set_status(MicrofeedItem* item, MicrofeedItemStatus status) {
	((Buffer*)item->buffer)->status = (char)status;
}

void microfeed_item_marshalled_set_status(const void* data, size_t data_size, MicrofeedItemStatus status) {
	((Buffer*)data)->status = (char)status;
}

const char* microfeed_item_get_property(MicrofeedItem* item, const char* key) {
	size_t length;
	const char* value;
	unsigned int index;
	
	length = strlen(key);
	if (!item->properties) {
		value = NULL;
	} else if (get_index(item, key, length, &index)) {	
		value = item->buffer + item->properties[index] + sizeof(property_size_t) + length + 1;
	} else {
		value = NULL;
	}
	
	return value;
}

MicrofeedItemIterator* microfeed_item_iterate_properties(MicrofeedItem* item, const char* start_key) {
	MicrofeedItemIterator* iterator;
	unsigned int index;
	
	iterator = microfeed_memory_allocate(MicrofeedItemIterator);
	iterator->item = item;

	if (start_key) {
		if (get_index(item, start_key, strlen(start_key), &index)) {
			iterator->index = index;
		} else {
			iterator->index = item->properties_length;
		}
	} else {
		iterator->index = 0;
	}
	
	if (item->iterators) {
		item->iterators->previous = iterator;
		iterator->next = item->iterators;
	} else {
		iterator->next = NULL;
	}
	iterator->previous = NULL;
	item->iterators = iterator;

	
	return iterator;
}

void microfeed_item_marshal(MicrofeedItem* item, const void** data_pointer, size_t* data_size_pointer) {
	*data_pointer = item->buffer;
	*data_size_pointer = item->buffer_size;
}

void microfeed_item_marshal_without_properties(MicrofeedItem* item, const void** data_pointer, size_t* data_size_pointer) {
	*data_pointer = item->buffer;
	*data_size_pointer = sizeof(Buffer) + ((Buffer*)item->buffer)->uid_length;
}

void microfeed_item_set_property(MicrofeedItem* item, const char* key, const char* value) {
	if (value) {
		microfeed_item_set_property_full(item, key, strlen(key), value, strlen(value));
	} else {
		/* TODO: Remove property. */
		microfeed_item_set_property_full(item, key, strlen(key), "", 0);
	}
}

void microfeed_item_set_property_with_length(MicrofeedItem* item, const char* key, const char* value, size_t value_length) {
	microfeed_item_set_property_full(item, key, strlen(key), value, value_length);	
}

void microfeed_item_set_property_full(MicrofeedItem* item, const char* key, size_t key_length, const char* value, size_t value_length) {
	unsigned int index;
	size_t length;
	size_t old_length;
	size_t buffer_size;
	unsigned int i;

	length = sizeof(uint16_t) + key_length + value_length + 2;
	length += length % sizeof(property_size_t);

	if (get_index(item, key, key_length, &index)) {
		old_length = *((property_size_t*)(item->buffer + item->properties[index]));
		if (length != old_length) {
			buffer_size = item->buffer_size;
			if (old_length < length) {
				item->buffer_size += length - old_length;
				item->buffer = (char*)realloc(item->buffer, item->buffer_size);
			}
			if (index < item->properties_length - 1) {
				memmove(item->buffer + item->properties[index + 1] + length - old_length,
				        item->buffer + item->properties[index + 1],
					buffer_size - item->properties[index + 1]);
				for (i = index + 1; i < item->properties_length; i++) {
					item->properties[i] += length - old_length;
				}
			}
			if (old_length > length) {
				item->buffer_size += length - old_length;
				item->buffer = (char*)realloc(item->buffer, item->buffer_size);
			}
		}
		*((property_size_t*)(item->buffer + item->properties[index])) = length;
		memcpy(item->buffer + item->properties[index] + sizeof(property_size_t) + key_length + 1, value, value_length);
		(item->buffer + item->properties[index] + sizeof(property_size_t) + key_length + 1)[value_length] = 0;
	} else {
		if (item->properties_length == item->properties_reserved) {
			item->properties_reserved += 8 * sizeof(char*);
			item->properties = (size_t*)realloc(item->properties, item->properties_reserved * sizeof(size_t));
		}

		buffer_size = item->buffer_size;
		item->buffer_size += length;
		item->buffer = (char*)realloc(item->buffer, item->buffer_size);

		if (index < item->properties_length) {
			memmove(item->properties + index + 1, item->properties + index,
			        (item->properties_length - index + 1) * sizeof(char*));
			memmove(item->buffer + item->properties[index + 1] + length,
				item->buffer + item->properties[index + 1],
				buffer_size - item->properties[index + 1]);
			for (i = index + 1; i <= item->properties_length; i++) {
				item->properties[i] += length;
			}
		} else {
			item->properties[index] = item->buffer_size - length;
		}
		*((property_size_t*)(item->buffer + item->properties[index])) = length;
		memcpy(item->buffer + item->properties[index] + sizeof(property_size_t), key, key_length);
		(item->buffer + item->properties[index] + sizeof(property_size_t))[key_length] = 0;
		memcpy(item->buffer + item->properties[index] + sizeof(property_size_t) + key_length + 1, value, value_length);
		(item->buffer + item->properties[index] + sizeof(property_size_t) + key_length + 1)[value_length] = 0;
		item->properties_length++;
	}
}

void microfeed_item_set_property_from_integer(MicrofeedItem* item, const char* key, long int integer) {
	char buffer[1024];
	
	microfeed_item_set_property_full(item, key, strlen(key), buffer, snprintf(buffer, 1024, "%ld", integer));
}

char* microfeed_item_get_properties_as_string(MicrofeedItem* item) {
	char* string;
	char* pointer;
	int i;
	property_size_t property_size;
	size_t length;
	
	string = pointer = (char*)microfeed_memory_allocate_bytes(item->buffer_size - item->properties_length * (sizeof(property_size_t) - 1) + 1);
	for (i = 0; i < item->properties_length; i++) {
		property_size = *((property_size_t*)(item->buffer + item->properties[i]));
		length = strlen(item->buffer + item->properties[i] + sizeof(property_size_t));
		memcpy(pointer, item->buffer + item->properties[i] + sizeof(property_size_t), length);
		pointer += length;
		*pointer++ = ':';
		*pointer++ = ' ';
		memcpy(pointer, item->buffer + item->properties[i] + sizeof(property_size_t) + length + 1, property_size - length - 1);
		pointer += property_size - length - 2 * sizeof(property_size_t) - 1;
		while (*pointer) {
			pointer++;
		}
		*pointer++ = '\n';
	}
	*pointer = 0;
	
	return string;
}

int microfeed_item_set_properties_from_string(MicrofeedItem* item, const char* string) {
	const char* key;
	size_t key_length;
	const char* value;
	size_t value_length;
	const char* end;
	
	while (*string) {
		key = string;
		while (*key == ' ' || *key == '\t') {
			key++;
		}
		value = key;
		while (*value != 0 && *value != ':' && *value != ' ' && *value != '\t') {
			value++;
		}
		key_length = value - key;
		while (*value == ' ' || *value == '\t') {
			value++;
		}
		if (*value != ':') {

			return (*value == 0 ? 1 : 0);
		}
		value++;
		while (*value == ' ' || *value == '\t') {
			value++;
		}
		end = value;
		while (*end != 0 && *end != '\n' && *end != '\r') {
			end++;
		}
		if (*end) {
			string = end + 1;
		} else {
			string = end;
		}
		end--;
		while ((*end == ' ' || *end == '\t') && end > value) {
			end--;
		}
		value_length = end - value + 1;
		
		microfeed_item_set_property_full(item, key, key_length, value, value_length);
	}
	
	return 1;
}

/* TODO: Should use defines. */
MicrofeedItemPermission microfeed_item_permission_from_string(const char* string) {
	MicrofeedItemPermission item_permission = MICROFEED_ITEM_PERMISSION_NONE;

	while (*string == ':') {
		string++;
		if (!strncmp(string, "modify", 6) && (string[6] == 0 || string[6] == ':')) {
			item_permission |= MICROFEED_ITEM_PERMISSION_MODIFY;
			string += 6;
		} else if (!strncmp(string, "remove", 6) && (string[6] == 0 || string[6] == ':')) {
			item_permission |= MICROFEED_ITEM_PERMISSION_REMOVE;
			string += 6;
		} else if (!strncmp(string, "reply", 5) && (string[5] == 0 || string[5] == ':')) {
			item_permission |= MICROFEED_ITEM_PERMISSION_REPLY;
			string += 5;
		}
	}
	
	return item_permission;
}

/* TODO: Should use defines. */
char* microfeed_item_permission_to_string(MicrofeedItemPermission item_permission) {
	char buffer[1024];
	char* string = buffer;
	
	if (item_permission & MICROFEED_ITEM_PERMISSION_MODIFY) {
		memcpy(string, ":modify", 7);
		string += 7;
	}
	if (item_permission & MICROFEED_ITEM_PERMISSION_REMOVE) {
		memcpy(string, ":remove", 7);
		string += 7;
	}
	if (item_permission & MICROFEED_ITEM_PERMISSION_REPLY) {
		memcpy(string, ":reply", 6);
		string += 6;
	}
	*string = 0;
	
	return strdup(buffer);
}

void microfeed_item_iterator_free(MicrofeedItemIterator* iterator) {
	if (iterator->next) {
		iterator->next->previous = iterator->previous;
	}
	if (iterator->previous) {
		iterator->previous->next = iterator->next;
	} else if (iterator->item) {
		iterator->item->iterators = iterator->next;
	}

	iterator->item = NULL;
	microfeed_memory_free(iterator);
}

int microfeed_item_iterator_get(MicrofeedItemIterator* iterator, const char** key, const char** value) {
	int success = 0;
	size_t length;

	if (iterator->item && iterator->index < iterator->item->properties_length) {
		*key = iterator->item->buffer + iterator->item->properties[iterator->index] + sizeof(property_size_t);
		length = strlen(iterator->item->buffer + iterator->item->properties[iterator->index] + sizeof(property_size_t)) + 1;
		*value = iterator->item->buffer + iterator->item->properties[iterator->index] + sizeof(property_size_t) + length;
		success = 1;
	}

	return success;
}

void microfeed_item_iterator_next(MicrofeedItemIterator* iterator) {
	if (iterator->item && iterator->index < iterator->item->properties_length) {
		iterator->index++;
	}
}

static int get_index(const MicrofeedItem* item, const char* key, size_t length, unsigned int* index) {
	int retval = 0;
	int result;
	unsigned int i, min, max;
	
	if (item->properties_length == 0) {
		*index = 0;
	} else if ((result = memcmp(key, item->buffer + item->properties[0] + sizeof(property_size_t), length)) == 0 && (item->buffer + item->properties[0] + sizeof(property_size_t))[length] == 0) {
		*index = 0;
		retval = 1;
	} else if (result < 0) {
		*index = 0;
	} else if (item->properties_length == 1) {
		*index = 1;
	} else if ((result = memcmp(key, item->buffer + item->properties[item->properties_length - 1] + sizeof(property_size_t), length)) == 0) {
		*index = item->properties_length - 1;
		retval = 1;
	} else if (result > 0) {
		*index = item->properties_length;
	} else if (item->properties_length == 2) {
		*index = item->properties_length - 1;
	} else {
		min = i = 0;
		max = item->properties_length - 1;
		
		while (min <= max) {
			i = (min + max) / 2;
			if ((result = memcmp(key, item->buffer + item->properties[i] + sizeof(property_size_t), length)) == 0 && (item->buffer + item->properties[i] + sizeof(property_size_t))[length] == 0) {
				retval = 1;
				break;
			} else if (result < 0) {
				max = i - 1;
			} else {
				i++;
				min = i;
			}
		}
		*index = i;
	}

	return retval;
}

