/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-provider/microfeedprovider.h>
#include <microfeed-provider/microfeedpublisher.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-provider/microfeederror.h>
#include <microfeed-common/microfeedprotocol.h>
#include <microfeed-common/microfeedobject.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <stdint.h>

typedef struct _Thread Thread;

struct _MicrofeedPublisher {
	MicrofeedObject object;

	char* identifier;
	char* object_path;
	char* data_directory;
	MicrofeedProvider* provider;
	MicrofeedPublisherCallbacks callbacks;
	void* user_data;
	MicrofeedDatabaseEnvironment* database_environment;
	
	MicrofeedStore* subscribers;
	MicrofeedStore* feeds;
	MicrofeedDatabase* data_database;
	MicrofeedStore* store_data_requests;
};

typedef struct {
	unsigned int reference_count;
	MicrofeedPublisher* publisher;
	char* unique_connection_name;
	time_t last_activity;
	MicrofeedStore* feeds;
} Subscriber;

typedef struct {
	const char* name;
	MicrofeedError* (*callback)(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
} MethodCallback;

typedef struct {
	const char* url;
	DBusMessage** messages;
	int messages_length;
} StoreDataRequest;

typedef struct {
	MicrofeedPublisher* publisher;
	Subscriber* subscriber;
	DBusMessage* message;
	MicrofeedError* (*callback)(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
} MethodThreadData;

typedef struct {
	MicrofeedPublisher* publisher;
	MicrofeedFeed* feed;
} UpdateThreadData;

static void free_function(MicrofeedPublisher* publisher);

static Subscriber* subscriber_new(MicrofeedPublisher* publisher, const char* unique_connection_name);
static void subscriber_free(Subscriber* subscriber);
static const char* subscriber_get_unique_connection_name(Subscriber* subscriber);

static const char* store_data_request_get_url(StoreDataRequest* store_data_request);

static MicrofeedError* method_add_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_modify_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_remove_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_read_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_republish_items(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_mark_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_unmark_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_ping(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_subscribe_feed(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_unsubscribe_feed(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_update_feed(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_create_publisher(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_destroy_publisher(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_store_data(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);

void subscriber_disconnected(MicrofeedProvider* provider, const char* unique_connection_name, void* user_data);
static MicrofeedFeed* get_or_instantiate_feed(MicrofeedPublisher* publisher, const char* uri, int create);
static MicrofeedError* settings_modify_item(MicrofeedFeed* feed, MicrofeedItem* existing_item, MicrofeedItem* new_item, void* user_data);

static MicrofeedFeedCallbacks settings_callbacks = {
	.modify_item = settings_modify_item
};

static MicrofeedClass microfeed_publisher_class = {
	"MicrofeedPublisher",
	(MicrofeedObjectFreeFunction)free_function
};

static MethodCallback method_callbacks[] = {
	{ MICROFEED_METHOD_NAME_ADD_ITEM, method_add_item },
	{ MICROFEED_METHOD_NAME_MODIFY_ITEM, method_modify_item },
	{ MICROFEED_METHOD_NAME_REMOVE_ITEM, method_remove_item },
	{ MICROFEED_METHOD_NAME_READ_ITEM, method_read_item },
	{ MICROFEED_METHOD_NAME_REPUBLISH_ITEMS, method_republish_items },
	{ MICROFEED_METHOD_NAME_MARK_ITEM, method_mark_item },
	{ MICROFEED_METHOD_NAME_UNMARK_ITEM, method_unmark_item },
	{ MICROFEED_METHOD_NAME_PING, method_ping },
	{ MICROFEED_METHOD_NAME_SUBSCRIBE_FEED, method_subscribe_feed },
	{ MICROFEED_METHOD_NAME_UNSUBSCRIBE_FEED, method_unsubscribe_feed },
	{ MICROFEED_METHOD_NAME_UPDATE_FEED, method_update_feed },
	{ MICROFEED_METHOD_NAME_CREATE_PUBLISHER, method_create_publisher },
	{ MICROFEED_METHOD_NAME_DESTROY_PUBLISHER, method_destroy_publisher },
	{ MICROFEED_METHOD_NAME_STORE_DATA, method_store_data },
	{ NULL, NULL }
};

/**
 * Instantiates a new publisher.
 * 
 * The publisher creates or opens
 * a database environment in the given directory using publisher_identifier as a name.
 * 
 * @param publisher_identifier A string containing unique publisher identifier.
 * @param directory A directory where the database environment is created or opened (if exists).
 * @param connection A DBus connection.
 * @param callbacks Functions that are called when a publisher receives a message from a subscriber.
 * @param user_data A pointer to user data that is given as a parameter when a callback function is called.
 * @return Instantiated MicrofeedPublisher.
 */
MicrofeedPublisher* microfeed_publisher_new(MicrofeedProvider* provider, const char* publisher_identifier, const char* directory, MicrofeedPublisherCallbacks* callbacks, void* user_data) {
	MicrofeedPublisher* publisher;
	char* separator;
	char* path;

	publisher = microfeed_object_new(&microfeed_publisher_class, MicrofeedPublisher);

	publisher->identifier = strdup(publisher_identifier);
	if (!(separator = strchr(publisher->identifier, MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_CHAR)) || separator[1] == 0) {
		free(publisher->identifier);
		microfeed_memory_free(publisher);
		publisher = NULL;
	} else {
		*separator = 0;
		publisher->object_path = microfeed_util_string_concatenate(MICROFEED_DBUS_OBJECT_PATH_PREFIX_PUBLISHER, publisher->identifier, NULL);
		publisher->data_directory = microfeed_util_string_concatenate(directory, "/", publisher_identifier, "/data/", NULL);
		*separator = MICROFEED_PUBLISHER_IDENTIFIER_SEPARATOR_CHAR;
		publisher->provider = provider;
		publisher->callbacks = *callbacks;
		publisher->user_data = user_data;

		if (!microfeed_util_create_directory_recursively(publisher->data_directory)) {
			/* TODO: Free */
		
			return NULL;
		}
		path = microfeed_util_string_concatenate(directory, "/", publisher_identifier, "/db", NULL);
		if (!(publisher->database_environment = microfeed_database_environment_new("berkeleydb", path))) {
			/* TODO: Free */
			
			return NULL;
		}
		free(path);
		publisher->subscribers = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)subscriber_get_unique_connection_name);
		publisher->feeds = microfeed_store_new_sorted_weak_references((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)microfeed_feed_get_uri);

		publisher->data_database = microfeed_database_environment_get_database(publisher->database_environment, ":microfeed.timestamps", NULL, 1);
		publisher->store_data_requests = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)store_data_request_get_url);
	}
	
	return publisher;
}

/**
 * Frees the resources allocated for the publisher.
 * 
 * @param publisher Instantiated MicrofeedPublisher.
 */
static void free_function(MicrofeedPublisher* publisher) {
	microfeed_assert(microfeed_store_get_size(publisher->subscribers) == 0);
	microfeed_assert(microfeed_store_get_size(publisher->store_data_requests) == 0);

	free(publisher->identifier);
	free(publisher->object_path);
	free(publisher->data_directory);
	microfeed_store_free(publisher->subscribers);
	microfeed_store_free(publisher->feeds);
	microfeed_object_unref(publisher->data_database, MicrofeedDatabase);
	microfeed_object_unref(publisher->database_environment, MicrofeedDatabaseEnvironment);
	microfeed_store_free(publisher->store_data_requests);
}

int microfeed_publisher_handle_message(MicrofeedPublisher* publisher, DBusMessage* message) {
	int retvalue = 0;
	const char* unique_connection_name;
	int i;
	Subscriber* subscriber;
	MicrofeedError* error;
	char buffer[1024];
	DBusMessage* reply;

	microfeed_object_lock(publisher, MicrofeedPublisher);

	unique_connection_name = dbus_message_get_sender(message);
	if (!(subscriber = microfeed_store_get(publisher->subscribers, unique_connection_name, Subscriber))) {
		subscriber = subscriber_new(publisher, unique_connection_name);
		microfeed_store_insert(publisher->subscribers, subscriber);
		
		microfeed_provider_add_subscriber_disconnected_callback(publisher->provider, unique_connection_name, subscriber_disconnected, publisher);
	} else {
		subscriber->reference_count++;
	}

	subscriber->last_activity = time(NULL);
	for (i = 0; method_callbacks[i].name; i++) {
		if (dbus_message_is_method_call(message, MICROFEED_DBUS_INTERFACE_PUBLISHER, method_callbacks[i].name)) {
			printf("%s < %s (%s)\n", publisher->identifier, method_callbacks[i].name, unique_connection_name);
	
			error = method_callbacks[i].callback(publisher, subscriber, message);

			if (!error) {
				printf("%s > OK\n", publisher->identifier);
				reply = dbus_message_new_method_return(message);
			} else if (error == (MicrofeedError*)-1) {
				printf("%s > OK (sent already)\n", publisher->identifier);
				reply = NULL;
			} else {
				printf("%s > %s (%s)\n", publisher->identifier, microfeed_error_get_name(error), microfeed_error_get_message(error));
				snprintf(buffer, 1024, "%s.%s", MICROFEED_DBUS_INTERFACE_ERROR, microfeed_error_get_name(error));
				reply = dbus_message_new_error(message, buffer, microfeed_error_get_message(error));
				microfeed_error_free(error);
			}
			if (reply) {
				microfeed_provider_send_message(publisher->provider, reply);
				dbus_message_unref(reply);
			}
			
			retvalue = 1;
			break;
		}
	}
	
	subscriber->reference_count--;
	if (subscriber->reference_count == 0 && microfeed_store_get_size(subscriber->feeds) == 0) {
		microfeed_provider_remove_subscriber_disconnected_callback(publisher->provider, unique_connection_name, subscriber_disconnected, publisher);

		microfeed_store_remove(publisher->subscribers, subscriber);
		subscriber_free(subscriber);
	}

	microfeed_object_unlock(publisher, MicrofeedPublisher);

/* TODO: This should not be here! */
	dbus_message_unref(message);

	return retvalue;
}


MicrofeedDatabaseEnvironment* microfeed_publisher_get_database_environment(MicrofeedPublisher* publisher) {

	return publisher->database_environment;
}

const char* microfeed_publisher_get_object_path(MicrofeedPublisher* publisher) {

	return publisher->object_path;
}

const char* microfeed_publisher_get_identifier(MicrofeedPublisher* publisher) {

	return publisher->identifier;
}

MicrofeedProvider* microfeed_publisher_get_provider(MicrofeedPublisher* publisher) {

	return publisher->provider;	
}

/**
 * ...
 *
 * This function calls the instantiate_feed callback, so be careful to not make infinite loops.
 */
MicrofeedFeed* microfeed_publisher_get_or_instantiate_feed(MicrofeedPublisher* publisher, const char* uri, int create) {
	MicrofeedFeed* feed;
	
	microfeed_object_lock(publisher, MicrofeedPublisher);
	
	feed = get_or_instantiate_feed(publisher, uri, create);
	
	microfeed_object_unlock(publisher, MicrofeedPublisher);

	return feed;
}
	
/**
 * Adds a new setting or changes other properties than the value of an existing setting in
 * the org.microfeed.Publisher.Settings feed.
 * 
 * @param publisher Instantiated MicrofeedPublisher.
 */
void microfeed_publisher_add_setting(MicrofeedPublisher* publisher, const char* uid, const char* text, const char* unit, const char* type, const char* length, const char* value) {
	MicrofeedFeed* feed;
	MicrofeedItem* item;
	MicrofeedItem* existing_item;
	
	microfeed_object_lock(publisher, MicrofeedPublisher);
	
	feed = get_or_instantiate_feed(publisher, MICROFEED_FEED_URI_SETTINGS, 1);

	microfeed_object_unlock(publisher, MicrofeedPublisher);
	
	if (feed) {
		item = microfeed_item_new(uid, time(NULL));
		microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_LENGTH, length);
		microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_TEXT, text);
		microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_TYPE, type);
		microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_UNIT, unit);	
		if ((existing_item = microfeed_feed_get_item(feed, uid))) {
			microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE, microfeed_item_get_property(existing_item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE));
		} else {
			microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE, value);
		}
		microfeed_feed_replace_item(feed, item);
		microfeed_item_free(item);

		microfeed_object_unref(feed, MicrofeedFeed);
	}
}

/**
 * Removes a setting from the org.microfeed.Publisher.Settings feed.
 * 
 * @param publisher Instantiated MicrofeedPublisher
 * @param uid The unique identifier of the setting item.
 */
void microfeed_publisher_remove_setting(MicrofeedPublisher* publisher, const char* uid) {
	MicrofeedFeed* feed;
	
	microfeed_object_lock(publisher, MicrofeedPublisher);
	
	feed = get_or_instantiate_feed(publisher, MICROFEED_FEED_URI_SETTINGS, 1);

	microfeed_object_unlock(publisher, MicrofeedPublisher);
	
	if (feed) {		
		microfeed_feed_remove_item(feed, uid);

		microfeed_object_unref(feed, MicrofeedFeed);
	}
}

/**
 * Returns a copy of a value of a setting, or if the setting does not exist returns a copy of the given default value.
 * 
 * @param publisher Instantiated MicrofeedPublisher
 * @param key The unique identifier of the setting item.
 * @param default_value The value that is returned if the setting does not exist.
 * @return The value of the setting (must be freed).
 */
char* microfeed_publisher_get_setting_value(MicrofeedPublisher* publisher, const char* uid, const char* default_value) {
	MicrofeedFeed* feed;
	const char* const_value;
	char* value;
	MicrofeedItem* item;
	
	microfeed_object_lock(publisher, MicrofeedPublisher);
	
	feed = get_or_instantiate_feed(publisher, MICROFEED_FEED_URI_SETTINGS, 1);

	microfeed_object_unlock(publisher, MicrofeedPublisher);
	
	if (feed && (item = microfeed_feed_get_item(feed, uid))) {
		if ((const_value = microfeed_item_get_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE))) {
			value = strdup(const_value);
		} else {
			value = (default_value ? strdup(default_value) : NULL);
		}
		microfeed_item_free(item);
	} else {
		value = (default_value ? strdup(default_value) : NULL);
	}
	
	if (feed) {
		microfeed_object_unref(feed, MicrofeedFeed);
	}
	
	return value;
}

/**
 * Returns a value of a setting in integer, or if the setting does not exist returns the given default value.
 * 
 * @param publisher Instantiated MicrofeedPublisher
 * @param key The unique identifier of the setting item.
 * @param default_value The value that is returned if the setting does not exist.
 * @return The value of the setting in integer.
 */
long int microfeed_publisher_get_setting_value_integer(MicrofeedPublisher* publisher, const char* uid, long int default_value) {
	MicrofeedFeed* feed;
	long int value;
	MicrofeedItem* item;
	const char* string;
	char* end;
	
	microfeed_object_lock(publisher, MicrofeedPublisher);
	
	feed = get_or_instantiate_feed(publisher, MICROFEED_FEED_URI_SETTINGS,1 );

	microfeed_object_unlock(publisher, MicrofeedPublisher);
	
	if (feed && (item = microfeed_feed_get_item(feed, uid))) {
		if ((string = microfeed_item_get_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE)) && *string != 0) {
			value = strtol(string, &end, 10);
			if (*end != 0) {
				value = default_value;
			}
		} else {
			value = default_value;
		}
		microfeed_item_free(item);
	} else {
		value = default_value;
	}
	
	if (feed) {
		microfeed_object_unref(feed, MicrofeedFeed);
	}

	return value;
	
}

/**
 * Sets the value of a setting if the setting exists.
 * 
 * @param publisher Instantiated MicrofeedPublisher
 * @param uid The unique identified of the setting item.
 * @param value The new value for the setting.
 * @return A boolean value: true, if the operation succeeded, or false if the setting does not exist.
 */
int microfeed_publisher_set_setting_value(MicrofeedPublisher* publisher, const char* uid, const char* value) {
	MicrofeedFeed* feed;
	int retvalue = 0;
	MicrofeedItem* item;

	microfeed_object_lock(publisher, MicrofeedPublisher);
	
	feed = get_or_instantiate_feed(publisher, MICROFEED_FEED_URI_SETTINGS, 1);

	microfeed_object_unlock(publisher, MicrofeedPublisher);
	
	if (feed && (item = microfeed_feed_get_item(feed, uid))) {
		microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE, value);
		microfeed_feed_replace_item(feed, item);
		retvalue = 1;
		microfeed_item_free(item);
	}
	
	if (feed) {
		microfeed_object_unref(feed, MicrofeedFeed);
	}

	return retvalue;
}

void microfeed_publisher_ref_data(MicrofeedPublisher* publisher, const char* url) {
	size_t url_size;
	uint64_t reference_count;
	size_t size;
	MicrofeedItem* item;
	uint64_t timestamp;
	void* key;
	size_t key_size;
	const void* data;
	size_t data_size;
	char status;

	microfeed_object_lock(publisher, MicrofeedPublisher);

	url_size = strlen(url) + 1;
	size = sizeof(uint64_t);
	if (microfeed_database_get_data_partial(publisher->data_database, url, url_size, &reference_count, &size, 0) && size == sizeof(uint64_t)) {
		reference_count++;
		microfeed_database_replace_data_partial(publisher->data_database, url, url_size, &reference_count, size, 0);
	} else {
		reference_count = 1;
		microfeed_database_replace_data(publisher->data_database, url, url_size, &reference_count, size);
	}

	microfeed_object_unlock(publisher, MicrofeedPublisher);
}

void microfeed_publisher_unref_data(MicrofeedPublisher* publisher, const char* url) {
	size_t url_size;
	void* data;
	size_t data_size;
	uint64_t* reference_count_pointer;
	char* path;

	microfeed_object_lock(publisher, MicrofeedPublisher);
	
	url_size = strlen(url) + 1;
	data_size = sizeof(uint64_t);
	if (microfeed_database_get_data(publisher->data_database, url, url_size, &data, &data_size)) {
		if (data_size >= sizeof(uint64_t)) {
			reference_count_pointer == (uint64_t*)data;
			(*reference_count_pointer)--;
			if (*reference_count_pointer == 0) {
				path = microfeed_util_string_concatenate(publisher->data_directory, data + sizeof(uint64_t), NULL);
				unlink(path);
				free(path);

				microfeed_database_remove_data(publisher->data_database, url, url_size);
			} else {
				microfeed_database_replace_data_partial(publisher->data_database, url, url_size, data, sizeof(uint64_t), 0);
			}
		}
		free(data);
	}
	
	microfeed_object_unlock(publisher, MicrofeedPublisher);
}

static Subscriber* subscriber_new(MicrofeedPublisher* publisher, const char* unique_connection_name) {
	Subscriber* subscriber;

	subscriber = (Subscriber*)microfeed_memory_allocate(Subscriber);
	subscriber->reference_count = 1;
	subscriber->publisher = publisher;
	subscriber->unique_connection_name = strdup(unique_connection_name);
	subscriber->last_activity = time(NULL);
	subscriber->feeds = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp,
	                                        	    (MicrofeedStoreGetKeyFunction)microfeed_feed_get_uri);

	return subscriber;	
}

static void subscriber_free(Subscriber* subscriber) {
	free(subscriber->unique_connection_name);
	microfeed_store_free(subscriber->feeds);
	
	microfeed_memory_free(subscriber);
}

static const char* subscriber_get_unique_connection_name(Subscriber* subscriber) {

	return subscriber->unique_connection_name;
}

static const char* store_data_request_get_url(StoreDataRequest* store_data_request) {

	return store_data_request->url;
}

static MicrofeedError* method_add_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	DBusMessageIter iter;
	const char* uri;
	const char* uid;
	MicrofeedFeed* feed;
	MicrofeedItem* item;
	const char* key;
	const char* value;
	MicrofeedError* error;
	
	if (!dbus_message_iter_init(message, &iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri argument in AddItem method.");
	}
	dbus_message_iter_get_basic(&iter, &uri);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected temporary item uid argument in AddItem method.");
	}
	dbus_message_iter_get_basic(&iter, &uid);
	if (!(feed = get_or_instantiate_feed(publisher, uri, 1))) {
		
		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to add an item into a feed that does not exist.");
	}
	/* TODO: Would be faster if microfeed_feed_check_item existed. */
	if ((item = microfeed_feed_get_item(feed, uid))) {
		microfeed_item_free(item);
		microfeed_object_unref(feed, MicrofeedFeed);
		
		return microfeed_error_new(MICROFEED_ERROR_ITEM_ALREADY_EXISTS, "Trying to add an item that already existed in a feed.");
	}
	item = microfeed_item_new(uid, time(NULL));
	while (dbus_message_iter_next(&iter)) {
		if (dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
			microfeed_item_free(item);
			microfeed_object_unref(feed, MicrofeedFeed);

			return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected string key in AddItem method.");
		}
		dbus_message_iter_get_basic(&iter, &key);

		if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
			microfeed_item_free(item);
			microfeed_object_unref(feed, MicrofeedFeed);

			return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected string value in AddItem method.");
		}
		dbus_message_iter_get_basic(&iter, &value);
		
		microfeed_item_set_property(item, key, value);
	}
	
	microfeed_object_unlock(publisher, MicrofeedPublisher);

	error = microfeed_feed_call_modify_item_callback(feed, NULL, item);

	microfeed_object_lock(publisher, MicrofeedPublisher);

	microfeed_item_free(item);
	microfeed_object_unref(feed, MicrofeedFeed);

	return error;
}

static MicrofeedError* method_modify_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	DBusMessageIter iter;
	const char* uri;
	const char* uid;
	MicrofeedFeed* feed;
	MicrofeedItem* existing_item;
	MicrofeedItem* new_item;
	const char* key;
	const char* value;
	MicrofeedError* error;
	
	if (!dbus_message_iter_init(message, &iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri argument in ModifyItem method.");
	}
	dbus_message_iter_get_basic(&iter, &uri);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected item uid argument in ModifyItem method.");
	}
	dbus_message_iter_get_basic(&iter, &uid);
	if (!(feed = get_or_instantiate_feed(publisher, uri, 1))) {
		
		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to modify an item of a feed that does not exist.");
	}

	if (!(existing_item = microfeed_feed_get_item(feed, uid))) {
		microfeed_object_unref(feed, MicrofeedFeed);

		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_ITEM, "Trying to modify an item that does not exist.");
	}

	new_item = microfeed_item_new(uid, 0); /* TODO: New timestamp from message? */
	while (dbus_message_iter_next(&iter)) {
		if (dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
			microfeed_item_free(new_item);
			microfeed_item_free(existing_item);
			microfeed_object_unref(feed, MicrofeedFeed);

			return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected string key in ModifyItem method.");
		}
		dbus_message_iter_get_basic(&iter, &key);

		if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
			microfeed_item_free(new_item);
			microfeed_item_free(existing_item);
			microfeed_object_unref(feed, MicrofeedFeed);

			return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected string value in ModifyItem method.");
		}
		dbus_message_iter_get_basic(&iter, &value);
		
		microfeed_item_set_property(new_item, key, value);
	}
	
	microfeed_object_unlock(publisher, MicrofeedPublisher);

	error = microfeed_feed_call_modify_item_callback(feed, existing_item, new_item);

	microfeed_object_lock(publisher, MicrofeedPublisher);

	microfeed_item_free(new_item);
	microfeed_item_free(existing_item);
	microfeed_object_unref(feed, MicrofeedFeed);

	return error;
}

static MicrofeedError* method_remove_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedError* error;
	DBusError dbus_error;
	const char* uri;
	const char* uid;
	MicrofeedFeed* feed;
	MicrofeedItem* item;

	dbus_error_init(&dbus_error);
	if (!dbus_message_get_args(message, &dbus_error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID)) {
		error = microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri and item uid arguments in RemoveItem method");
	} else if (!(feed = get_or_instantiate_feed(publisher, uri, 1))) {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to remove an item of a feed that does not exist.");
	} else if (!(item = microfeed_feed_get_item(feed, uid))) {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_ITEM, "Trying to remove an item that does not exist.");
	} else {
		microfeed_object_unlock(publisher, MicrofeedPublisher);

		error = microfeed_feed_call_modify_item_callback(feed, item, NULL);

		microfeed_object_lock(publisher, MicrofeedPublisher);

		microfeed_item_free(item);
		microfeed_object_unref(feed, MicrofeedFeed);
	}
	
	return error;
}

static MicrofeedError* method_read_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedError* error;
	DBusMessageIter iter;
	const char* uri;
	const char* uid;
	MicrofeedFeed* feed;
	
	if (!dbus_message_iter_init(message, &iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri argument in ReadItem method.");
	}
	dbus_message_iter_get_basic(&iter, &uri);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected item uid argument in ReadItem method.");
	}
	dbus_message_iter_get_basic(&iter, &uid);
	if (!(feed = get_or_instantiate_feed(publisher, uri, 1))) {
		
		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to read item of a feed that does not exist.");
	}
	
	microfeed_object_unlock(publisher, MicrofeedPublisher);

	error = microfeed_feed_unset_item_status(feed, uid, MICROFEED_ITEM_STATUS_UNREAD);

	microfeed_object_lock(publisher, MicrofeedPublisher);

	microfeed_object_unref(feed, MicrofeedFeed);

	return error;
}

static MicrofeedError* method_republish_items(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	DBusMessageIter iter;
	const char* uri;
	const char* start_uid;
	const char* end_uid;
	dbus_uint16_t max_count;
	MicrofeedFeed* feed;

	if (!dbus_message_iter_init(message, &iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected a feed uri argument in PublishItems method");
	}
	dbus_message_iter_get_basic(&iter, &uri);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected a start item uid argument in PublishItems method");
	}
	dbus_message_iter_get_basic(&iter, &start_uid);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected an end item uid argument in PublishItems method");
	}
	dbus_message_iter_get_basic(&iter, &end_uid);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_UINT16) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected an maximum count argument in PublishItems method");
	}
	dbus_message_iter_get_basic(&iter, &max_count);

	if (!(feed = microfeed_store_get(subscriber->feeds, uri, MicrofeedFeed))) {

		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to republish items of a feed that is not subscribed.");
	}
	microfeed_object_ref(feed, MicrofeedFeed);
	
	microfeed_object_unlock(publisher, MicrofeedPublisher);
	
	microfeed_feed_republish(feed, (*start_uid ? start_uid : NULL), (*end_uid ? end_uid : NULL), max_count, subscriber->unique_connection_name);

	microfeed_object_lock(publisher, MicrofeedPublisher);

	microfeed_object_unref(feed, MicrofeedFeed);
	
	return NULL;
}

static MicrofeedError* method_mark_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedError* error = NULL;
	DBusError dbus_error;
	const char* uri;
	const char* uid;
	MicrofeedFeed* feed;
	
	dbus_error_init(&dbus_error);
	if (!dbus_message_get_args(message, &dbus_error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID)) {
		error = microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri and item uid arguments in MarkItem method");
	} else if (!(feed = get_or_instantiate_feed(publisher, uri, 1))) {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to mark an item of a feed that does not exist.");
	} else {
		microfeed_object_unlock(publisher, MicrofeedPublisher);

		error = microfeed_feed_set_item_status(feed, uid, MICROFEED_ITEM_STATUS_MARKED);

		microfeed_object_lock(publisher, MicrofeedPublisher);

		microfeed_object_unref(feed, MicrofeedFeed);
	}

	return error;
}

static MicrofeedError* method_unmark_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedError* error = NULL;
	DBusError dbus_error;
	const char* uri;
	const char* uid;
	MicrofeedFeed* feed;
	
	dbus_error_init(&dbus_error);
	if (!dbus_message_get_args(message, &dbus_error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID)) {
		error = microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri and item uid arguments in UnarkItem method");
	} else if (!(feed = get_or_instantiate_feed(publisher, uri, 1))) {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to unmark an item of a feed that does not exist.");
	} else {
		microfeed_object_unlock(publisher, MicrofeedPublisher);

		error = microfeed_feed_unset_item_status(feed, uid, MICROFEED_ITEM_STATUS_MARKED);

		microfeed_object_lock(publisher, MicrofeedPublisher);

		microfeed_object_unref(feed, MicrofeedFeed);
	}

	return error;
}

static MicrofeedError* method_ping(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {

	return NULL;
}

static MicrofeedError* method_subscribe_feed(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedFeed* feed;
	DBusError error;
	char* uri;
	DBusMessage* reply;
	MicrofeedItem* item;
	unsigned long int interval;
	long last_time = 0;
	time_t elapsed;

	dbus_error_init(&error);
	if (!dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_INVALID)) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri argument in SubscribeFeed method");
	}
	
	if (microfeed_store_get(subscriber->feeds, uri, MicrofeedFeed)) {

		return microfeed_error_new(MICROFEED_ERROR_FEED_ALREADY_SUBSCRIBED, "Cannot subscribe the feed twice.");
	}

	if (!(feed = get_or_instantiate_feed(publisher, uri, 1))) {
	
		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Cannot subscribe the feed.");
	}
	
	microfeed_store_insert(subscriber->feeds, microfeed_object_ref(feed, MicrofeedFeed));

printf(" SUBCRIBE: %s(%p) %p %s\n", microfeed_feed_get_uri(feed), feed, publisher, publisher->identifier);

	printf("%s > OK (early sending)\n", publisher->identifier);
	reply = dbus_message_new_method_return(message);
	microfeed_provider_send_message(publisher->provider, reply);
	dbus_message_unref(reply);

	microfeed_object_unlock(publisher, MicrofeedPublisher);

	microfeed_feed_send_metadata_item(feed, subscriber->unique_connection_name);
	interval = (unsigned long int)microfeed_publisher_get_setting_value_integer(publisher, "update_interval", 0) * 60;
	
	microfeed_object_lock(publisher, MicrofeedPublisher);
	
	if (interval) {
		last_time = microfeed_feed_get_last_updated_time(feed);
printf("LAST UPDATE: %s %s %ld\n", publisher->identifier, uri, last_time);

		elapsed = time(NULL) - last_time;
		microfeed_provider_add_timeout(publisher->provider, (elapsed > interval ? 0 : interval - elapsed), interval, (MicrofeedProviderTimeoutFunction)microfeed_feed_update, microfeed_object_get_weak_reference(feed, MicrofeedFeed), NULL);
	}
	
	microfeed_object_unref(feed, MicrofeedFeed);

	return (MicrofeedError*)-1;
}

static MicrofeedError* method_unsubscribe_feed(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	DBusError error;
	char* uri;
	MicrofeedFeed* feed;
	
	dbus_error_init(&error);
	if (!dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_INVALID)) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri argument in UnsubscribeFeed method");
	}
	if (!(feed = microfeed_store_remove_key(subscriber->feeds, uri, MicrofeedFeed))) {
	
		return microfeed_error_new(MICROFEED_ERROR_FEED_NOT_SUBSCRIBED, "Trying to unsubscribe a feed that is not subscribed");
	}

	microfeed_object_unref(feed, MicrofeedFeed);

	return NULL;
}

static MicrofeedError* method_update_feed(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedFeed* feed;
	DBusError error;
	char* uri;

	dbus_error_init(&error);
	if (!dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_INVALID)) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri argument in UpdateFeed method");
	}
	
	if (!(feed = get_or_instantiate_feed(publisher, uri, 1))) {
	
		return microfeed_error_new(MICROFEED_ERROR_FEED_NOT_SUBSCRIBED, "Trying to update a feed that is not subscribed");
	}

	microfeed_object_unlock(publisher, MicrofeedPublisher);

	microfeed_feed_update(feed, subscriber->unique_connection_name);

	microfeed_object_lock(publisher, MicrofeedPublisher);

	microfeed_object_unref(feed, MicrofeedFeed);

	return NULL;
}

static MicrofeedError* method_create_publisher(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	/* TODO: Check that publisher does not exist. */
	
	microfeed_object_unlock(publisher, MicrofeedPublisher);

	microfeed_publisher_add_setting(publisher, "archive_old_items", "Archive old items", "hours", "non-negative integer", "5", "5");
	microfeed_publisher_add_setting(publisher, "update_interval", "Update interval", "minutes", "non-negative integer", "5", "5");

	if (publisher->callbacks.initialize_settings) {
		publisher->callbacks.initialize_settings(publisher, publisher->user_data);
	}

	microfeed_object_lock(publisher, MicrofeedPublisher);

	return NULL;
}

static MicrofeedError* method_destroy_publisher(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedStoreIterator* subscriber_iterator;
	Subscriber* feed_subscriber;
	MicrofeedStoreIterator* feed_iterator;
	MicrofeedFeed* feed;

	for (subscriber_iterator = microfeed_store_iterate(publisher->subscribers, NULL);
	     (feed_subscriber = microfeed_store_iterator_get(subscriber_iterator, Subscriber));
	     microfeed_store_iterator_next(subscriber_iterator)) {
		for (feed_iterator = microfeed_store_iterate(feed_subscriber->feeds, NULL);
		     (feed = microfeed_store_iterator_get(feed_iterator, MicrofeedFeed));
		     microfeed_store_iterator_next(feed_iterator)) {
			microfeed_object_unref(feed, MicrofeedFeed);
			microfeed_provider_send_error_signal(publisher->provider, subscriber->unique_connection_name, MICROFEED_ERROR_NO_SUCH_PUBLISHER, publisher->object_path, NULL, NULL, "Publisher destroyed.");
		}
	}

	return NULL;	
}

static MicrofeedError* method_store_data(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedError* error;
	DBusError dbus_error;
	const char* url;
	size_t url_size;
	void* data;
	size_t data_size;
	char* path;
	StoreDataRequest* store_data_request;
	void* stored_data;
	size_t stored_data_size;
	char* name;
	int i;
	DBusMessage* reply;
	char buffer[1024];

	dbus_error_init(&dbus_error);
	if (!dbus_message_get_args(message, &dbus_error, DBUS_TYPE_STRING, &url, DBUS_TYPE_INVALID)) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected an url argument in StoreData method");
	}
	url_size = strlen(url) + 1;
	if (!publisher->callbacks.store_data) {
		
		return microfeed_error_new(MICROFEED_ERROR_DATA_NOT_SUPPORTED, "Publisher does not support storing data");
	}
	if (!microfeed_database_get_data(publisher->data_database, url, url_size, &data, &data_size)) {

		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_DATA, "Requested data does not exist");
	}
	if (data_size > sizeof(uint64_t)) {
		path = microfeed_util_string_concatenate(publisher->data_directory, data + sizeof(uint64_t), NULL);
		if (microfeed_util_file_exists(path)) {
			microfeed_provider_send_data_stored_signal(publisher->provider, subscriber->unique_connection_name, publisher->object_path, url, data + sizeof(uint64_t));
			free(data);
			free(path);
			
			return NULL;
		} else {
			microfeed_database_replace_data(publisher->data_database, url, url_size, data, sizeof(uint64_t));
		}
		free(path);
	}
	if ((store_data_request = microfeed_store_get(publisher->store_data_requests, url, StoreDataRequest))) {
		store_data_request->messages = (DBusMessage**)realloc(store_data_request->messages, (store_data_request->messages_length + 1) * sizeof(DBusMessage*));
		store_data_request->messages[store_data_request->messages_length++] = dbus_message_ref(message);
		free(data);		
		
		return (MicrofeedError*)-1;
	}

	store_data_request = microfeed_memory_allocate(StoreDataRequest);
	store_data_request->url = url;
	microfeed_store_insert(publisher->store_data_requests, store_data_request);

	microfeed_object_unlock(publisher, MicrofeedPublisher);

	error = publisher->callbacks.store_data(publisher, url, &stored_data, &stored_data_size, publisher->user_data);

	microfeed_object_lock(publisher, MicrofeedPublisher);

	if (!error) {
		name = microfeed_util_create_unique_identifier();
		path = microfeed_util_string_concatenate(publisher->data_directory, name, NULL);

		microfeed_util_file_create(path, stored_data, stored_data_size);

		data_size = sizeof(uint64_t) + strlen(name) + 1;
		data = realloc(data, data_size);
		memcpy(data, name, data_size - sizeof(uint64_t));
		microfeed_database_replace_data(publisher->data_database, url, url_size, data, data_size);

		microfeed_provider_send_data_stored_signal(publisher->provider, NULL, publisher->object_path, url, name);

		free(name);
		free(path);
	}

	free(stored_data);
	free(data);

	microfeed_store_remove(publisher->store_data_requests, store_data_request);
	for (i = 0; i < store_data_request->messages_length; i++) {
		if (!error) {
			reply = dbus_message_new_method_return(store_data_request->messages[i]);
		} else {
			snprintf(buffer, 1024, "%s.%s", MICROFEED_DBUS_INTERFACE_ERROR, microfeed_error_get_name(error));
			reply = dbus_message_new_error(store_data_request->messages[i], buffer, microfeed_error_get_message(error));
		}
		microfeed_provider_send_message(publisher->provider, reply);
		dbus_message_unref(reply);
		dbus_message_unref(store_data_request->messages[i]);
	}
	free(store_data_request->messages);
	microfeed_memory_free(store_data_request);

	return error;
}

void subscriber_disconnected(MicrofeedProvider* provider, const char* unique_connection_name, void* user_data) {
	MicrofeedPublisher* publisher;
	Subscriber* subscriber;
	MicrofeedStoreIterator* iterator;
	MicrofeedFeed* feed;
	char buffer[1024];

	publisher = (MicrofeedPublisher*)user_data;

	microfeed_object_ref(publisher, MicrofeedPublisher);
	microfeed_object_lock(publisher, MicrofeedPublisher);
	
	if ((subscriber = microfeed_store_get(publisher->subscribers, unique_connection_name, Subscriber))) {
		for (iterator = microfeed_store_iterate(subscriber->feeds, NULL);
		     (feed = (MicrofeedFeed*)microfeed_store_iterator_get(iterator, MicrofeedFeed));
		     microfeed_store_iterator_next(iterator)) {
			microfeed_store_remove(subscriber->feeds, feed);
			microfeed_object_unref(feed, MicrofeedFeed);
		}
		microfeed_store_iterator_free(iterator);

		if (subscriber->reference_count == 0) {
			microfeed_store_remove(subscriber->publisher->subscribers, subscriber);
			subscriber_free(subscriber);
		}		
	} else {
		printf("UNKNOWN SUBSCRIBER DISCONNECTED: %s\n", unique_connection_name);
	}

	microfeed_object_unlock(publisher, MicrofeedPublisher);
	microfeed_object_unref(publisher, MicrofeedPublisher);
}

static MicrofeedFeed* get_or_instantiate_feed(MicrofeedPublisher* publisher, const char* uri, int create) {
	MicrofeedFeed* feed;
	MicrofeedFeed* new_feed;
	
	if (!(feed = microfeed_store_get(publisher->feeds, uri, MicrofeedFeed))) {
		if (!strcmp(uri, MICROFEED_FEED_URI_SETTINGS)) {
			microfeed_object_unlock(publisher, MicrofeedPublisher);
			
			new_feed = microfeed_feed_new(publisher, MICROFEED_FEED_URI_SETTINGS, "Settings", MICROFEED_FEED_PERMISSION_NONE, 1, &settings_callbacks, publisher);

			microfeed_object_lock(publisher, MicrofeedPublisher);
			
			/* If some other thread made the same feed... */
			if ((feed = microfeed_store_get(publisher->feeds, uri, MicrofeedFeed))) {
				microfeed_object_unref(new_feed, MicrofeedFeed);
			} else {
				microfeed_store_insert(publisher->feeds, new_feed);
				feed = new_feed;
			}
		} else {
			microfeed_object_unlock(publisher, MicrofeedPublisher);
			
			new_feed = publisher->callbacks.instantiate_feed(publisher, uri, create, publisher->user_data);
			
			microfeed_object_lock(publisher, MicrofeedPublisher);
			
			if (new_feed) {
				/* If some other thread made the same feed... */
				if ((feed = microfeed_store_get(publisher->feeds, uri, MicrofeedFeed))) {
					microfeed_object_unref(new_feed, MicrofeedFeed);
				} else {
					microfeed_store_insert(publisher->feeds, new_feed);
					feed = new_feed;
				}
			}
		}
	}
	
	return feed;
}

static MicrofeedError* settings_modify_item(MicrofeedFeed* feed, MicrofeedItem* existing_item, MicrofeedItem* new_item, void* user_data) {
	MicrofeedError* error = NULL;;
	MicrofeedPublisher* publisher;
	const char* value;
	
	publisher = (MicrofeedPublisher*)user_data;
	
	if (!existing_item) {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_ITEM, "No such setting.");
	} else {
		if ((value = microfeed_item_get_property(new_item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE))) {
			if (publisher->callbacks.update_setting(publisher, microfeed_item_get_uid(new_item), value, publisher->user_data)) {
				microfeed_item_set_property(existing_item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE, value);
				microfeed_feed_replace_item(feed, existing_item);
			}
			/* TODO: If internal setting (such as update_interval), do necessary actions. */
		} else {
			error = microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "No setting.value property in settings item.");
		}
	}
	
	return error;
}
