/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MICROFEEDPROVIDER_H
#define MICROFEEDPROVIDER_H

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>

/**
 * @addtogroup provider libmicrofeed-provider - Modules that are used in the provider side
 * @{
 * @addtogroup MicrofeedProvider
 * 
 * A providers connection to the DBus system.
 *
 * Provider instantiates existing publishers as requested by subscribers. In addition, a new or existing
 * publisher can also be instantiated using #microfeed_publisher_new function and then added into Provider
 * with the #microfeed_provider_add_publisher function. Provider also monitors name changes in the DBus and
 * informs when a subscriber has disconnected with a callback mechanism.
 * 
 * @{
 */

/**
 * An opaque datatype presenting a provider.
 */
typedef struct _MicrofeedProvider MicrofeedProvider;

/** @} */

#include <microfeed-provider/microfeedpublisher.h>
#include <microfeed-common/microfeedconfiguration.h>
#include <microfeed-common/microfeedobject.h>

/** 
 * @addtogroup MicrofeedProvider
 * @{
 */

typedef int (*MicrofeedProviderTimeoutHandler)(void* data);
typedef void (*MicrofeedProviderTimeoutFunction)(MicrofeedObject* object, void* data);

typedef MicrofeedPublisher* (*MicrofeedProviderInstantiatePublisherCallback)(MicrofeedProvider* provider, const char* publisher_identifier, const char* directory, void* user_data);
typedef void (*MicrofeedProviderNoMorePublishersCallback)(MicrofeedProvider* provider, void* user_data);
typedef void* (*MicrofeedProviderAddTimeoutCallback)(MicrofeedProvider* provider, unsigned long int interval, MicrofeedProviderTimeoutHandler handler, void* data, void* user_data);
typedef void (*MicrofeedProviderRemoveTimeoutCallback)(MicrofeedProvider* provider, void* timeout_implementation, void* user_data);


typedef void (*MicrofeedProviderSubscriberDisconnectedCallback)(MicrofeedProvider* provider, const char* unique_connection_name, void* user_data);

typedef struct _MicrofeedProviderCallbacks {
	MicrofeedProviderInstantiatePublisherCallback instantiate_publisher;
	MicrofeedProviderNoMorePublishersCallback no_more_publishers;
	MicrofeedProviderAddTimeoutCallback add_timeout;
	MicrofeedProviderRemoveTimeoutCallback remove_timeout;

} MicrofeedProviderCallbacks;

MicrofeedProvider* microfeed_provider_new(const char* bus_name, DBusConnection* connection, MicrofeedProviderCallbacks* callbacks, void* user_data);
void microfeed_provider_free(MicrofeedProvider* provider);
void microfeed_provider_add_publisher(MicrofeedProvider* provider, MicrofeedPublisher* publisher);
void microfeed_provider_remove_publisher(MicrofeedProvider* provider, MicrofeedPublisher* publisher);
DBusConnection* microfeed_provider_get_dbus_connection(MicrofeedProvider* provider);
const char* microfeed_provider_get_bus_name(MicrofeedProvider* provider);
void microfeed_provider_send_message(MicrofeedProvider* provider, DBusMessage* message);
void microfeed_provider_add_subscriber_disconnected_callback(MicrofeedProvider* provider, const char* unique_connection_name, MicrofeedProviderSubscriberDisconnectedCallback callback, void* user_data);
void microfeed_provider_remove_subscriber_disconnected_callback(MicrofeedProvider* provider, const char* unique_connection_name, MicrofeedProviderSubscriberDisconnectedCallback callback, void* user_data);
void microfeed_proivder_send_message(MicrofeedProvider* provider, DBusMessage* message);
void microfeed_provider_send_item_signal(MicrofeedProvider* provider, const char* destination, const char* signal_name, const char* object_path, const char* uri, MicrofeedItem* item);
void microfeed_provider_send_item_uid_signal(MicrofeedProvider* provider, const char* destination, const char* signal_name, const char* object_path, const char* uri, const char* uid);
void microfeed_provider_send_status_changed_signal(MicrofeedProvider* provider, const char* destination, const char* object_path, const char* uri, const char* uid, const char status);
void microfeed_provider_send_feed_signal(MicrofeedProvider* provider, const char* destination, const char* signal_name, const char* object_path, const char* uri);
void microfeed_provider_send_error_signal(MicrofeedProvider* provider, const char* destination, const char* error_name, const char* object_path, const char* uri, const char* uid, const char* error_message);
void microfeed_provider_send_item_data_signal(MicrofeedProvider* provider, const char* destination, const char* object_path, const char* uri, const char* uid, const void* data, size_t data_size);
void microfeed_provider_add_timeout(MicrofeedProvider* provider, unsigned long int first_milliseconds, unsigned long int interval_milliseconds, MicrofeedProviderTimeoutFunction function, MicrofeedWeakReference* weak_reference, void* data);
void microfeed_provider_remove_timeout(MicrofeedProvider* provider, MicrofeedProviderTimeoutFunction function, MicrofeedWeakReference* weak_reference, void* user_data);

/**
 * @}
 * @}
 */

#endif
