/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include <microfeed-provider/microfeedfeed.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeeditem.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-provider/microfeederror.h>
#include <microfeed-common/microfeedprotocol.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <stdint.h>

#define METADATA_LAST_UPDATED "last_updated"

typedef struct _ItemDataProperty ItemDataProperty;

struct _MicrofeedFeed {
	MicrofeedObject object;

	char* uri;
	char* name;
	MicrofeedPublisher* publisher;
	char* object_path;
	MicrofeedFeedPermission feed_permission;
	MicrofeedItem* metadata_item;
	MicrofeedFeedCallbacks callbacks;
	void* user_data;
	MicrofeedFeedIterator* iterators;
	int subscriber_count;
	int updating;
	MicrofeedDatabase* database;
	MicrofeedDatabaseIndex* database_index;
	MicrofeedDatabase* statuses_database;
	MicrofeedDatabase* metadata_database;
	MicrofeedDatabase* item_data_database;
	MicrofeedStore* get_item_data_mutexes;
	MicrofeedStore* item_data_properties;
};

struct _MicrofeedFeedIterator {
	MicrofeedFeedIterator* previous;
	MicrofeedFeedIterator* next;

	MicrofeedFeed* feed;
	MicrofeedDatabaseIterator* database_iterator;
	int backwards : 1;
	int timeline : 1;
};

typedef struct {
	unsigned int reference_count;
	char* uid;	
	MicrofeedMutex* mutex;
} GetItemDataMutex;

struct _ItemDataProperty {
	char* property_name;
	MicrofeedFeed* item_data_feed;
};

static void free_function(MicrofeedFeed* feed);
static const char* item_data_property_get_property_name(ItemDataProperty* item_data_property);
static void handle_item_property_change(MicrofeedFeed* feed, MicrofeedItem* old_item, MicrofeedItem* new_item);
static void index_function(void* key, const size_t key_size, void* data, const size_t data_size, void** index_key, size_t* index_key_size);
static int compare_function(const void* key1, const size_t key1_size, const void* key2, const size_t key2_size);
static void remove_old_items(MicrofeedFeed* feed);
static const char* get_item_data_mutex_get_uid(GetItemDataMutex* get_item_data_mutex);

static MicrofeedClass microfeed_feed_class = {
	"MicrofeedFeed",
	(MicrofeedObjectFreeFunction)free_function
};

MicrofeedFeed* microfeed_feed_new(MicrofeedPublisher* publisher, const char* uri, const char* name, MicrofeedFeedPermission feed_permission, MicrofeedFeedCallbacks* callbacks, void* user_data) {
	MicrofeedFeed* feed;
	char* database_name;
	char* s;
	
	feed = microfeed_object_new(&microfeed_feed_class, MicrofeedFeed);
	feed->uri = strdup(uri);
	feed->name = strdup(name);
	feed->publisher = microfeed_object_ref(publisher, MicrofeedPublisher);
	feed->object_path = strdup(microfeed_publisher_get_object_path(publisher));
	feed->feed_permission = feed_permission;
	feed->metadata_item = microfeed_item_new(MICROFEED_ITEM_UID_FEED_METADATA, time(NULL));
	microfeed_item_set_property(feed->metadata_item, MICROFEED_ITEM_PROPERTY_NAME_FEED_NAME, name);
	s = microfeed_feed_permission_to_string(feed_permission);
	if (s && *s) {
		microfeed_item_set_property(feed->metadata_item, MICROFEED_ITEM_PROPERTY_NAME_FEED_PERMISSION, s);
	}
	free(s);
	feed->iterators = NULL;
	if (callbacks) {
		feed->callbacks = *callbacks;
	}
	feed->user_data = user_data;

	feed->database = microfeed_database_environment_get_database(microfeed_publisher_get_database_environment(publisher), uri, compare_function);
	feed->database_index = microfeed_database_get_index(feed->database, "uid", index_function);
	database_name = microfeed_util_string_concatenate(uri, ":microfeed.statuses", NULL);
	feed->statuses_database = microfeed_database_environment_get_database(microfeed_publisher_get_database_environment(publisher), database_name, NULL);
	free(database_name);
	database_name = microfeed_util_string_concatenate(uri, ":microfeed.metadata", NULL);
	feed->metadata_database = microfeed_database_environment_get_database(microfeed_publisher_get_database_environment(publisher), database_name, NULL);
	free(database_name);
	if (feed->callbacks.download_item_data) {
		database_name = microfeed_util_string_concatenate(uri, ":microfeed.item_data", NULL);
		feed->item_data_database = microfeed_database_environment_get_database(microfeed_publisher_get_database_environment(publisher), database_name, NULL);
		free(database_name);
		feed->get_item_data_mutexes = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)get_item_data_mutex_get_uid);
	}
	feed->item_data_properties = microfeed_store_new_sorted_data((MicrofeedStoreCompareKeysFunction)strcmp, (MicrofeedStoreGetKeyFunction)item_data_property_get_property_name);

	return feed;	
}

/* Does NOT remove subscribers */
static void free_function(MicrofeedFeed* feed) {
	MicrofeedFeedIterator* iterator;
	MicrofeedStoreIterator* store_iterator;
	ItemDataProperty* item_data_property;

	if (feed->callbacks.destroy) {
		feed->callbacks.destroy(feed, feed->user_data);
	}
	
	free(feed->uri);
	free(feed->name);
	microfeed_object_unref(feed->publisher, MicrofeedPublisher);
	free(feed->object_path);
	microfeed_item_free(feed->metadata_item);
	for (iterator = feed->iterators; iterator; iterator = iterator->next) {
		iterator->feed = NULL;
	}	
	
	microfeed_object_unref(feed->database, MicrofeedDatabase);
	microfeed_object_unref(feed->statuses_database, MicrofeedDatabase);
	microfeed_object_unref(feed->metadata_database, MicrofeedDatabase);
	if (feed->item_data_database) {
		microfeed_object_unref(feed->item_data_database, MicrofeedDatabase);
		microfeed_store_free(feed->get_item_data_mutexes);
	}

	for (store_iterator = microfeed_store_iterate(feed->item_data_properties, NULL);
	     (item_data_property = microfeed_store_iterator_get(store_iterator, ItemDataProperty));
	     microfeed_store_iterator_next(store_iterator)) {
		free(item_data_property->property_name);
		microfeed_object_unref(item_data_property->item_data_feed, MicrofeedFeed);
		microfeed_memory_free(item_data_property);
	}
	microfeed_store_iterator_free(store_iterator);
	microfeed_store_free(feed->item_data_properties);

/* TODO: Free metadata item etc. */
	
	feed->uri = NULL;
	feed->callbacks.destroy = NULL;
	feed->callbacks.update = NULL;
	feed->subscriber_count = 0;
	feed->iterators = NULL;
}

MicrofeedPublisher* microfeed_feed_get_publisher(MicrofeedFeed* feed) {

	return feed->publisher;
}

void microfeed_feed_replace_item(MicrofeedFeed* feed, MicrofeedItem* item) {
	const char* uid;
	size_t uid_size;
	uint64_t timestamp;
	void* key;
	size_t key_size;
	const void* data;
	size_t data_size;
	void* existing_key;
	size_t existing_key_size;
	void* existing_data;
	size_t existing_size;
	uint64_t existing_timestamp;
	void* status_data;
	size_t status_data_size;
	char status;
	MicrofeedItem* existing_item;

	microfeed_object_lock(feed, MicrofeedFeed);

	uid = microfeed_item_get_uid(item);
	uid_size = strlen(uid) + 1;
	timestamp = (uint64_t)microfeed_item_get_timestamp(item);
	key_size = sizeof(uint64_t) + uid_size;
	key = (char*)malloc(key_size);
	*((uint64_t*)key) = timestamp;
	memcpy(key + sizeof(uint64_t), uid, uid_size);
	microfeed_item_marshal_properties(item, &data, &data_size);
	if (microfeed_database_index_get_data(feed->database_index, uid, uid_size, &existing_key, &existing_key_size, &existing_data, &existing_size)) {
		existing_timestamp = *((uint64_t*)existing_key);
		if (timestamp != existing_timestamp || data_size != existing_size || memcmp(data, existing_data, existing_size)) {
			if (microfeed_database_get_data(feed->statuses_database, uid, uid_size, &status_data, &status_data_size)) {
				status = *((char*)status_data);
				free(status_data);
				if (!(status & (char)MICROFEED_ITEM_STATUS_ACTIVE)) {
					status |= (char)(MICROFEED_ITEM_STATUS_ACTIVE);
					microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
				}			
			} else {
				status = (char)MICROFEED_ITEM_STATUS_ACTIVE;
				microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
			}
				
			existing_item = microfeed_item_new_with_status(uid, existing_timestamp, status);
			microfeed_item_demarshal_properties(existing_item, existing_data, existing_size);
			handle_item_property_change(feed, existing_item, item);
			microfeed_item_free(existing_item);
			
			microfeed_provider_send_item_signal(microfeed_publisher_get_provider(feed->publisher), NULL, MICROFEED_SIGNAL_NAME_ITEM_CHANGED, feed->object_path, feed->uri, item);
//			if (timestamp != existing_timestamp) {
				microfeed_database_index_remove_data(feed->database_index, uid, uid_size);
//			}
			microfeed_database_replace_data(feed->database, key, key_size, data, data_size);
		}
		free(existing_key);
		free(existing_data);
	} else {	
		status = (char)MICROFEED_ITEM_STATUS_NEW;
		handle_item_property_change(feed, NULL, item);
		microfeed_provider_send_item_signal(microfeed_publisher_get_provider(feed->publisher), NULL, MICROFEED_SIGNAL_NAME_ITEM_ADDED, feed->object_path, feed->uri, item);
		microfeed_database_replace_data(feed->database, key, key_size, data, data_size);
		microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
	}
	free(key);
	
	microfeed_object_unlock(feed, MicrofeedFeed);
}

void microfeed_feed_remove_item(MicrofeedFeed* feed, const char* uid) {	
	size_t uid_size;
	void* key;
	size_t key_size;
	void* data;
	size_t data_size;
	uint64_t timestamp;
	MicrofeedItem* item;

	microfeed_object_lock(feed, MicrofeedFeed);

	uid_size = strlen(uid) + 1;
	if (microfeed_database_index_get_data(feed->database_index, uid, uid_size, &key, &key_size, &data, &data_size)) {
		timestamp = *((uint64_t*)key);
		item = microfeed_item_new(uid, timestamp);
		microfeed_item_demarshal_properties(item, data, data_size);
		handle_item_property_change(feed, item, NULL);
		microfeed_item_free(item);
		free(key);
		free(data);
		
		microfeed_database_index_remove_data(feed->database_index, uid, uid_size);
		microfeed_database_remove_data(feed->statuses_database, uid, uid_size);
		if (feed->item_data_database) {
			microfeed_database_remove_data(feed->item_data_database, uid, uid_size);
		}
		microfeed_provider_send_item_uid_signal(microfeed_publisher_get_provider(feed->publisher), NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, feed->object_path, feed->uri, uid);
	}

	microfeed_object_unlock(feed, MicrofeedFeed);
}

void microfeed_feed_remove_items(MicrofeedFeed* feed, const char* start_uid, const char* end_uid) {
	size_t uid_size;
	MicrofeedDatabaseIterator* iterator;
	const void* key;
	size_t key_size;
	const void* data;
	size_t data_size;
	uint64_t timestamp;
	MicrofeedItem* item;
	
	microfeed_object_lock(feed, MicrofeedFeed);
	
	uid_size = (start_uid ? strlen(start_uid) + 1 : 0);
	for (iterator = microfeed_database_index_iterate(feed->database_index, start_uid, uid_size, 0);
	     (microfeed_database_iterator_get(iterator, &key, &key_size, &data, &data_size)) && (!end_uid || strcmp ((const char*)key, end_uid) <= 0);
	     microfeed_database_iterator_next(iterator)) {
		timestamp = *((uint64_t*)key);
		item = microfeed_item_new(key + sizeof(uint64_t), timestamp);
		microfeed_item_demarshal_properties(item, data, data_size);
		handle_item_property_change(feed, item, NULL);
		microfeed_item_free(item);
		
		microfeed_database_index_remove_data(feed->database_index, key + sizeof(uint64_t), key_size - sizeof(uint64_t));
		microfeed_database_remove_data(feed->statuses_database, key + sizeof(uint64_t), key_size - sizeof(uint64_t));
		if (feed->item_data_database) {
			microfeed_database_remove_data(feed->item_data_database, key + sizeof(uint64_t), key_size - sizeof(uint64_t));
		}
		microfeed_provider_send_item_uid_signal(microfeed_publisher_get_provider(feed->publisher), NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, feed->object_path, feed->uri, key + sizeof(uint64_t));
	}
	microfeed_database_iterator_free(iterator);
	
	microfeed_object_unlock(feed, MicrofeedFeed);	
}

MicrofeedItem* microfeed_feed_get_item(MicrofeedFeed* feed, const char* uid) {
	void* key;
	size_t key_size;
	void* data;
	size_t data_size;
	uint64_t timestamp;
	MicrofeedItem* item;

	microfeed_object_lock(feed, MicrofeedFeed);
	
	if (!microfeed_database_index_get_data(feed->database_index, uid, strlen(uid) + 1, &key, &key_size, &data, &data_size)) {
		item = NULL;
	} else {
		timestamp = *((uint64_t*)key);
		item = microfeed_item_new((char*)(key + sizeof(uint64_t)), (time_t)timestamp);
		microfeed_item_demarshal_properties(item, data, data_size);
		free(key);
		free(data);
	}

	microfeed_object_unlock(feed, MicrofeedFeed);
	
	return item;	
}

const char* microfeed_feed_get_uri(MicrofeedFeed* feed) {

	return feed->uri;
}

const char* microfeed_feed_get_name(MicrofeedFeed* feed) {

	/* TODO: What if the name changes? */
	return feed->name;
}

void microfeed_feed_set_name(MicrofeedFeed* feed, const char* name) {
	microfeed_object_lock(feed, MicrofeedFeed);

	if (strcmp(name, feed->name)) {
		free(feed->name);
		feed->name = strdup(name);
		/* TODO: Change metadata item. */
	}

	microfeed_object_unlock(feed, MicrofeedFeed);
}

MicrofeedItem* microfeed_feed_get_metadata_item(MicrofeedFeed* feed) {
	MicrofeedItem* item;

	microfeed_object_lock(feed, MicrofeedFeed);

	item = microfeed_item_duplicate(feed->metadata_item);

	microfeed_object_unlock(feed, MicrofeedFeed);

	return item;
}

MicrofeedItem* microfeed_feed_send_metadata_item(MicrofeedFeed* feed, const char* destination) {
	MicrofeedItem* item;

	microfeed_object_lock(feed, MicrofeedFeed);

	microfeed_provider_send_item_signal(microfeed_publisher_get_provider(feed->publisher), destination, MICROFEED_SIGNAL_NAME_ITEM_ADDED, feed->object_path, feed->uri, feed->metadata_item);

	microfeed_object_unlock(feed, MicrofeedFeed);
}



MicrofeedFeedIterator* microfeed_feed_iterate(MicrofeedFeed* feed, const char* start_uid, int backwards) {
	MicrofeedFeedIterator* iterator;

	microfeed_object_lock(feed, MicrofeedFeed);
	
	iterator = (MicrofeedFeedIterator*)microfeed_memory_allocate(MicrofeedFeedIterator);
	iterator->feed = feed;
	iterator->backwards = backwards;
	iterator->timeline = 0;
	iterator->database_iterator = microfeed_database_index_iterate(feed->database_index, start_uid, (start_uid ? strlen(start_uid) + 1 : 0), backwards);

	if (feed->iterators) {
		feed->iterators->previous = iterator;
		iterator->next = feed->iterators;
	} else {
		iterator->next = NULL;
	}
	iterator->previous = NULL;
	feed->iterators = iterator;

	microfeed_object_unlock(feed, MicrofeedFeed);

	return iterator;
}

MicrofeedFeedIterator* microfeed_feed_iterate_timeline(MicrofeedFeed* feed, const time_t start_timestamp, int backwards) {
	MicrofeedFeedIterator* iterator;
	char* key;

	microfeed_object_lock(feed, MicrofeedFeed);
	
	iterator = (MicrofeedFeedIterator*)microfeed_memory_allocate(MicrofeedFeedIterator);
	iterator->feed = feed;
	iterator->backwards = backwards;
	iterator->timeline = 1;
	if (start_timestamp) {
		key = (char*)malloc(sizeof(uint64_t) + 1);
		*((uint64_t*)key) = (uint64_t)start_timestamp;
		*(key + sizeof(uint64_t)) = 0;
		iterator->database_iterator = microfeed_database_iterate(feed->database, key, sizeof(uint64_t) + 1, backwards);
		free(key);
	} else {
		iterator->database_iterator = microfeed_database_iterate(feed->database, NULL, 0, backwards);
	}		

	if (feed->iterators) {
		feed->iterators->previous = iterator;
		iterator->next = feed->iterators;
	} else {
		iterator->next = NULL;
	}
	iterator->previous = NULL;
	feed->iterators = iterator;

	microfeed_object_unlock(feed, MicrofeedFeed);

	return iterator;
}

void microfeed_feed_update(MicrofeedFeed* feed, const char* bus_name) {
	MicrofeedError* error;
	long timestamp;

	microfeed_object_lock(feed, MicrofeedFeed);

	if (feed->updating) {
		if (bus_name) {
			microfeed_provider_send_feed_signal(microfeed_publisher_get_provider(feed->publisher), bus_name, MICROFEED_SIGNAL_NAME_FEED_UPDATE_STARTED, feed->object_path, feed->uri);
		}
	} else {
		microfeed_provider_send_feed_signal(microfeed_publisher_get_provider(feed->publisher), NULL, MICROFEED_SIGNAL_NAME_FEED_UPDATE_STARTED, feed->object_path, feed->uri);
		feed->updating = 1;
	
		if (feed->callbacks.update) {

			microfeed_object_unlock(feed, MicrofeedFeed);
				
			error = feed->callbacks.update(feed, (bus_name ? 1 : 0), feed->user_data);

			microfeed_object_lock(feed, MicrofeedFeed);
				
			if (error) {
				microfeed_provider_send_error_signal(microfeed_publisher_get_provider(feed->publisher), NULL, microfeed_error_get_name(error), feed->object_path, feed->uri, NULL, microfeed_error_get_message(error));
				microfeed_error_free(error);
			} else {
				timestamp = (long)time(NULL);
				microfeed_database_replace_data(feed->metadata_database, METADATA_LAST_UPDATED, strlen(METADATA_LAST_UPDATED) + 1, &timestamp, sizeof(long));
			}
		}

		microfeed_provider_send_feed_signal(microfeed_publisher_get_provider(feed->publisher), NULL, MICROFEED_SIGNAL_NAME_FEED_UPDATE_ENDED, feed->object_path, feed->uri);
		feed->updating = 0;
	}

	microfeed_object_unlock(feed, MicrofeedFeed);
}

void microfeed_feed_republish(MicrofeedFeed* feed, const char* start_uid, const char* end_uid, unsigned int max_count, const char* bus_name) {
	unsigned int count;
	MicrofeedDatabaseIterator* iterator;
	const void* key;
	size_t key_size;
	const void* item_data;
	size_t item_data_size;
	uint64_t timestamp;
	MicrofeedItem* item;
	void* status_data;
	size_t status_data_size;
	char status;
	
	microfeed_object_lock(feed, MicrofeedFeed);
	
	microfeed_provider_send_feed_signal(microfeed_publisher_get_provider(feed->publisher), bus_name, MICROFEED_SIGNAL_NAME_FEED_REPUBLISHING_STARTED, feed->object_path, feed->uri);

	count = 0;
	for (iterator = microfeed_database_iterate(feed->database, end_uid, (end_uid ? strlen(end_uid) + 1 : 0), 1);
	     microfeed_database_iterator_get(iterator, &key, &key_size, &item_data, &item_data_size);
	     microfeed_database_iterator_next(iterator)) {
	    timestamp = *((uint64_t*)key);
		if (microfeed_database_get_data(feed->statuses_database, key + sizeof(uint64_t), key_size - sizeof(uint64_t), &status_data, &status_data_size)) {
			status = *((char*)status_data);
		} else {
			status = (char)MICROFEED_ITEM_STATUS_NONE;
		}
		item = microfeed_item_new_with_status(key + sizeof(uint64_t), (time_t)timestamp, (MicrofeedItemStatus)status);
		microfeed_item_demarshal_properties(item, item_data, item_data_size);
		microfeed_provider_send_item_signal(microfeed_publisher_get_provider(feed->publisher), bus_name, MICROFEED_SIGNAL_NAME_ITEM_REPUBLISHED, feed->object_path, feed->uri, item);
		microfeed_item_free(item);
		free(status_data);
	    if (start_uid && strcmp(start_uid, key + sizeof(uint64_t)) <= 0) {
	    	break;
	    }
	    if (++count == max_count) {
	    	break;
	    }
	}
	microfeed_database_iterator_free(iterator);

	microfeed_provider_send_feed_signal(microfeed_publisher_get_provider(feed->publisher), bus_name, MICROFEED_SIGNAL_NAME_FEED_REPUBLISHING_ENDED, feed->object_path, feed->uri);

	microfeed_object_unlock(feed, MicrofeedFeed);	
}

MicrofeedError* microfeed_feed_set_item_status(MicrofeedFeed* feed, const char* uid, MicrofeedItemStatus status_to_set) {
	MicrofeedError* error = NULL;
	size_t uid_size;
	void* key;
	size_t key_size;
	void* data;
	size_t data_size;
	char status;
	
	microfeed_object_lock(feed, MicrofeedFeed);	

	uid_size = strlen(uid) + 1;
	if (microfeed_database_get_data(feed->statuses_database, uid, uid_size, &data, &data_size)) {
		status = *((char*)data);
		if (!(status & (char)status_to_set)) {
			if (status_to_set == MICROFEED_ITEM_STATUS_MARKED && feed->callbacks.mark_item) {
				error = feed->callbacks.mark_item(feed, uid, 1, feed->user_data);
			}
			if (!error) {
				status |= (char)(status_to_set);
				microfeed_provider_send_status_changed_signal(microfeed_publisher_get_provider(feed->publisher), NULL, feed->object_path, feed->uri, uid, status);
				if (status) {
					microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
				} else {
					microfeed_database_remove_data(feed->statuses_database, uid, uid_size);
				}
			}
		}
	} else if (microfeed_database_index_get_data(feed->database_index, uid, uid_size, &key, &key_size, &data, &data_size)) {
		if (status_to_set == MICROFEED_ITEM_STATUS_MARKED && feed->callbacks.mark_item) {
			error = feed->callbacks.mark_item(feed, uid, 1, feed->user_data);
		}
		if (!error) {
			status = (char)status_to_set;
			microfeed_provider_send_status_changed_signal(microfeed_publisher_get_provider(feed->publisher), NULL, feed->object_path, feed->uri, uid, status);
			microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
		}
	} else {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_ITEM, "No such item when setting status.");
	}

	microfeed_object_unlock(feed, MicrofeedFeed);	

	return error;
}

MicrofeedError* microfeed_feed_unset_item_status(MicrofeedFeed* feed, const char* uid, MicrofeedItemStatus status_to_unset) {
	MicrofeedError* error = NULL;
	size_t uid_size;
	void* data;
	size_t data_size;
	char status;
	
	microfeed_object_lock(feed, MicrofeedFeed);	

	uid_size = strlen(uid) + 1;
	if (microfeed_database_get_data(feed->statuses_database, uid, uid_size, &data, &data_size)) {
		status = *((char*)data);
		if (status & (char)status_to_unset) {
			if (status_to_unset == MICROFEED_ITEM_STATUS_MARKED && feed->callbacks.mark_item) {
				error = feed->callbacks.mark_item(feed, uid, 0, feed->user_data);
			}
			if (!error) {
				status &= (char)(~status_to_unset);
				microfeed_provider_send_status_changed_signal(microfeed_publisher_get_provider(feed->publisher), NULL, feed->object_path, feed->uri, uid, status);
				if (status) {
					microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
				} else {
					microfeed_database_remove_data(feed->statuses_database, uid, uid_size);
				}
			}
		}			
	} else {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_ITEM, "No such item when unsetting status.");
	}

	microfeed_object_unlock(feed, MicrofeedFeed);	

	return error;
}

MicrofeedError* microfeed_feed_unset_item_statuses(MicrofeedFeed* feed, const char* start_uid, const char* end_uid, MicrofeedItemStatus status_to_unset) {
	MicrofeedError* error = NULL;
	size_t uid_size;
	MicrofeedDatabaseIterator* iterator;
	const void* key;
	size_t key_size;
	const void* data;
	size_t data_size;
	char status;
	
	microfeed_object_lock(feed, MicrofeedFeed);
	
	uid_size = (start_uid ? strlen(start_uid) + 1 : 0);
	for (iterator = microfeed_database_iterate(feed->statuses_database, start_uid, uid_size, 0);
	     !error && (microfeed_database_iterator_get(iterator, &key, &key_size, &data, &data_size)) && (!end_uid || strcmp ((const char*)key, end_uid) <= 0);
	     microfeed_database_iterator_next(iterator)) {
		status = *((char*)data);
		if (status & (char)status_to_unset) {
			if (status_to_unset == MICROFEED_ITEM_STATUS_MARKED && feed->callbacks.mark_item) {
				error = feed->callbacks.mark_item(feed, key, 0, feed->user_data);
			}
			if (!error) {
				status &= (char)(~status_to_unset);
				microfeed_provider_send_status_changed_signal(microfeed_publisher_get_provider(feed->publisher), NULL, feed->object_path, feed->uri, (const char*)key, status);
				if (status) {
					microfeed_database_replace_data(feed->statuses_database, key, key_size, &status, sizeof(char));
				} else {
					microfeed_database_remove_data(feed->statuses_database, key, key_size);
				}
			}
		}     	
	}
	microfeed_database_iterator_free(iterator);
	
	microfeed_object_unlock(feed, MicrofeedFeed);	

	return error;
}

MicrofeedError* microfeed_feed_send_item_data(MicrofeedFeed* feed, const char* uid, const char* bus_name) {
	MicrofeedError* error = NULL;
	void* key;
	size_t key_size;
	void* data = NULL;
	size_t data_size;
	GetItemDataMutex* gidm;
	
	microfeed_object_lock(feed, MicrofeedFeed);
	
	if (!feed->item_data_database) {
		error = microfeed_error_new(MICROFEED_ERROR_NO_ITEM_DATA, "Feed does not support item data");
	} else if (microfeed_database_get_data(feed->item_data_database, uid, strlen(uid) + 1, &data, &data_size) && data_size > sizeof(uint64_t)) {
		microfeed_provider_send_item_data_signal(microfeed_publisher_get_provider(feed->publisher), bus_name, feed->object_path, feed->uri, uid, data + sizeof(uint64_t), data_size - sizeof(uint64_t));
		free(data);
	} else if (free(data), !microfeed_database_index_get_data(feed->database_index, uid, strlen(uid) + 1, &key, &key_size, &data, &data_size)) {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_ITEM, "Item does not exist");
	} else {
		free(key);
		free(data);
		
		if ((gidm = microfeed_store_get(feed->get_item_data_mutexes, uid, GetItemDataMutex))) {
			gidm->reference_count++;

			microfeed_object_unlock(feed, MicrofeedFeed);

			microfeed_mutex_lock(gidm->mutex);

			microfeed_object_lock(feed, MicrofeedFeed);
			if (microfeed_database_get_data(feed->item_data_database, uid, strlen(uid) + 1, &data, &data_size) && data_size > sizeof(uint64_t)) {
				microfeed_provider_send_item_data_signal(microfeed_publisher_get_provider(feed->publisher), bus_name, feed->object_path, feed->uri, uid, data + sizeof(uint64_t), data_size - sizeof(uint64_t));
				free(data);
			} else {
				error = microfeed_error_new(MICROFEED_ERROR_NO_ITEM_DATA, "Item data does not exist");
			}
		} else {
			gidm = microfeed_memory_allocate(GetItemDataMutex);
			gidm->reference_count = 1;
			gidm->uid = strdup(uid);
			gidm->mutex = microfeed_mutex_new();
			microfeed_mutex_lock(gidm->mutex);
			microfeed_store_insert(feed->get_item_data_mutexes, gidm);

			microfeed_object_unlock(feed, MicrofeedFeed);

			error = feed->callbacks.download_item_data(feed, uid, &data, &data_size, feed->user_data);

			microfeed_object_lock(feed, MicrofeedFeed);

			if (!error) {
				microfeed_provider_send_item_data_signal(microfeed_publisher_get_provider(feed->publisher), bus_name, feed->object_path, feed->uri, uid, data, data_size);

				data = realloc(data, data_size + sizeof(uint64_t));
				memmove(data + sizeof(uint64_t), data, data_size);
				*((uint64_t*)data) = 1;
				microfeed_database_replace_data(feed->item_data_database, uid, strlen(uid) + 1, data, data_size + sizeof(uint64_t));
				free(data);
			}
		}
		gidm->reference_count--;
		if (gidm->reference_count == 0) {
			microfeed_store_remove(feed->get_item_data_mutexes, gidm);
			free(gidm->uid);
			microfeed_mutex_free(gidm->mutex);
			microfeed_memory_free(gidm);
		} else {
			microfeed_mutex_unlock(gidm->mutex);
		}
	}
	
	microfeed_object_unlock(feed, MicrofeedFeed);	
	
	return error;
}

void microfeed_feed_ref_item_data(MicrofeedFeed* feed, const char* uid) {
	size_t uid_size;
	uint64_t reference_count;
	size_t size;
	MicrofeedItem* item;
	uint64_t timestamp;
	void* key;
	size_t key_size;
	const void* data;
	size_t data_size;
	char status;

	microfeed_object_lock(feed, MicrofeedFeed);

	if (feed->item_data_database) {
		uid_size = strlen(uid) + 1;
		size = sizeof(uint64_t);
		if (microfeed_database_get_data_partial(feed->item_data_database, uid, uid_size, &reference_count, &size, 0) && size == sizeof(uint64_t)) {
			reference_count++;
			microfeed_database_replace_data_partial(feed->item_data_database, uid, uid_size, &reference_count, size, 0);
		} else {
			reference_count = 1;
			microfeed_database_replace_data(feed->item_data_database, uid, uid_size, &reference_count, size);
			item = microfeed_item_new(uid, time(NULL));
			timestamp = (uint64_t)microfeed_item_get_timestamp(item);
			key_size = sizeof(uint64_t) + uid_size;
			key = (char*)malloc(key_size);
			*((uint64_t*)key) = timestamp;
			memcpy(key + sizeof(uint64_t), uid, uid_size);
			microfeed_item_marshal_properties(item, &data, &data_size);
			microfeed_provider_send_item_signal(microfeed_publisher_get_provider(feed->publisher), NULL, MICROFEED_SIGNAL_NAME_ITEM_ADDED, feed->object_path, feed->uri, item);
			microfeed_database_replace_data(feed->database, key, key_size, data, data_size);
			microfeed_database_replace_data(feed->statuses_database, uid, uid_size, &status, sizeof(char));
			free(key);
		}
	}

	microfeed_object_unlock(feed, MicrofeedFeed);
}

void microfeed_feed_unref_item_data(MicrofeedFeed* feed, const char* uid) {
	size_t uid_size;
	uint64_t reference_count;
	size_t size;

	microfeed_object_lock(feed, MicrofeedFeed);
	
	if (feed->item_data_database) {	
		uid_size = strlen(uid) + 1;
		size = sizeof(uint64_t);
		if (microfeed_database_get_data_partial(feed->item_data_database, uid, uid_size, &reference_count, &size, 0) && size == sizeof(uint64_t)) {
			reference_count--;
			if (reference_count == 0) {
				microfeed_database_index_remove_data(feed->database_index, uid, uid_size);
				microfeed_database_remove_data(feed->statuses_database, uid, uid_size);
				microfeed_database_remove_data(feed->item_data_database, uid, uid_size);
				microfeed_provider_send_item_uid_signal(microfeed_publisher_get_provider(feed->publisher), NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, feed->object_path, feed->uri, uid);				
			} else {
				microfeed_database_replace_data_partial(feed->item_data_database, uid, uid_size, &reference_count, size, 0);
			}
		}
	}
	
	microfeed_object_unlock(feed, MicrofeedFeed);
}

/* TODO: Should use defines. */
MicrofeedFeedPermission microfeedfeed_permission_from_string(const char* string) {
	MicrofeedFeedPermission feed_permission = MICROFEED_FEED_PERMISSION_NONE;

	while (*string == ':') {
		string++;
		if (!strncmp(string, "add", 3) && (string[3] == 0 || string[3] == ':')) {
			feed_permission |= MICROFEED_FEED_PERMISSION_ADD;
			string += 3;
		}
	}
	
	return feed_permission;
}

/* TODO: Should use defines. */
char* microfeed_feed_permission_to_string(MicrofeedFeedPermission feed_permission) {
	char buffer[1024];
	char* string = buffer;
	
	if (feed_permission & MICROFEED_FEED_PERMISSION_ADD) {
		memcpy(string, ":add", 4);
		string += 4;
	}
	*string = 0;
	
	return strdup(buffer);
}

void microfeed_feed_add_item_data_property(MicrofeedFeed* feed, const char* property_name, MicrofeedFeed* item_data_feed) {
	ItemDataProperty* item_data_property;
	
	item_data_property = microfeed_memory_allocate(ItemDataProperty);
	item_data_property->property_name = strdup(property_name);
	item_data_property->item_data_feed = microfeed_object_ref(item_data_feed, MicrofeedFeed);
	microfeed_store_insert(feed->item_data_properties, item_data_property);
}

long microfeed_feed_get_last_updated_time(MicrofeedFeed* feed) {
	long last_updated_time = 0;
	void* data;
	size_t data_size;	

	if ((microfeed_database_get_data(feed->metadata_database, METADATA_LAST_UPDATED, strlen(METADATA_LAST_UPDATED) + 1, &data, &data_size))) {
		if (data_size == sizeof(long)) {
			last_updated_time = *((long*)data);
		}
		free(data);
	}
	
	return last_updated_time;
}

void microfeed_feed_iterator_free(MicrofeedFeedIterator* iterator) {
	if (iterator->next) {
		iterator->next->previous = iterator->previous;
	}
	if (iterator->previous) {
		iterator->previous->next = iterator->next;
	} else if (iterator->feed) {
		iterator->feed->iterators = iterator->next;
	}

	iterator->feed = NULL;
	microfeed_database_iterator_free(iterator->database_iterator);
	microfeed_memory_free(iterator);
}

MicrofeedFeed* microfeed_feed_iterator_get_feed(MicrofeedFeedIterator* iterator) {

	return iterator->feed;
}

MicrofeedItem* microfeed_feed_iterator_get_item(MicrofeedFeedIterator* iterator) {
	const void* key;
	size_t key_size;
	const void* data;
	size_t data_size;
	uint64_t timestamp;
	MicrofeedItem* item;

	microfeed_object_lock(iterator->feed, MicrofeedFeed);
	
	if (!microfeed_database_iterator_get(iterator->database_iterator, &key, &key_size, &data, &data_size)) {
		item = NULL;
	} else {
		timestamp = *((uint64_t*)key);
		item = microfeed_item_new(key + sizeof(uint64_t), (time_t)timestamp);
		microfeed_item_demarshal_properties(item, data, data_size);
	}

	microfeed_object_unlock(iterator->feed, MicrofeedFeed);
	
	return item;
}

void microfeed_feed_iterator_next(MicrofeedFeedIterator* iterator) {
	microfeed_database_iterator_next(iterator->database_iterator);
}

int microfeed_feed_iterator_jump_and_remove_previous_items(MicrofeedFeedIterator* iterator, const char* uid) {
	int retval = 0;
	size_t uid_size;
	void* start_key;
	size_t start_key_size;
	void* end_key;
	size_t end_key_size;
	void* jump_key;
	size_t jump_key_size;
	const void* key;
	size_t key_size;
	void* data;
	const void* const_data;
	size_t data_size;
	
	microfeed_object_lock(iterator->feed, MicrofeedFeed);
	
	uid_size = strlen(uid) + 1;
	if (microfeed_database_index_get_data(iterator->feed->database_index, uid, uid_size, &jump_key, &jump_key_size, &data, &data_size)) {
		retval = 1;
		free(data);
		if (microfeed_database_iterator_get(iterator->database_iterator, &key, &key_size, &const_data, &data_size) && (key_size - sizeof(uint64_t) != uid_size || memcmp(key + sizeof(uint64_t), uid, uid_size))) {
			start_key = malloc(key_size);
			memcpy(start_key, key, key_size);
			start_key_size = key_size;
			end_key = malloc(key_size);
			memcpy(end_key, key, key_size);
			end_key_size = key_size;
			while (microfeed_database_iterator_get(iterator->database_iterator, &key, &key_size, &const_data, &data_size) && (key_size != jump_key_size || memcmp(key, jump_key, jump_key_size))) {
				end_key = realloc(end_key, key_size);
				memcpy(end_key, key, key_size);
				end_key_size = key_size;
				microfeed_provider_send_item_uid_signal(microfeed_publisher_get_provider(iterator->feed->publisher), NULL, MICROFEED_SIGNAL_NAME_ITEM_REMOVED, iterator->feed->object_path, iterator->feed->uri, key + sizeof(uint64_t));
				microfeed_database_iterator_next(iterator->database_iterator);
			}
			if (iterator->backwards) {		
				if (iterator->timeline) {
					microfeed_database_remove_data_range(iterator->feed->database, end_key, end_key_size, start_key, start_key_size);
				} else {
					microfeed_database_index_remove_data_range(iterator->feed->database_index, end_key + sizeof(uint64_t), end_key_size - sizeof(uint64_t), start_key + sizeof(uint64_t), start_key_size - sizeof(uint64_t));			
				}
				microfeed_database_remove_data_range(iterator->feed->statuses_database, end_key + sizeof(uint64_t), end_key_size - sizeof(uint64_t), start_key + sizeof(uint64_t), start_key_size - sizeof(uint64_t));
			} else {
				if (iterator->timeline) {
					microfeed_database_remove_data_range(iterator->feed->database, start_key, start_key_size, end_key, end_key_size);
				} else {
					microfeed_database_index_remove_data_range(iterator->feed->database_index, start_key + sizeof(uint64_t), start_key_size - sizeof(uint64_t), end_key + sizeof(uint64_t), end_key_size - sizeof(uint64_t));
				}
				microfeed_database_remove_data_range(iterator->feed->statuses_database, start_key + sizeof(uint64_t), start_key_size - sizeof(uint64_t), end_key + sizeof(uint64_t), end_key_size - sizeof(uint64_t));
			}
			free(start_key);
			free(end_key);
		}
		free(jump_key);
	}
	
	microfeed_object_unlock(iterator->feed, MicrofeedFeed);

	return retval;
}

MicrofeedError* microfeed_feed_call_modify_item_callback(MicrofeedFeed* feed, MicrofeedItem* existing_item, MicrofeedItem* new_item) {
	MicrofeedError* error;
	
	microfeed_object_lock(feed, MicrofeedFeed);

/* TODO: There must NOT be on-going update. Wait somewhere until update ends... */

	if (feed->callbacks.modify_item) {

		microfeed_object_unlock(feed, MicrofeedFeed);

		error = feed->callbacks.modify_item(feed, existing_item, new_item, feed->user_data);

		microfeed_object_lock(feed, MicrofeedFeed);
	
		if (error) {
			microfeed_provider_send_error_signal(microfeed_publisher_get_provider(feed->publisher), NULL, microfeed_error_get_name(error), feed->object_path, feed->uri, microfeed_item_get_uid((existing_item ? existing_item : new_item)), microfeed_error_get_message(error));
			microfeed_error_free(error);
		}
	} else {
		/* TODO: Send CannotModifyItem signal */
	}
	
	microfeed_object_unlock(feed, MicrofeedFeed);
}

static const char* item_data_property_get_property_name(ItemDataProperty* item_data_property) {

	return item_data_property->property_name;
}

static void handle_item_property_change(MicrofeedFeed* feed, MicrofeedItem* old_item, MicrofeedItem* new_item) {
	MicrofeedStoreIterator* iterator;
	ItemDataProperty* item_data_property;
	const char* old_value;
	const char* new_value;	

	for (iterator = microfeed_store_iterate(feed->item_data_properties, NULL);
	     (item_data_property = microfeed_store_iterator_get(iterator, ItemDataProperty));
	     microfeed_store_iterator_next(iterator)) {
		if (old_item && (old_value = microfeed_item_get_property(old_item, item_data_property->property_name))) {
			if (new_item && (new_value = microfeed_item_get_property(new_item, item_data_property->property_name))) {
				if (strcmp(old_value, new_value)) {
					microfeed_feed_unref_item_data(item_data_property->item_data_feed, old_value);
					microfeed_feed_ref_item_data(item_data_property->item_data_feed, new_value); 
				}
			} else {
				microfeed_feed_unref_item_data(item_data_property->item_data_feed, old_value);
			}
		} else if (new_item && (new_value = microfeed_item_get_property(new_item, item_data_property->property_name))) {
			microfeed_feed_ref_item_data(item_data_property->item_data_feed, new_value);
		}
	}
	microfeed_store_iterator_free(iterator);
}

static void index_function(void* key, const size_t key_size, void* data, const size_t data_size, void** index_key, size_t* index_key_size) {
	*index_key = key + sizeof(uint64_t);
	*index_key_size = key_size - sizeof(uint64_t);
}

static int compare_function(const void* key1, const size_t key1_size, const void* key2, const size_t key2_size) {
	int retvalue = 0;
	
	if (key1_size > sizeof(uint64_t) && key2_size > sizeof(int64_t)) {
		if (!(retvalue = *((uint64_t*)key1) - *((uint64_t*)key2))) {
			retvalue = strcmp(key1 + sizeof(uint64_t), key2 + sizeof(uint64_t));
		}
	}

	return retvalue;
}

static void remove_old_items(MicrofeedFeed* feed) {
	time_t timestamp;
	int keep_old_items;
	
/*	keep_old_items = atoi(microfeed_publisher_get_setting_value(feed->publisher, "keep_old_items", "0"));
	timestamp = time(NULL) + 60 * 60 * keep_old_items;
*/	
	/* TODO! */
	
}

static const char* get_item_data_mutex_get_uid(GetItemDataMutex* get_item_data_mutex) {
	
	return get_item_data_mutex->uid;
}
