/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation, or under the terms of the GNU Lesser General
 *  Public License version 2.1 as published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MICROFEEDMAIN_H
#define MICROFEEDMAIN_H

#define DBUS_API_SUBJECT_TO_CHANGE
#include <dbus/dbus.h>

/**
 * @addtogroup common libmicrofeed-common - Common modules used both in the publisher and in the subscriber side
 * @{
 * @addtogroup MicrofeedMain
 *
 * An optional main loop.
 *
 * This module is usually used only in the provider side, when the subscriber side is using
 * some other main loop, such as GTK+/GLib.
 *
 * MicrofeedMain is thread-aware, because it implements internal lock. All callbacks are called
 * without locks being held.
 *
 * @{
 */

/**
 * Opaque data type representing a main loop.
 */
typedef struct _MicrofeedMain MicrofeedMain;

/**
 * Opaque data type representing a timeout in the main loop.
 */
typedef struct _MicrofeedTimeout MicrofeedTimeout;

/**
 * Opaque data type representing a file descriptor to watch in the main loop.
 */
typedef struct _MicrofeedWatch MicrofeedWatch;

/**
 * Possible watch types of a watched file descriptor.
 */
typedef enum {
	MICROFEED_WATCH_TYPE_NONE = 0,
	MICROFEED_WATCH_TYPE_READ = 1,
	MICROFEED_WATCH_TYPE_WRITE = 2,
	MICROFEED_WATCH_TYPE_READ_WRITE = 3
} MicrofeedWatchType;

/**
 * A function that is called when a timeout has occurred.
 * 
 * The timeout is removed automatically, so if you want to call the functin repeatedly, you have to
 * add a new timeout here.
 * 
 * @param microfeed_main The main loop where the timeout occurred.
 * @param user_data The user data given when the timeout was added.
 */
typedef void (*MicrofeedTimeoutCallback)(MicrofeedMain* microfeed_main, void* user_data);

/**
 * A  function that is called when a watched file descriptor has something to read or/and write.
 *
 * @param microfeed_main The main loop where the timeout occurred.
 * @param fd The file descriptor that is ready.
 * @param type Specifies if the file descriptor is ready to be read, written or both.
 * @param user_data The user data given when the watch was added.
 */
typedef void (*MicrofeedWatchCallback)(MicrofeedMain* microfeed_main, int fd, MicrofeedWatchType type, void* user_data);

MicrofeedMain* microfeed_main_new();
MicrofeedMain* microfeed_main_new_with_dbus_connection(DBusConnection* connection);
void microfeed_main_free(MicrofeedMain* microfeed_main);

DBusConnection* microfeed_main_get_dbus_connection(MicrofeedMain* microfeed_main);
void microfeed_main_loop(MicrofeedMain* microfeed_main);
void microfeed_main_exit(MicrofeedMain* microfeed_main);
MicrofeedTimeout* microfeed_main_add_timeout(MicrofeedMain* microfeed_main, unsigned long int milliseconds, MicrofeedTimeoutCallback callback, void* user_data);
MicrofeedWatch* microfeed_main_add_watch(MicrofeedMain* microfeed_main, int fd, MicrofeedWatchType type, MicrofeedWatchCallback callback, void* user_data);
void microfeed_main_remove_timeout(MicrofeedMain* microfeed_main, MicrofeedTimeout* timeout);
void microfeed_main_remove_watch(MicrofeedMain* microfeed_main, MicrofeedWatch* watch);

/**
 * @}
 * @}
 */

#endif
