/*
 *  Microfeed - Backend for accessing feed-based services
 *  Copyright (C) 2009 Henrik Hedberg <henrik.hedberg@innologies.fi>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as published by
 *  the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#ifndef MICROFEEDPUBLISHER_H
#define MICROFEEDPUBLISHER_H

#include <time.h>
#include <dbus/dbus.h>

/**
 * @addtogroup provider libmicrofeed-provider - Modules that are used in the provider side
 * @{
 * @addtogroup MicrofeedPublisher
 *
 * A publisher's connection to the DBus system.
 *
 * Publisher can be used as a stand-alone module, but it is suggested that an application uses
 * #MicrofeedProvider instead and leaves the instantiation of specific publishers to the provider.
 * @{
 */

/**
 * Opaque data type representing a publisher.
 */
typedef struct _MicrofeedPublisher MicrofeedPublisher;


/** @} */

#include <microfeed-common/microfeedstore.h>
#include <microfeed-provider/microfeedfeed.h>
#include <microfeed-provider/microfeeddatabase.h>
#include <microfeed-common/microfeeditem.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-provider/microfeederror.h>

/**
 * @addtogroup MicrofeedPublisher
 * @{
 */

/**
 * A function that is called when a subscriber subscribes a feed that is not previously
 * subscribed by any subscriber.
 * 
 * The function call is a consequence of the SubscribeFeed DBus Message.
 * 
 * Note that this function just instantiates a #MicrofeedFeed instance with suitable callbacks
 * rather than creates a whole new feed into a service.
 */
typedef MicrofeedFeed* (*MicrofeedPublisherInstantiateFeedCallback)(MicrofeedPublisher* publisher, const char* uri, void* user_data);

typedef void (*MicrofeedPublisherNoMoreSubscribersCallback)(MicrofeedPublisher* publisher, void* user_data);

/**
 * A function that is called after a new publisher is created.
 * 
 * The function call is a consequence of the CreatePublisher DBus Message.
 * 
 * This function should initialize settings by calling #microfeed_publisher_add_setting function.
 */
typedef void (*MicrofeedPublisherInitializeSettingsCallback)(MicrofeedPublisher* publisher, void* user_data);

/**
 * A function that is called when a subscriber tries to update a setting.
 * 
 * This function call is a consequence of the ModifyItem DBus Message targeted at org.microfeed.Publisher.Settings feed.
 * 
 * The setting value is actually changed only if this function returns true. In other case, the error message is
 * automatically generated.
 * 
 * @return Boolean value telling if the updating of the setting value succeeded.
 */
typedef int (*MicrofeedPublisherUpdateSettingCallback)(MicrofeedPublisher* publisher, const char* key, const char* value, void* user_data);

typedef MicrofeedError* (*MicrofeedPublisherDownloadImageCallback)(MicrofeedPublisher* publisher, const char* uid, void** data, size_t* length, void* user_data);

typedef void (*MicrofeedPublisherTimeoutHandler)(void* data);

typedef void* (*MicrofeedPublisherAddTimeoutCallback)(MicrofeedPublisher* publisher, unsigned long int interval, MicrofeedPublisherTimeoutHandler handler, void* data, void* user_data);

typedef void (*MicrofeedPublisherRemoveTimeoutCallback)(MicrofeedPublisher* publisher, void* timeout_implementation, void* user_data);

/**
 * Functions that are called when a publisher receives a message from a subscriber.
 */
typedef struct _MicrofeedPublisherCallbacks {
	/**
	 * A function that is called when a subscriber subscribes a feed that is not previously
	 * subscribed by any subscriber.
	 */
	MicrofeedPublisherInstantiateFeedCallback instantiate_feed;
	/**
	 * A function that is called after a new publisher is created.
	 */
	MicrofeedPublisherInitializeSettingsCallback initialize_settings;
	/**
	 * A function that is called when a subscriber tries to update a setting.
	 */
	MicrofeedPublisherUpdateSettingCallback update_setting;
	
	MicrofeedPublisherDownloadImageCallback download_image;
	
	MicrofeedPublisherNoMoreSubscribersCallback no_more_subscribers;
	
	MicrofeedPublisherAddTimeoutCallback add_timeout;
	MicrofeedPublisherRemoveTimeoutCallback remove_timeout;
} MicrofeedPublisherCallbacks;

MicrofeedPublisher* microfeed_publisher_new(const char* unique_identifier, const char* directory, DBusConnection* connection, MicrofeedPublisherCallbacks* callbacks, void* user_data);
void microfeed_publisher_free(MicrofeedPublisher* publisher);

void microfeed_publisher_add_setting(MicrofeedPublisher* publisher, const char* uid, const char* text, const char* unit, const char* type, const char* length, const char* value);
void microfeed_publisher_remove_setting(MicrofeedPublisher* publisher, const char* uid);
char* microfeed_publisher_get_setting_value(MicrofeedPublisher* publisher, const char* uid, const char* default_value);
long int microfeed_publisher_get_setting_value_integer(MicrofeedPublisher* publisher, const char* uid, long int default_value);
int microfeed_publisher_set_setting_value(MicrofeedPublisher* publisher, const char* uid, const char* value);
MicrofeedThread* microfeed_publisher_queue_thread(MicrofeedPublisher* publisher, MicrofeedThreadFunction function, void* data);
MicrofeedThread* microfeed_publisher_start_singleton_update_thread(MicrofeedPublisher* publisher, MicrofeedThreadFunction function, void* user_data);
MicrofeedThread* microfeed_publisher_get_singleton_update_thread(MicrofeedPublisher* publisher);

/**
 * @}
 * @}
 */

/* For internal use only. */
MicrofeedDatabaseEnvironment* microfeed_publisher_get_database_environment(MicrofeedPublisher* publisher);

/* For internal use only. */
void microfeed_publisher_handle_item_property_change(MicrofeedPublisher* publisher, MicrofeedItem* old_item, MicrofeedItem* new_item);

/* For internal use only. */
void microfeed_publisher_send_item_signal(MicrofeedPublisher* publisher, const char* destination, const char* signal_name, const char* uri, MicrofeedItem* item);

/* For internal use only. */
void microfeed_publisher_send_item_uid_signal(MicrofeedPublisher* publisher, const char* destination, const char* signal_name, const char* uri, const char* uid);

/* For internal use only. */
void microfeed_publisher_send_status_changed_signal(MicrofeedPublisher* publisher, const char* destination, const char* uri, const char* uid, const char status);

/* For internal use only. */
void microfeed_publisher_send_feed_signal(MicrofeedPublisher* publisher, const char* destination, const char* signal_name, const char* uri);

/* For internal use only. */
void microfeed_publisher_send_error_signal(MicrofeedPublisher* publisher, const char* destination, const char* error_name, const char* uri, const char* uid, const char* error_message);

/* For internal use only. */
void microfeed_publisher_send_item_data_signal(MicrofeedPublisher* publisher, const char* destination, const char* uri, const char* uid, const void* data, size_t data_size);

/* For internal use only. */
const char* microfeed_publisher_get_object_path(MicrofeedPublisher* publisher);

/* For internal use only. */
const char* microfeed_publisher_get_identifier(MicrofeedPublisher* publisher);

/* For internal use only. */
void microfeed_publisher_remove_subscriber(MicrofeedPublisher* publisher, const char* unique_connection_name);

#include <microfeed-provider/microfeedprovider.h>

/* For internal use only. */
MicrofeedProvider* microfeed_publisher_get_provider(MicrofeedPublisher* publisher);

/* For internal use only. */
void microfeed_publisher_set_provider(MicrofeedPublisher* publisher, MicrofeedProvider* provider);

#endif
