
#ifndef MICROFEED_STORE_H
#define MICROFEED_STORE_H

/**
 * @addtogroup MicrofeedStore
 *
 * This module implementing a dynamic array is used internally in the library,
 * but can be also utilized outside of the library implementation.
 *
 * A store can be either sorted or unsorted.
 * 
 * @{
 */

/**
 * Opaque data type representing sorted array that grows automatically when items are added.
 */
typedef struct _MicrofeedStore MicrofeedStore;

/**
 * Opaque data type representing sorted array iterator that traverses over the elements of the array.
 */
typedef struct _MicrofeedStoreIterator MicrofeedStoreIterator;

/**
 * 
 */

typedef int (*MicrofeedStoreCompareKeysFunction)(const void* key1, const void* key2);
typedef const void* (*MicrofeedStoreGetKeyFunction)(const void* item);
typedef void (*MicrofeedStoreForeachFunction)(void* data, void* user_data);
typedef int (*MicrofeedStoreCompareDatasFunction)(const void* data1, const void* data2, unsigned int index1, unsigned int index2, void* user_data);
typedef void (*MicrofeedStoreFreeDataFunction)(void* data);

MicrofeedStore* microfeed_store_new_sorted(MicrofeedStoreCompareKeysFunction compare_keys, MicrofeedStoreGetKeyFunction get_key);
MicrofeedStore* microfeed_store_new_unsorted(MicrofeedStoreCompareKeysFunction compare_keys, MicrofeedStoreGetKeyFunction get_key);
void microfeed_store_free(MicrofeedStore* store);

int microfeed_store_is_sorted(MicrofeedStore* store);
void microfeed_store_foreach(MicrofeedStore* store, MicrofeedStoreForeachFunction foreach, void* user_data);
#define microfeed_store_get(s, k, t) ((t*)microfeed_store_get_impl(s, k))
void* microfeed_store_get_impl(MicrofeedStore* store, const void* key);
#define microfeed_store_get_index(s, i, t) ((t*)microfeed_store_get_index_impl(s, i))
void* microfeed_store_get_index_impl(MicrofeedStore* store, unsigned int index);
unsigned int microfeed_store_get_size(MicrofeedStore* store);
int microfeed_store_insert(MicrofeedStore* store, void* data);
#define microfeed_store_replace(s, d, t) ((t*)microfeed_store_replace_impl(s, d))
void* microfeed_store_replace_impl(MicrofeedStore* store, void* data);
int microfeed_store_remove(MicrofeedStore* store, const void* data);
#define microfeed_store_remove_key(s, k, t) ((t*)microfeed_store_remove_key_impl(s, k))
void* microfeed_store_remove_key_impl(MicrofeedStore* store, const void* key);
#define microfeed_store_remove_index(s, i, t) ((t*)microfeed_store_remove_index_impl(s, i))
void* microfeed_store_remove_index_impl(MicrofeedStore* store, unsigned int index);
void microfeed_store_remove_and_free_all(MicrofeedStore* store, MicrofeedStoreFreeDataFunction free_data);
MicrofeedStoreIterator* microfeed_store_iterate(MicrofeedStore* store, const void* start_key);
void microfeed_store_sort(MicrofeedStore* store, MicrofeedStoreCompareDatasFunction compare_datas, void* user_data);

void microfeed_store_iterator_free(MicrofeedStoreIterator* iterator);
void* microfeed_store_iterator_get_impl(MicrofeedStoreIterator* iterator);
#define microfeed_store_iterator_get(i, t) ((t*)microfeed_store_iterator_get_impl(i))
void microfeed_store_iterator_next(MicrofeedStoreIterator* iterator);

int microfeed_store_compare_keys_direct(const void* key1, const void* key2);
const void* microfeed_store_get_key_direct(const void* data);

/** @} */

#endif
