
#include <microfeed-provider/microfeedprovider.h>
#include <microfeed-provider/microfeedpublisher.h>
#include <microfeed-common/microfeedmisc.h>
#include <microfeed-common/microfeedthread.h>
#include <microfeed-provider/microfeederror.h>
#include <microfeed-common/microfeedprotocol.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

typedef struct _Thread Thread;

struct _MicrofeedPublisher {
	char* identifier;
	char* object_path;
	DBusConnection* connection;
	MicrofeedPublisherCallbacks callbacks;
	void* user_data;
	MicrofeedDatabaseEnvironment* database_environment;
	char** image_properties;
	MicrofeedProvider* provider;
	DBusMessage* current_message;
	
	MicrofeedStore* subscribers;
	MicrofeedStore* feeds;
	MicrofeedFeed* settings_feed;
	MicrofeedFeed* images_feed;
	MicrofeedThread* singleton_update_thread;
	MicrofeedThreadPool* thread_pool;
	MicrofeedStore* timeouts;
};

typedef struct {
	MicrofeedPublisher* publisher;
	char* unique_connection_name;
	time_t last_activity;
	MicrofeedStore* feeds;
} Subscriber;

typedef struct {
	MicrofeedFeed* feed;
	void* implementation;
} Timeout;

typedef struct {
	const char* name;
	MicrofeedError* (*callback)(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
} MethodCallback;

static void object_unregister(DBusConnection* connection, void* user_data);
static DBusHandlerResult object_message(DBusConnection* connection, DBusMessage* message, void* user_data);
static Subscriber* subscriber_new(MicrofeedPublisher* publisher, const char* unique_connection_name);
static void subscriber_free(Subscriber* subscriber);
static const char* subscriber_get_unique_connection_name(Subscriber* subscriber);
static Timeout* timeout_new(MicrofeedPublisher* publisher, MicrofeedFeed* feed);
static void timeout_free(Timeout* timeout);
static MicrofeedFeed* timeout_get_feed(Timeout* timeout);

static MicrofeedError* method_add_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_modify_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_remove_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_read_items(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_republish_items(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_send_item_data(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_mark_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_unmark_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_ping(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_subscribe_feed(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_unsubscribe_feed(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_update_feed(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_create_publisher(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);
static MicrofeedError* method_destroy_publisher(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message);

static void reply_to_current_message(MicrofeedPublisher* publisher, MicrofeedError* error);
static MicrofeedError* settings_feed_modify_item(MicrofeedFeed* feed, MicrofeedItem* existing_item, MicrofeedItem* new_item, void* user_data);
static MicrofeedError* images_feed_download_item_data(MicrofeedFeed* feed, const char* uid, void** data, size_t* length, void* user_data);
static void thread_exited(MicrofeedThread* thread, void* user_data);
static void timeout_handler(void* data);

static MethodCallback method_callbacks[] = {
	{ MICROFEED_METHOD_NAME_ADD_ITEM, method_add_item },
	{ MICROFEED_METHOD_NAME_MODIFY_ITEM, method_modify_item },
	{ MICROFEED_METHOD_NAME_REMOVE_ITEM, method_remove_item },
	{ MICROFEED_METHOD_NAME_READ_ITEMS, method_read_items },
	{ MICROFEED_METHOD_NAME_REPUBLISH_ITEMS, method_republish_items },
	{ MICROFEED_METHOD_NAME_SEND_ITEM_DATA, method_send_item_data },
	{ MICROFEED_METHOD_NAME_MARK_ITEM, method_mark_item },
	{ MICROFEED_METHOD_NAME_UNMARK_ITEM, method_unmark_item },
	{ MICROFEED_METHOD_NAME_PING, method_ping },
	{ MICROFEED_METHOD_NAME_SUBSCRIBE_FEED, method_subscribe_feed },
	{ MICROFEED_METHOD_NAME_UNSUBSCRIBE_FEED, method_unsubscribe_feed },
	{ MICROFEED_METHOD_NAME_UPDATE_FEED, method_update_feed },
	{ MICROFEED_METHOD_NAME_CREATE_PUBLISHER, method_create_publisher },
	{ MICROFEED_METHOD_NAME_DESTROY_PUBLISHER, method_destroy_publisher },
	{ NULL, NULL }
};

static MicrofeedFeedCallbacks settings_feed_callbacks = {
	NULL,
	NULL,
	settings_feed_modify_item
};

static MicrofeedFeedCallbacks images_feed_callbacks = {
	NULL,
	NULL,
	NULL,
	images_feed_download_item_data
};

static DBusObjectPathVTable object_vtable = {
	object_unregister,
	object_message
};	

/**
 * Instantiates a new publisher.
 * 
 * The publisher registers a DBus object path /org/microfeed/publisher/<publisher_identifier> and creates or opens
 * a database environment in the given directory using publisher_identifier as a name.
 * 
 * @param publisher_identifier A string containing unique publisher identifier.
 * @param directory A directory where the database environment is created or opened (if exists).
 * @param connection A DBus connection.
 * @param callbacks Functions that are called when a publisher receives a message from a subscriber.
 * @param user_data A pointer to user data that is given as a parameter when a callback function is called.
 * @return Instantiated MicrofeedPublisher.
 */
MicrofeedPublisher* microfeed_publisher_new(const char* publisher_identifier, const char* directory, DBusConnection* connection, MicrofeedPublisherCallbacks* callbacks, void* user_data) {
	MicrofeedPublisher* publisher;
	char* colon;
	
	publisher = microfeed_memory_allocate(MicrofeedPublisher);
	publisher->identifier = strdup(publisher_identifier);
	if (!(colon = strchr(publisher->identifier, ':')) || colon[1] == 0) {
		free(publisher->identifier);
		microfeed_memory_free(publisher);
		publisher = NULL;
	} else {
		*colon = 0;
		publisher->object_path = microfeed_util_string_concatenate(MICROFEED_DBUS_OBJECT_PATH_PREFIX_PUBLISHER, publisher->identifier, NULL);
		*colon = ':';
		publisher->connection = connection;
		publisher->callbacks = *callbacks;
		publisher->user_data = user_data;
		publisher->image_properties = (char**)malloc(3 * sizeof(char*));
		publisher->image_properties[0] = strdup(MICROFEED_ITEM_PROPERTY_NAME_CONTENT_IMAGE);
		publisher->image_properties[1] = strdup(MICROFEED_ITEM_PROPERTY_NAME_USER_IMAGE);
		publisher->image_properties[2] = 0;

		if (!dbus_connection_register_object_path(connection, publisher->object_path, &object_vtable, publisher)) {
		
			return NULL;
		}
		
		if (!(publisher->database_environment = microfeed_database_environment_new(publisher_identifier, directory))) {
			
			return NULL;
		}
		publisher->subscribers = microfeed_store_new_sorted((MicrofeedStoreCompareKeysFunction)strcmp,
		                                             (MicrofeedStoreGetKeyFunction)subscriber_get_unique_connection_name);
		publisher->feeds = microfeed_store_new_sorted((MicrofeedStoreCompareKeysFunction)strcmp,
		                                       (MicrofeedStoreGetKeyFunction)microfeed_feed_get_uri);

		publisher->settings_feed = microfeed_feed_new(publisher, MICROFEED_FEED_URI_SETTINGS, "Settings", MICROFEED_FEED_PERMISSION_NONE, &settings_feed_callbacks, NULL);
		microfeed_store_insert(publisher->feeds, publisher->settings_feed);
		microfeed_feed_add_subscriber(publisher->settings_feed, NULL);		

		publisher->images_feed = microfeed_feed_new(publisher, MICROFEED_FEED_URI_IMAGES, "Images", MICROFEED_FEED_PERMISSION_NONE, &images_feed_callbacks, NULL);
		microfeed_store_insert(publisher->feeds, publisher->images_feed);
		microfeed_feed_add_subscriber(publisher->images_feed, NULL);		

		publisher->thread_pool = microfeed_thread_pool_new_with_exit_callback(microfeed_publisher_get_setting_value_integer(publisher, "threads.max", 10), thread_exited, publisher);
		publisher->timeouts = microfeed_store_new_sorted(microfeed_store_compare_keys_direct, (MicrofeedStoreGetKeyFunction)timeout_get_feed);
	}
	
	return publisher;
}

/**
 * Frees the resources allocated for the publisher.
 * 
 * TODO: This function is not yet implemented.
 *
 * @param publisher Instantiated MicrofeedPublisher.
 */
void microfeed_publisher_free(MicrofeedPublisher* publisher) {
	
}

/**
 * Adds a new setting or changes other properties than the value of an existing setting in
 * the org.microfeed.Publisher.Settings feed.
 * 
 * @param publisher Instantiated MicrofeedPublisher.
 */
void microfeed_publisher_add_setting(MicrofeedPublisher* publisher, const char* uid, const char* text, const char* unit, const char* type, const char* length, const char* value) {
	MicrofeedItem* item;
	MicrofeedItem* existing_item;
	
	item = microfeed_item_new(uid, time(NULL));
	microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_LENGTH, length);
	microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_TEXT, text);
	microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_TYPE, type);
	microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_UNIT, unit);	
	if ((existing_item = microfeed_feed_get_item(publisher->settings_feed, uid))) {
		microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE, microfeed_item_get_property(existing_item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE));
	} else {
		microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE, value);
	}	
	microfeed_feed_replace_item(publisher->settings_feed, item);
}

/**
 * Removes a setting from the org.microfeed.Publisher.Settings feed.
 * 
 * @param publisher Instantiated MicrofeedPublisher.
 * @param uid The unique identifier of the setting item.
 */
void microfeed_publisher_remove_setting(MicrofeedPublisher* publisher, const char* uid) {
	microfeed_feed_remove_item(publisher->settings_feed, uid);
}

/**
 * Returns a copy of a value of a setting, or if the setting does not exist returns a copy of the given default value.
 * 
 * @param publisher Instantiated MicrofeedPublisher.
 * @param key The unique identifier of the setting item.
 * @param default_value The value that is returned if the setting does not exist.
 * @return The value of the setting (must be freed).
 */
char* microfeed_publisher_get_setting_value(MicrofeedPublisher* publisher, const char* uid, const char* default_value) {
	const char* const_value;
	char* value;
	MicrofeedItem* item;
	
	if ((item = microfeed_feed_get_item(publisher->settings_feed, uid))) {
		if ((const_value = microfeed_item_get_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE))) {
			value = strdup(const_value);
		} else {
			value = (default_value ? strdup(default_value) : NULL);
		}
		microfeed_item_free(item);
	} else {
		value = (default_value ? strdup(default_value) : NULL);
	}
	
	return value;
}

/**
 * Returns a value of a setting in integer, or if the setting does not exist returns the given default value.
 * 
 * @param publisher Instantiated MicrofeedPublisher.
 * @param key The unique identifier of the setting item.
 * @param default_value The value that is returned if the setting does not exist.
 * @return The value of the setting in integer.
 */
long int microfeed_publisher_get_setting_value_integer(MicrofeedPublisher* publisher, const char* uid, long int default_value) {
	long int value;
	MicrofeedItem* item;
	const char* string;
	char* end;
	
	if ((item = microfeed_feed_get_item(publisher->settings_feed, uid))) {
		if ((string = microfeed_item_get_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE)) && *string != 0) {
			value = strtol(string, &end, 10);
			if (*end != 0) {
				value = default_value;
			}
		} else {
			value = default_value;
		}
		microfeed_item_free(item);
	} else {
		value = default_value;
	}
	
	return value;
	
}

/**
 * Sets the value of a setting if the setting exists.
 * 
 * @param publisher Instantiated MicrofeedPublisher.
 * @param uid The unique identified of the setting item.
 * @param value The new value for the setting.
 * @return A boolean value: true, if the operation succeeded, or false if the setting does not exist.
 */
int microfeed_publisher_set_setting_value(MicrofeedPublisher* publisher, const char* uid, const char* value) {
	int retvalue = 0;
	MicrofeedItem* item;
	
	if ((item = microfeed_feed_get_item(publisher->settings_feed, uid))) {
		microfeed_item_set_property(item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE, value);
		microfeed_feed_replace_item(publisher->settings_feed, item);
		retvalue = 1;
		microfeed_item_free(item);
	}	

	return retvalue;
}

MicrofeedThread* microfeed_publisher_queue_thread(MicrofeedPublisher* publisher, MicrofeedThreadFunction function, void* data) {
	
	return microfeed_thread_pool_queue_thread(publisher->thread_pool, function, data);
}

MicrofeedThread* microfeed_publisher_start_singleton_update_thread(MicrofeedPublisher* publisher, MicrofeedThreadFunction function, void* user_data) {
	if (!publisher->singleton_update_thread) {
		publisher->singleton_update_thread = microfeed_thread_new(function, user_data);		
	}
		
	return publisher->singleton_update_thread;
}

MicrofeedThread* microfeed_publisher_get_singleton_update_thread(MicrofeedPublisher* publisher) {
	
	return publisher->singleton_update_thread;
}

MicrofeedDatabaseEnvironment* microfeed_publisher_get_database_environment(MicrofeedPublisher* publisher) {

	return publisher->database_environment;
}

void microfeed_publisher_handle_item_property_change(MicrofeedPublisher* publisher, MicrofeedItem* old_item, MicrofeedItem* new_item) {
	char** key;
	const char* old_value;
	const char* new_value;	

	for (key = publisher->image_properties; *key; key++) {
		if (old_item && (old_value = microfeed_item_get_property(old_item, *key))) {
			if (new_item && (new_value = microfeed_item_get_property(new_item, *key))) {
				if (strcmp(old_value, new_value)) {
					microfeed_feed_unref_item_data(publisher->images_feed, old_value);
					microfeed_feed_ref_item_data(publisher->images_feed, new_value); 
				}
			} else {
				microfeed_feed_unref_item_data(publisher->images_feed, old_value);
			}
		} else if (new_item && (new_value = microfeed_item_get_property(new_item, *key))) {
			microfeed_feed_ref_item_data(publisher->images_feed, new_value);
		}
	}
}

void microfeed_publisher_send_item_signal(MicrofeedPublisher* publisher, const char* destination, const char* signal_name, const char* uri, MicrofeedItem* item) {
	DBusMessage* message;
	DBusMessageIter iter;
	MicrofeedItemIterator* item_iterator;
	const char* uid;
	dbus_uint64_t timestamp;
	char status;
	const char* key;
	const char* value;
	
	message = dbus_message_new_signal(publisher->object_path, (destination ? MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION : MICROFEED_DBUS_INTERFACE_PUBLISHER), signal_name);
	if (destination) {
		dbus_message_set_destination(message,destination);
	}
	dbus_message_iter_init_append(message, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uri);
	uid = microfeed_item_get_uid(item);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uid);
	timestamp = (dbus_uint64_t)microfeed_item_get_timestamp(item);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_UINT64, &timestamp);
	status = (char)microfeed_item_get_status(item);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_BYTE, &status);
	for (item_iterator = microfeed_item_iterate_properties(item, NULL);
	     microfeed_item_iterator_get(item_iterator, &key, &value);
	     microfeed_item_iterator_next(item_iterator)) {
		dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &key);
		dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &value);
	}
	microfeed_item_iterator_free(item_iterator);
	dbus_connection_send(publisher->connection, message, NULL);
	dbus_message_unref(message);	
}

void microfeed_publisher_send_item_uid_signal(MicrofeedPublisher* publisher, const char* destination, const char* signal_name, const char* uri, const char* uid) {
	DBusMessage* message;

	message = dbus_message_new_signal(publisher->object_path, (destination ? MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION : MICROFEED_DBUS_INTERFACE_PUBLISHER), signal_name);
	if (destination) {
		dbus_message_set_destination(message, destination);
	}
	dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID);
	dbus_connection_send(publisher->connection, message, NULL);
	dbus_message_unref(message);
}

void microfeed_publisher_send_status_changed_signal(MicrofeedPublisher* publisher, const char* destination, const char* uri, const char* uid, const char status) {
	DBusMessage* message;

	message = dbus_message_new_signal(publisher->object_path, (destination ? MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION : MICROFEED_DBUS_INTERFACE_PUBLISHER), MICROFEED_SIGNAL_NAME_ITEM_STATUS_CHANGED);
	if (destination) {
		dbus_message_set_destination(message, destination);
	}
	dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_BYTE, &status, DBUS_TYPE_INVALID);
	dbus_connection_send(publisher->connection, message, NULL);
	dbus_message_unref(message);		
}

void microfeed_publisher_send_feed_signal(MicrofeedPublisher* publisher, const char* destination, const char* signal_name, const char* uri) {
	DBusMessage* message;

	message = dbus_message_new_signal(publisher->object_path, (destination ? MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION : MICROFEED_DBUS_INTERFACE_PUBLISHER), signal_name);
	if (destination) {
		dbus_message_set_destination(message, destination);
	}
	dbus_message_append_args(message, DBUS_TYPE_STRING, &uri, DBUS_TYPE_INVALID);
	dbus_connection_send(publisher->connection, message, NULL);
	dbus_message_unref(message);
}

void microfeed_publisher_send_error_signal(MicrofeedPublisher* publisher, const char* destination, const char* error_name, const char* uri, const char* uid, const char* error_message) {
	DBusMessage* message;
	DBusMessageIter iter;
	const char* string;

	message = dbus_message_new_signal(publisher->object_path, (destination ? MICROFEED_DBUS_INTERFACE_ERROR_TO_DESTINATION : MICROFEED_DBUS_INTERFACE_ERROR), error_name);
	if (destination) {
		dbus_message_set_destination(message, destination);
	}
	dbus_message_iter_init_append(message, &iter);
	string = (uri ? uri : "");
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &string);
	string = (uid ? uid : "");
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &string);
	string = (error_message ? error_message : "");
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &string);
	dbus_connection_send(publisher->connection, message, NULL);
	dbus_message_unref(message);	
}

void microfeed_publisher_send_item_data_signal(MicrofeedPublisher* publisher, const char* destination, const char* uri, const char* uid, const void* data, size_t data_size) {
	DBusMessage* message;
	DBusMessageIter iter;
	DBusMessageIter subiter;

	message = dbus_message_new_signal(publisher->object_path, (destination ? MICROFEED_DBUS_INTERFACE_PUBLISHER_TO_DESTINATION : MICROFEED_DBUS_INTERFACE_PUBLISHER), MICROFEED_SIGNAL_NAME_ITEM_DATA);
	if (destination) {
		dbus_message_set_destination(message,destination);
	}
	dbus_message_iter_init_append(message, &iter);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uri);
	dbus_message_iter_append_basic(&iter, DBUS_TYPE_STRING, &uid);
	dbus_message_iter_open_container(&iter, DBUS_TYPE_ARRAY, DBUS_TYPE_BYTE_AS_STRING, &subiter);
	dbus_message_iter_append_fixed_array(&subiter, DBUS_TYPE_BYTE, &data, data_size);
	dbus_message_iter_close_container(&iter, &subiter);
	dbus_connection_send(publisher->connection, message, NULL);
	dbus_message_unref(message);
}

const char* microfeed_publisher_get_object_path(MicrofeedPublisher* publisher) {
	return publisher->object_path;
}

const char* microfeed_publisher_get_identifier(MicrofeedPublisher* publisher) {
	return publisher->identifier;
}

void microfeed_publisher_remove_subscriber(MicrofeedPublisher* publisher, const char* unique_connection_name) {
	Subscriber* subscriber;
	MicrofeedStoreIterator* iterator;
	MicrofeedFeed* feed;
	char buffer[1024];
	
	if ((subscriber = microfeed_store_get(publisher->subscribers, unique_connection_name, Subscriber))) {
		for (iterator = microfeed_store_iterate(subscriber->feeds, NULL);
		     (feed = (MicrofeedFeed*)microfeed_store_iterator_get(iterator, MicrofeedFeed));
		     microfeed_store_iterator_next(iterator)) {
			microfeed_feed_remove_subscriber(feed, unique_connection_name);
			microfeed_store_remove(subscriber->feeds, feed);
	
			if (microfeed_feed_get_subscriber_count(feed) == 0) {
				microfeed_store_remove(publisher->feeds, feed);
				microfeed_feed_unref(feed);
			}
		}

		snprintf(buffer, 1024, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus',method='NameOwnerChanged',arg0='%s'", unique_connection_name);
		dbus_bus_remove_match(subscriber->publisher->connection, buffer, NULL);
		
		microfeed_store_remove(subscriber->publisher->subscribers, subscriber);
		subscriber_free(subscriber);
		
		if (publisher->callbacks.no_more_subscribers &&
		    microfeed_store_get_size(publisher->subscribers) == 0 &&
		    microfeed_thread_pool_get_started_thread_count(publisher->thread_pool) == 0) {
			publisher->callbacks.no_more_subscribers(publisher, publisher->user_data);
		}
	}			
}

MicrofeedProvider* microfeed_publisher_get_provider(MicrofeedPublisher* publisher) {

	return publisher->provider;	
}

void microfeed_publisher_set_provider(MicrofeedPublisher* publisher, MicrofeedProvider* provider) {
	publisher->provider = provider;
}

static Subscriber* subscriber_new(MicrofeedPublisher* publisher, const char* unique_connection_name) {
	Subscriber* subscriber;

	subscriber = (Subscriber*)microfeed_memory_allocate(Subscriber);
	subscriber->publisher = publisher;
	subscriber->unique_connection_name = strdup(unique_connection_name);
	subscriber->last_activity = time(NULL);
	subscriber->feeds = microfeed_store_new_sorted((MicrofeedStoreCompareKeysFunction)strcmp,
	                                        	(MicrofeedStoreGetKeyFunction)microfeed_feed_get_uri);

	return subscriber;	
}

static void subscriber_free(Subscriber* subscriber) {
	free(subscriber->unique_connection_name);
	microfeed_store_free(subscriber->feeds);
	
	microfeed_memory_free(subscriber);
}

static const char* subscriber_get_unique_connection_name(Subscriber* subscriber) {

	return subscriber->unique_connection_name;
}

static Timeout* timeout_new(MicrofeedPublisher* publisher, MicrofeedFeed* feed) {
	Timeout* timeout = NULL;
	unsigned long int interval;
	
	if ((interval = (unsigned long int)microfeed_publisher_get_setting_value_integer(publisher, "update_interval", 0))) {
		timeout = microfeed_memory_allocate(Timeout);
		timeout->feed = feed;
		timeout->implementation = publisher->callbacks.add_timeout(publisher, interval * 60000, timeout_handler, timeout, publisher->user_data);
	}
	
	return timeout;
}

static void timeout_free(Timeout* timeout) {
	MicrofeedPublisher* publisher;
	
	publisher = microfeed_feed_get_publisher(timeout->feed);
	publisher->callbacks.remove_timeout(publisher, timeout->implementation, publisher->user_data);	
}

static MicrofeedFeed* timeout_get_feed(Timeout* timeout) {
	return timeout->feed;	
}

static MicrofeedError* method_add_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	DBusMessageIter iter;
	const char* uri;
	const char* uid;
	MicrofeedFeed* feed;
	MicrofeedItem* item;
	const char* key;
	const char* value;
	
	if (!dbus_message_iter_init(message, &iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri argument in AddItem method.");
	}
	dbus_message_iter_get_basic(&iter, &uri);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected temporary item uid argument in AddItem method.");
	}
	dbus_message_iter_get_basic(&iter, &uid);
	if (!(feed = microfeed_store_get(subscriber->feeds, uri, MicrofeedFeed))) {
		
		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to add an item into a feed that does not exist.");
	}
	/* TODO: Would be faster if microfeed_feed_check_item existed. */
	if ((item = microfeed_feed_get_item(feed, uid))) {
		microfeed_item_free(item);
		
		return microfeed_error_new(MICROFEED_ERROR_ITEM_ALREADY_EXISTS, "Trying to add an item that already existed in a feed.");
	}
	item = microfeed_item_new(uid, time(NULL));
	while (dbus_message_iter_next(&iter)) {
		if (dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
			microfeed_item_free(item);

			return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected string key in AddItem method.");
		}
		dbus_message_iter_get_basic(&iter, &key);

		if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
			microfeed_item_free(item);

			return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected string value in AddItem method.");
		}
		dbus_message_iter_get_basic(&iter, &value);
		
		microfeed_item_set_property(item, key, value);
	}
	
	reply_to_current_message(publisher, NULL);
	
	microfeed_feed_call_modify_item_callback(feed, NULL, item);
	microfeed_item_free(item);
	
	return NULL;
}

static MicrofeedError* method_modify_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	DBusMessageIter iter;
	const char* uri;
	const char* uid;
	MicrofeedFeed* feed;
	MicrofeedItem* existing_item;
	MicrofeedItem* new_item;
	const char* key;
	const char* value;
	
	if (!dbus_message_iter_init(message, &iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri argument in ModifyItem method.");
	}
	dbus_message_iter_get_basic(&iter, &uri);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected item uid argument in ModifyItem method.");
	}
	dbus_message_iter_get_basic(&iter, &uid);
	if (!(feed = microfeed_store_get(subscriber->feeds, uri, MicrofeedFeed))) {
		
		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to modify an item of a feed that does not exist.");
	}

	if (!(existing_item = microfeed_feed_get_item(feed, uid))) {

		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_ITEM, "Trying to modify an item that does not exist.");
	}

	new_item = microfeed_item_new(uid, 0); /* TODO: New timestamp from message? */
	while (dbus_message_iter_next(&iter)) {
		if (dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
			microfeed_item_free(new_item);

			return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected string key in ModifyItem method.");
		}
		dbus_message_iter_get_basic(&iter, &key);

		if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
			microfeed_item_free(new_item);

			return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected string value in ModifyItem method.");
		}
		dbus_message_iter_get_basic(&iter, &value);
		
		microfeed_item_set_property(new_item, key, value);
	}
	
	reply_to_current_message(publisher, NULL);
	
	microfeed_feed_call_modify_item_callback(feed, existing_item, new_item);
	microfeed_item_free(new_item);
	
	return NULL;
}

static MicrofeedError* method_remove_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedError* error = NULL;
	DBusError dbus_error;
	const char* uri;
	const char* uid;
	MicrofeedFeed* feed;
	MicrofeedItem* item;

	dbus_error_init(&dbus_error);
	if (!dbus_message_get_args(message, &dbus_error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID)) {
		error = microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri and item uid arguments in RemoveItem method");
	} else if (!(feed = microfeed_store_get(subscriber->feeds, uri, MicrofeedFeed))) {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to remove an item of a feed that does not exist.");
	} else if (!(item = microfeed_feed_get_item(feed, uid))) {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_ITEM, "Trying to remove an item that does not exist.");
	} else {
		reply_to_current_message(publisher, NULL);
	
		microfeed_feed_call_modify_item_callback(feed, item, NULL);
		microfeed_item_free(item);
	}
	
	return error;
}

static MicrofeedError* method_read_items(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	DBusMessageIter iter;
	const char* uri;
	const char* start_uid;
	const char* end_uid;
	MicrofeedFeed* feed;
	
	if (!dbus_message_iter_init(message, &iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri argument in ReadItem method.");
	}
	dbus_message_iter_get_basic(&iter, &uri);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected start item uid argument in ReadItem method.");
	}
	dbus_message_iter_get_basic(&iter, &start_uid);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected end item uid argument in ReadItem method.");
	}
	dbus_message_iter_get_basic(&iter, &end_uid);
	if (!(feed = microfeed_store_get(subscriber->feeds, uri, MicrofeedFeed))) {
		
		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to read items of a feed that does not exist.");
	}
	
	reply_to_current_message(publisher, NULL);

	microfeed_feed_unset_item_statuses(feed, start_uid, end_uid, MICROFEED_ITEM_STATUS_UNREAD);

	return NULL;
}

static MicrofeedError* method_republish_items(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	DBusMessageIter iter;
	const char* uri;
	const char* start_uid;
	const char* end_uid;
	dbus_uint16_t max_count;
	MicrofeedFeed* feed;

	if (!dbus_message_iter_init(message, &iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected a feed uri argument in PublishItems method");
	}
	dbus_message_iter_get_basic(&iter, &uri);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected a start item uid argument in PublishItems method");
	}
	dbus_message_iter_get_basic(&iter, &start_uid);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected an end item uid argument in PublishItems method");
	}
	dbus_message_iter_get_basic(&iter, &end_uid);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_UINT16) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected an maximum count argument in PublishItems method");
	}
	dbus_message_iter_get_basic(&iter, &max_count);

	if (!(feed = microfeed_store_get(subscriber->feeds, uri, MicrofeedFeed))) {

		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to republish items of a feed that does not exist.");
	}
	
	reply_to_current_message(publisher, NULL);
	
	microfeed_feed_republish(feed, (*start_uid ? start_uid : NULL), (*end_uid ? end_uid : NULL), max_count, subscriber->unique_connection_name);
	     
	return NULL;
}

static MicrofeedError* method_send_item_data(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	DBusMessageIter iter;
	const char* uri;
	const char* uid;
	MicrofeedFeed* feed;

	if (!publisher->callbacks.download_image) {

		return microfeed_error_new(MICROFEED_ERROR_NO_ITEM_DATA, "Publisher does not support image downloading");
	}
	if (!dbus_message_iter_init(message, &iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected a feed uri argument in SendItemData method");
	}
	dbus_message_iter_get_basic(&iter, &uri);
	if (!dbus_message_iter_next(&iter) || dbus_message_iter_get_arg_type(&iter) != DBUS_TYPE_STRING) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected an item uid argument in SendItemData method");
	}
	dbus_message_iter_get_basic(&iter, &uid);
	
	if (!(feed = microfeed_store_get(subscriber->feeds, uri, MicrofeedFeed))) {

		return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to republish items of a feed that does not exist.");
	}
	
	reply_to_current_message(publisher, NULL);
	
	microfeed_feed_send_item_data(feed, uid, subscriber->unique_connection_name);
		
	return NULL;
}

static MicrofeedError* method_mark_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedError* error = NULL;
	DBusError dbus_error;
	const char* uri;
	const char* uid;
	MicrofeedFeed* feed;
	
	dbus_error_init(&dbus_error);
	if (!dbus_message_get_args(message, &dbus_error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID)) {
		error = microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri and item uid arguments in MarkItem method");
	} else if (!(feed = microfeed_store_get(subscriber->feeds, uri, MicrofeedFeed))) {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to mark an item of a feed that does not exist.");
	} else {
		reply_to_current_message(publisher, NULL);
		
		microfeed_feed_set_item_status(feed, uid, MICROFEED_ITEM_STATUS_MARKED);
	}

	return error;
}

static MicrofeedError* method_unmark_item(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedError* error = NULL;
	DBusError dbus_error;
	const char* uri;
	const char* uid;
	MicrofeedFeed* feed;
	
	dbus_error_init(&dbus_error);
	if (!dbus_message_get_args(message, &dbus_error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_STRING, &uid, DBUS_TYPE_INVALID)) {
		error = microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri and item uid arguments in UnarkItem method");
	} else if (!(feed = microfeed_store_get(subscriber->feeds, uri, MicrofeedFeed))) {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Trying to unmark an item of a feed that does not exist.");
	} else {
		reply_to_current_message(publisher, NULL);
		
		microfeed_feed_unset_item_status(feed, uid, MICROFEED_ITEM_STATUS_MARKED);
	}

	return error;
}

static MicrofeedError* method_ping(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {

	return NULL;
}

static MicrofeedError* method_subscribe_feed(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedFeed* feed;
	DBusError error;
	char* uri;
	Timeout* timeout;
	int new_feed = 0;

	dbus_error_init(&error);
	if (!dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_INVALID)) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri argument in SubscribeFeed method");
	}
	
	if (!(feed = microfeed_store_get(publisher->feeds, uri, MicrofeedFeed))) {
		new_feed = 1;
		if ((feed = publisher->callbacks.instantiate_feed(publisher, uri, publisher->user_data))) {
			microfeed_store_insert(publisher->feeds, feed);
			if ((timeout = timeout_new(publisher, feed))) {
				microfeed_store_insert(publisher->timeouts, timeout);
			}
		} else {

			return microfeed_error_new(MICROFEED_ERROR_NO_SUCH_FEED, "Failed to subscribe a new feed");
		}
	}
	microfeed_store_insert(subscriber->feeds, feed);

	reply_to_current_message(publisher, NULL);

	microfeed_feed_add_subscriber(feed, subscriber->unique_connection_name);
	microfeed_publisher_send_item_signal(publisher, subscriber->unique_connection_name, MICROFEED_SIGNAL_NAME_ITEM_ADDED, uri, microfeed_feed_get_metadata_item(feed));
	if (new_feed) {
		microfeed_feed_update(feed, NULL);
	}

	return NULL;
}

static MicrofeedError* method_unsubscribe_feed(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	DBusError error;
	char* uri;
	MicrofeedFeed* feed;
	Timeout* timeout;
	
	dbus_error_init(&error);
	if (!dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_INVALID)) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri argument in UnsubscribeFeed method");
	}
	if (!(feed = microfeed_store_remove_key(subscriber->feeds, uri, MicrofeedFeed))) {
	
		return microfeed_error_new(MICROFEED_ERROR_FEED_NOT_SUBSCRIBED, "Trying to unsubscribe a feed that is not subscribed");
	}

	reply_to_current_message(publisher, NULL);
	
	microfeed_feed_remove_subscriber(feed, subscriber->unique_connection_name);
	if (microfeed_feed_get_subscriber_count(feed) == 0) {
		if ((timeout = microfeed_store_get(publisher->timeouts, feed, Timeout))) {
			timeout_free(timeout);
		}
		microfeed_store_remove(publisher->feeds, feed);
		microfeed_feed_unref(feed);
	}

	return NULL;
}

static MicrofeedError* method_update_feed(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedFeed* feed;
	DBusError error;
	char* uri;

	dbus_error_init(&error);
	if (!dbus_message_get_args(message, &error, DBUS_TYPE_STRING, &uri, DBUS_TYPE_INVALID)) {
		
		return microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "Expected feed uri argument in UpdateFeed method");
	}
	
	if (!(feed = microfeed_store_get(subscriber->feeds, uri, MicrofeedFeed))) {
	
		return microfeed_error_new(MICROFEED_ERROR_FEED_NOT_SUBSCRIBED, "Trying to update a feed that is not subscribed");
	}

	reply_to_current_message(publisher, NULL);
	
	microfeed_feed_update(feed, subscriber->unique_connection_name);
	
	return NULL;
}

static MicrofeedError* method_create_publisher(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	/* TODO: Check that publisher does not exist. */

	reply_to_current_message(publisher, NULL);
	
	microfeed_publisher_add_setting(publisher, "archive_old_items", "Archive old items", "hours", "non-negative integer", "5", "5");
	microfeed_publisher_add_setting(publisher, "update_interval", "Update interval", "minutes", "non-negative integer", "5", "1");
	
	publisher->callbacks.initialize_settings(publisher, publisher->user_data);

	return NULL;
}

static MicrofeedError* method_destroy_publisher(MicrofeedPublisher* publisher, Subscriber* subscriber, DBusMessage* message) {
	MicrofeedStoreIterator* subscriber_iterator;
	Subscriber* feed_subscriber;
	MicrofeedStoreIterator* feed_iterator;
	MicrofeedFeed* feed;

	reply_to_current_message(publisher, NULL);
	
	dbus_connection_unregister_object_path(publisher->connection, publisher->object_path);
	
	for (subscriber_iterator = microfeed_store_iterate(publisher->subscribers, NULL);
	     (feed_subscriber = microfeed_store_iterator_get(subscriber_iterator, Subscriber));
	     microfeed_store_iterator_next(subscriber_iterator)) {
		for (feed_iterator = microfeed_store_iterate(feed_subscriber->feeds, NULL);
		     (feed = microfeed_store_iterator_get(feed_iterator, MicrofeedFeed));
		     microfeed_store_iterator_next(feed_iterator)) {
			microfeed_feed_remove_subscriber(feed, subscriber->unique_connection_name);
			microfeed_publisher_send_error_signal(publisher, subscriber->unique_connection_name, MICROFEED_ERROR_NO_SUCH_PUBLISHER, NULL, NULL, "Publisher destroyed.");
		}
	}
	for (feed_iterator = microfeed_store_iterate(publisher->feeds, NULL);
	     (feed = microfeed_store_iterator_get(feed_iterator, MicrofeedFeed));
	     microfeed_store_iterator_next(feed_iterator)) {
		microfeed_feed_unref(feed);
	}

	return NULL;	
}

static void object_unregister(DBusConnection* connection, void* user_data) {
}

static DBusHandlerResult object_message(DBusConnection* connection, DBusMessage* message, void* user_data) {
	DBusHandlerResult result = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
	MicrofeedPublisher* publisher;
	const char* unique_connection_name;
	int i;
	Subscriber* subscriber;
	char buffer[1024];
	MicrofeedError* error;

	publisher = (MicrofeedPublisher*)user_data;	
	unique_connection_name = dbus_message_get_sender(message);
	if (!(subscriber = (Subscriber*)microfeed_store_get(publisher->subscribers, unique_connection_name, Subscriber))) {
		subscriber = subscriber_new(publisher, unique_connection_name);
		microfeed_store_insert(publisher->subscribers, subscriber);
		
		snprintf(buffer, 1024, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus',member='NameOwnerChanged',arg0='%s'", unique_connection_name);
		dbus_bus_add_match(connection, buffer, NULL);
	}

	subscriber->last_activity = time(NULL);
	for (i = 0; method_callbacks[i].name; i++) {
		if (dbus_message_is_method_call(message, MICROFEED_DBUS_INTERFACE_PUBLISHER, method_callbacks[i].name)) {
			printf("%s (%s)\n", method_callbacks[i].name, unique_connection_name);
			publisher->current_message = message;
			error = method_callbacks[i].callback(publisher, subscriber, message);
			reply_to_current_message(publisher, error);
			if (error) {
				microfeed_error_free(error);
			}
			result = DBUS_HANDLER_RESULT_HANDLED;
			break;
		}
	}
	
	if (microfeed_store_get_size(subscriber->feeds) == 0) {
		snprintf(buffer, 1024, "type='signal',sender='org.freedesktop.DBus',interface='org.freedesktop.DBus',method='NameOwnerChanged',arg0='%s'", unique_connection_name);
		dbus_bus_remove_match(connection, buffer, NULL);

		microfeed_store_remove(publisher->subscribers, subscriber);
		subscriber_free(subscriber);
	}

	if (publisher->callbacks.no_more_subscribers &&
	    microfeed_store_get_size(publisher->subscribers) == 0 &&
	    microfeed_thread_pool_get_started_thread_count(publisher->thread_pool) == 0) {
		publisher->callbacks.no_more_subscribers(publisher, publisher->user_data);
	}

	return result;
}

static void reply_to_current_message(MicrofeedPublisher* publisher, MicrofeedError* error) {
	char buffer[1024];
	DBusMessage* reply;

	if (publisher->current_message) {
		if (error) {
			snprintf(buffer, 1024, "%s.%s", MICROFEED_DBUS_INTERFACE_ERROR, microfeed_error_get_name(error));
			reply = dbus_message_new_error(publisher->current_message, buffer, microfeed_error_get_message(error));
		} else {
			reply = dbus_message_new_method_return(publisher->current_message);
		}
		dbus_connection_send(publisher->connection, reply, NULL);
		dbus_message_unref(reply);
		publisher->current_message = NULL;
	}	
}

static MicrofeedError* settings_feed_modify_item(MicrofeedFeed* feed, MicrofeedItem* existing_item, MicrofeedItem* new_item, void* user_data) {
	MicrofeedError* error = NULL;;
	MicrofeedPublisher* publisher;
	const char* value;
	
	publisher = microfeed_feed_get_publisher(feed);
	if (!existing_item) {
		error = microfeed_error_new(MICROFEED_ERROR_NO_SUCH_ITEM, "No such setting.");
	} else {
		if ((value = microfeed_item_get_property(new_item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE))) {
			if (publisher->callbacks.update_setting(publisher, microfeed_item_get_uid(new_item), value, publisher->user_data)) {
				microfeed_item_set_property(existing_item, MICROFEED_ITEM_PROPERTY_NAME_SETTING_VALUE, value);
				microfeed_feed_replace_item(feed, existing_item);
			}
			/* TODO: If internal setting (such as update_interval), do necessary actions. */
		} else {
			error = microfeed_error_new(MICROFEED_ERROR_INVALID_ARGUMENTS, "No setting.value property in settings item.");
		}
	}
	
	return error;
}

static MicrofeedError* images_feed_download_item_data(MicrofeedFeed* feed, const char* uid, void** data, size_t* length, void* user_data) {
	MicrofeedPublisher* publisher;
	MicrofeedError* error;
	
	publisher = microfeed_feed_get_publisher(feed);
	error = publisher->callbacks.download_image(publisher, uid, data, length, publisher->user_data);
	if (!error) {
		microfeed_feed_unset_item_status(feed, uid, MICROFEED_ITEM_STATUS_UNREAD);
	}
	
	return error;
}

static void thread_exited(MicrofeedThread* thread, void* user_data) {
	MicrofeedPublisher* publisher;
	
	publisher = (MicrofeedPublisher*)user_data;
	if (publisher->callbacks.no_more_subscribers &&
	    microfeed_store_get_size(publisher->subscribers) == 0 &&
	    microfeed_thread_pool_get_started_thread_count(publisher->thread_pool) == 0) {
		publisher->callbacks.no_more_subscribers(publisher, publisher->user_data);
	}
}	

static void timeout_handler(void* data) {
	Timeout* timeout;

	timeout = (Timeout*)data;
	microfeed_feed_update(timeout->feed, NULL);
}
