/*
 * interface.c -  MySocials driver.
 * This file is part of MSA program.
 *
 * Copyright (C) 2009, 2010 - MySocials team
 *
 * MSA program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MSA program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MSA program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include <libxml/tree.h>
#include <libxml/parser.h>

#include "driver.h"
#include "interface.h"
#include "profile.h"

/**
 * @brief Function process requests to driver
 */
static error_type driver_send(xmlDocPtr request, xmlDocPtr* response, const struct msa_module* info);

static error_type driver_shutdown(const struct msa_module*);

void msa_module_get_info(struct msa_module* d)
{
    d->name = DRIVER_NAME;
    d->driverName = DRIVER_ID;
    d->pic = DRIVER_PIC;
}

/**
 * Module initialization. Prepare driver for work.
 *
 * @param d - msa_module, information for dispatcher
 * @return 0 on success or 1 otherwise.
 */
error_type msa_module_init(struct msa_module* d)
{
    struct ProfileData *cp;
    
    msa_debug("driver name = %s, id = %s: init START", DRIVER_NAME, d->id);

    if(d->id == NULL)
        d->id = DRIVER_ID;
    
    cp = profile_create(d->id);

/* dispatcher must free memory(struct msa_module) when application shutdown */
    d->name = DRIVER_NAME;
    d->driverName = DRIVER_ID;
    d->send = driver_send;
    d->shutdown = driver_shutdown;
    d->pic = DRIVER_PIC;

    g_type_init();

    msa_debug("driver %s: init END", DRIVER_NAME);
    return SUCCESS;
}

/**
 * Driver handler
 *
 * @param request -  
 * @param response - 
 * @return 0 on success or 1 otherwise.
 */
static error_type driver_send(xmlDocPtr req, xmlDocPtr* response, const struct msa_module* info)
{
    xmlDocPtr respXml = NULL;
    xmlNodePtr respNode = NULL;
    gchar *tmp;

    if (info == NULL) {
        msa_debug("msa_module in NULL! can't perform request!");
        return FAILURE;
    } else {
        msa_debug("get proxy settings %s %d ", info->proxy, info->port);
    }

    msa_debug("driver_send: START");

    if(request == NULL) {
        g_warning("Error! Can't read request\n");
        return FAILURE;
    }

    upload_size = 0;
    download_size = 0;

    request(req, &respXml, info);
    xmlFreeDoc(req);

    respNode = xmlDocGetRootElement(respXml);
    if(respNode != NULL){
        tmp = g_strdup_printf("%d", upload_size);
        xmlSetProp(respNode, BAD_CAST "upload", BAD_CAST tmp);
        g_free(tmp);

        tmp = g_strdup_printf("%d", download_size);
        xmlSetProp(respNode, BAD_CAST "download", BAD_CAST tmp);
        g_free(tmp);
    }
    *response = xmlCopyDoc(respXml, 1);
    xmlFreeDoc(respXml);
    msa_debug("driver_send: END");
    return SUCCESS;
}

/**
 * Shutdown driver
 *
 * @return SUCCESS
 */
static error_type driver_shutdown(const struct msa_module *info)
{
    struct ProfileData *cp = (struct ProfileData *)profile_get_by_id(info->id);
    msa_debug(__FUNCTION__);
    if (cp == NULL)
        return FAILURE;
    if (cp == profile_list) {
        profile_list = profile_list->next;
    } else {
        /* try to find previous profile */
        struct ProfileData *prev = (struct ProfileData *)profile_list;
        while(prev->next != cp)
            prev = prev->next;
        prev->next = cp->next;
    }
    profile_remove(cp);

/* maybe remove cookie and logout from server */
    return SUCCESS;
}
