/*
 * facebook.c -  driver facebook of MSA program
 * This file is part of MSA program.
 *
 * Copyright (C) 2009-2010 - Sergey Zakharov
 * Copyright (C) 2010 - Pavel Shiryaev
 *
 * MSA program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * MSA program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with MSA program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include <libxml/tree.h>
#include <libxml/parser.h>

#include "driver.h"
#include "facebook.h"
#include "profile.h"

/*
 * @brief The empty response XML string. Contains structure and main items
 */
const char* response_string = "\
<TransitData>\
<TargetID></TargetID>\
<SourceID></SourceID>\
<Content></Content>\
</TransitData>";

/**
 * @brief Function process requests to driver
 */
static error_type fb_driver_send(xmlDocPtr request, xmlDocPtr* response, const struct msa_module* info);

static error_type fb_driver_shutdown(const struct msa_module*);

void msa_module_get_info(struct msa_module* d)
{
    d->name = DRIVER_NAME;
    d->pic = FB_PIC;
}

/**
 * Module initialization. Prepare driverFB for work.
 *
 * @param d - msa_module, information for dispatcher
 * @return 0 on success or 1 otherwise.
 */
error_type msa_module_init(struct msa_module* d)
{
    struct ProfileData *cp;
#ifdef TEST
    g_debug("TEST = %d", TEST);
#endif
    g_debug("driver name = %s, id = %s: init START", DRIVER_NAME, d->id);

    if(d->id == NULL)
        cp = profile_create(DRIVER_ID);
    else
        cp = profile_create(d->id);

/*dispatcher must free memory(struct msa_module) when application shutdown*/
    d->name = g_strdup(DRIVER_NAME);
    d->id = g_strdup(cp->id);
    /*d->name = DRIVER_NAME;*/
    d->send = fb_driver_send;
    d->shutdown = fb_driver_shutdown;
    d->pic = g_strdup(FB_PIC);
    /*d->pic = FB_PIC;*/

    d->state = NOT_CONF;
    d->status = OFF;

    g_debug("driver %s: init END", DRIVER_NAME);
    return SUCCESS;
}

/**
 * DriverFB handler
 *
 * @param request -  
 * @param response - 
 * @return 0 on success or 1 otherwise.
 */
static error_type fb_driver_send(xmlDocPtr request, xmlDocPtr* response, const struct msa_module* info)
{
    xmlDocPtr reqXml = NULL, respXml = NULL;
    xmlDocPtr buffXml = NULL;
    xmlNodePtr node = NULL, node_request = NULL, node_response = NULL;
    xmlChar* source;

    if (info == NULL) {
        g_debug("msa_module in NULL! can't perform request!");
        return FAILURE;
    } else {
        g_debug("get proxy settings %s %d ", info->proxy, info->port);
    }

    g_debug("fb_driver_send: START");

    respXml = xmlNewDoc(BAD_CAST "1.0");
    /*respXml->encoding = UTF8;*/


    if(request == NULL) {
        g_warning("Error! Can't read request\n");
        return FAILURE;
    }

    node = xpath_get_node("//TransitData/SourceID", request);
    if(node != NULL)
        source = xmlNodeGetContent(node);
    else
        return FAILURE;

    node = xpath_get_node("//TransitData/Content/Request", request);
    if(node != NULL){
        node_request = xmlCopyNode(node, 1);
        xmlUnlinkNode(node_request);
        reqXml = xmlNewDoc(BAD_CAST "1.0");
        /*reqXml->encoding = g_strdup(UTF8);*/
        xmlDocSetRootElement(reqXml, node_request);
    } else {
        return FAILURE;
      }

    xmlFreeDoc(request);
    
    fb_req(reqXml, &respXml, info);
    xmlFreeDoc(reqXml);

    node = xmlDocGetRootElement(respXml);
    node_response = xmlCopyNode(node, 1);
    xmlUnlinkNode(node_response);
    xmlFreeDoc(respXml);
    
    /*buffXml = xmlNewDoc( BAD_CAST "1.0");*/

    buffXml = xmlParseDoc(BAD_CAST response_string);
    /*buffXml->encoding = g_strdup(UTF8);*/
    
    node = xpath_get_node("//TransitData/TargetID", buffXml);
    if(node != NULL)
        xmlNodeSetContent(node, source);
    else
        return FAILURE;

    node = xpath_get_node("//TransitData/Content", buffXml);
    if(node != NULL)
        xmlAddChild(node, node_response);
    else
        return FAILURE;
    
    *response = xmlCopyDoc(buffXml, 1);
    xmlFreeDoc(buffXml);
    g_free(source);
    g_debug("fb_driver_send: END");
    return SUCCESS;
}

/**
 * Shutdown driver
 *
 * @return SUCCESS
 */
static error_type fb_driver_shutdown(const struct msa_module *info)
{
    struct ProfileData *cp = (struct ProfileData *)profile_get_by_id(info->id);
    g_debug("fb_driver_shutdown");
    if (cp == NULL)
        return FAILURE;
    if (cp == profile_list) {
        profile_list = profile_list->next;
    }else{
        /* try to find previous profile */
        struct ProfileData *prev = (struct ProfileData *)profile_list;
        while(prev->next != cp)
        {
            prev = prev->next;
        }
        prev->next = cp->next;
    }
    profile_remove(cp);

/*maybe remove cookie and logout from server*/
    return SUCCESS;
}
