/*
 * dbus_functions.c - methods for work with webauth application
 * This file is part of libvkontakte library
 *
 * Copyright (C) 2010 - Sergey Zakharov
 *
 * libvkontakte library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libvkontakte library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libvkontakte library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor,
 * Boston, MA  02110-1301  USA
 */

#include "dbus_functions.h"

#define ATTEMPTS_COUNT 5
/**
 * Utility to terminate if given DBusError is set.
 * Will print out the message and error before terminating.
 *
 * If error is not set, will do nothing.
 */
void terminateOnError(const char* msg,
                             const DBusError* error) {

  assert(msg != NULL);
  assert(error != NULL);

  if (dbus_error_is_set(error)) {
    fprintf(stderr, msg);
    fprintf(stderr, "DBusError.name: %s\n", error->name);
    fprintf(stderr, "DBusError.message: %s\n", error->message);
    exit(EXIT_FAILURE);
  }
}

/**
 * start webauth application, set proxy
 *
 * @param url - url for authorization
 * @param proxy - proxy host 
 * @param port - proxy port
 * @param user - username
 * @param pass - password
 * @return back url
 */
char* webauth(char* url, char* proxy, char* port, char* user, char* pass, char** expr, int elen, char* client_id) {
    
    char* back_url = NULL;
    DBusConnection* bus = NULL;
    DBusMessage* msg = NULL;
    DBusError error;

  /* Clean the error state. */
    dbus_error_init(&error);

    printf("Connecting to Session D-Bus\n");
    bus = dbus_bus_get(DBUS_BUS_SESSION, &error);
    terminateOnError("Failed to open Session bus\n", &error);
    assert(bus != NULL);

    terminateOnError("Failed to check for name ownership\n", &error);

  /* Construct a DBusMessage that represents a method call.
     Parameters will be added later. The internal type of the message
     will be DBUS_MESSAGE_TYPE_METHOD_CALL. */
    printf("Creating a message object\n");
    if (proxy != NULL && port != NULL) {
        msg = dbus_message_new_method_call(SYSNOTE_NAME, /* destination */
                                         SYSNOTE_OPATH,  /* obj. path */
                                         SYSNOTE_IFACE,  /* interface */
                                         SYSNOTE_NOTE_PROXY); /* method str */
        if (msg == NULL) {
          fprintf(stderr, "Ran out of memory when creating a message\n");
          return NULL;
        }
        
        printf("Appending arguments to the message\n");
        if (!dbus_message_append_args(msg,
                                    DBUS_TYPE_STRING, &proxy,
                                    DBUS_TYPE_STRING, &port,
                                    DBUS_TYPE_STRING, &user,
                                    DBUS_TYPE_STRING, &pass,
                                    DBUS_TYPE_INVALID)) {
           fprintf(stderr, "Ran out of memory while constructing args\n");
           dbus_message_unref(msg);
           dbus_connection_unref(bus);
           return NULL;
        }

        printf("Adding message to client's send-queue\n");

        if (!dbus_connection_send(bus, msg, NULL)) {
            fprintf(stderr, "Ran out of memory while queueing message\n");
            dbus_message_unref(msg);
            dbus_connection_unref(bus);
            return NULL;
        }

        printf("Waiting for send-queue to be sent out\n");
        dbus_connection_flush(bus);

        printf("Queue is now empty\n");

      /* Free up the allocated message.*/
        dbus_message_unref(msg);
        msg = NULL;
    }

/*Authorization request*/

    printf("Creating a message object\n");
    msg = dbus_message_new_method_call(SYSNOTE_NAME, /* destination */
                                     SYSNOTE_OPATH,  /* obj. path */
                                     SYSNOTE_IFACE,  /* interface */
                                     SYSNOTE_NOTE_AUTH); /* method str */

    if (msg == NULL) {
        fprintf(stderr, "Ran out of memory when creating a message\n");
        dbus_connection_unref(bus);
        return NULL;
    }

    printf("Appending arguments to the message\n");
       if (elen == 0) {
        if (!dbus_message_append_args(msg,
                                    DBUS_TYPE_STRING, &url,
                                    DBUS_TYPE_INVALID)) {
            fprintf(stderr, "Ran out of memory while constructing args\n");
            dbus_message_unref(msg);
            dbus_connection_unref(bus);
            return NULL;
        }
    } else {
        if (!dbus_message_append_args(msg,
                                    DBUS_TYPE_STRING, &url,
                                    DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &expr, elen,
                                    DBUS_TYPE_INVALID)) {
            fprintf(stderr, "Ran out of memory while constructing args\n");
            dbus_message_unref(msg);
            dbus_connection_unref(bus);
            return NULL;
        }
    }

    printf("Adding message to client's send-queue\n");
    if (!dbus_connection_send(bus, msg, NULL)) {
        fprintf(stderr, "Ran out of memory while queueing message\n");
        dbus_message_unref(msg);
        dbus_connection_unref(bus);
        return NULL;
    }

    printf("Waiting for send-queue to be sent out\n");
    dbus_connection_flush(bus);

    printf("Queue is now empty\n");

    /* Now we could in theory wait for exceptions on the bus, but since
     this is only a simple D-Bus example, we'll skip that. */

    receive(&back_url, client_id);
    printf("Cleaning up\n");

    /* Free up the allocated message.*/
    dbus_message_unref(msg);
    msg = NULL;

    dbus_connection_unref(bus);
    bus = NULL;

    printf("Quitting (success)\n");

return back_url;
}

/**
 * receive response from webauth application
 *
 * @param url - pointer to back url
 */

void receive(char** url, char* client_id)
{
    DBusMessage* msg;
    DBusMessageIter args;
    DBusConnection* conn;
    DBusError err;
    char **response = NULL;
    int attempts = 0;
    int len;

    printf("Listening for signals for %s\n", client_id);

    /* initialise the errors */
    dbus_error_init(&err);

    /* connect to the bus and check for errors */
    conn = dbus_bus_get(DBUS_BUS_SESSION, &err);
    if (dbus_error_is_set(&err)) { 
        fprintf(stderr, "Connection Error (%s)\n", err.message);
        dbus_error_free(&err); 
    }

    if (NULL == conn) { 
        return;
    }


    /* add a rule for which messages we want to see */
    dbus_bus_add_match(conn, SEARCH_RULE, &err); 
    dbus_connection_flush(conn);
    if (dbus_error_is_set(&err)) { 
        fprintf(stderr, "Match Error (%s)\n", err.message);
        dbus_connection_unref(conn);
        return; 
    }
    printf("Match rule sent\n");

    /* loop listening for signals being emmitted */
    /* wait wile webauth application start */
    sleep(3);
    while (1) {
        printf("wait response\n");

        /* non blocking read of the next available message */

        if(!dbus_connection_read_write(conn, 0))
            attempts++;
        if(attempts == ATTEMPTS_COUNT)
            break;

        msg = dbus_connection_pop_message(conn);

        /* loop again if we haven't read a message */
        if (NULL == msg) { 
            if (!dbus_bus_name_has_owner(conn, SYSNOTE_NAME, &err)) {
                fprintf(stderr, "Name has no owner on the bus!\n");
                printf("no webauth on dbus\n");
                break;
            }
            sleep(2);
            continue;
        }

        /* check if the message is a signal from the correct interface and with the correct name */
        if (dbus_message_is_signal(msg, SYSNOTE_NAME, SYSNOTE_SIGNAL)) {
         
            /* read the parameters */
            if (!dbus_message_iter_init(msg, &args))
                fprintf(stderr, "Message Has No Parameters\n");
            else if (DBUS_TYPE_ARRAY != dbus_message_iter_get_arg_type(&args)) {
                fprintf(stderr, "Argument is not array!\n"); 
            }
            else {
                if (dbus_message_get_args(msg,
                    &err, DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &response, &len,  
                    DBUS_TYPE_INVALID)) {
                    g_print("Received: %s %s %s %s %d\n", response[0], response[1], 
                                                          response[2], response[3], len);
                    if(g_strcmp0(response[0], RESULT_NAME) == 0 
                        && g_strcmp0(response[2], ACCOUNT_ID_NAME) == 0
                        && g_strcmp0(response[3], client_id) == 0){
                        *url = g_strdup(response[1]);
                    } else {
                        dbus_free_string_array(response);
                        dbus_message_unref(msg); 	
                        continue;
                    }
                    dbus_free_string_array(response); 	
                }
            }

            dbus_message_unref(msg);
            break;
        }

        /* free the message */
        dbus_message_unref(msg);
    }
   dbus_connection_unref(conn); 
}

