#include "../ColumbusController.h"

#include "LoggingManager.h"

#include "FileLogger.h"
#include "FileLoggerConfig.h"

#include "NMEAGeoLoggingFormatter.h"

#include <QDebug>
#include <QtCore>
#include <QDesktopServices>

class LoggingManagerPrivate
{
public:
    ColumbusController *controller;

    QHash<QString,AbstractGeoLoggingFormatter*>               formatters;
    QHash<AbstractLoggingPerformer*,AbstractGeoLoggingFormatter*>   loggers;

    AbstractLoggingPerformer *defaultLogger;
};

LoggingManager::LoggingManager(QObject *parent)
    : QObject(parent)
{
    this->d = new LoggingManagerPrivate;
    d->controller       = NULL;
    d->defaultLogger    = NULL;
}

LoggingManager::~LoggingManager()
{
    delete this->d;
}

QHash<QString,AbstractGeoLoggingFormatter*> LoggingManager::formatters () const
{
    return d->formatters;
}

QHash<AbstractLoggingPerformer*,AbstractGeoLoggingFormatter*> LoggingManager::loggers () const
{
    return d->loggers;
}

void LoggingManager::initialize(ColumbusController *controller)
{
    d->controller = controller;

    this->addDataFormat(new NMEAGeoLoggingFormatter(this));

    // If we're configured to do file logging, do it!
    if(FileLoggerConfig().isEnabled() && FileLoggerConfig().isRunOnStartup())
    {
        this->setupLogging();
    }
    else
    {
        FileLoggerConfig().setEnabled(false);
    }
}

void LoggingManager::setupLogging()
{
    FileLoggerConfig config;

    if(d->defaultLogger != NULL)
    {
        qDebug() << "LoggingManager: Destroying redundant logger.";
        this->removeLogger(d->defaultLogger);
        delete d->defaultLogger;
        d->defaultLogger = NULL;
    }

    if(config.isEnabled())
    {
        qDebug() << "LoggingManager: Setting up data logging.";

        AbstractGeoLoggingFormatter *formatter = this->formatters().value(config.formatId());

        const QString logdir = QDesktopServices::storageLocation(QDesktopServices::CacheLocation);
        const QString filename = QString("yyyy-MM-dd-%1").arg(config.formatId());

        if(!QDir().exists(logdir))
        {
            qDebug() << "LoggingManager: Creating log directory:" << logdir;
            QDir().mkpath(logdir);
        }

        d->defaultLogger = new FileLogger(logdir, filename, this);
        this->addLogger(formatter, d->defaultLogger);
    }
}

void LoggingManager::addDataFormat(AbstractGeoLoggingFormatter *formatter)
{
    d->formatters.insert(formatter->id(), formatter);
}

void LoggingManager::addLogger(AbstractGeoLoggingFormatter *formatter, AbstractLoggingPerformer *logger)
{
    QObject::connect(formatter, SIGNAL(dataReady(QByteArray)), logger, SLOT(onDataReady(QByteArray)));

    // If this formatter isn't in the active loggers repository, initialize is for receiving telemetry.
    if(!d->loggers.values().contains(formatter))
    {
        qDebug() << "LoggingManager: Connecting formatter" << formatter->id() << "to telemetry sources.";
        QObject::connect(d->controller, SIGNAL(positionUpdated(QGeoPositionInfo)),
                         formatter, SLOT(onPositionUpdate(QGeoPositionInfo)));
        QObject::connect(d->controller, SIGNAL(satellitesInUseUpdated(QList<QGeoSatelliteInfo>)),
                         formatter, SLOT(onSatellitesInUse(QList<QGeoSatelliteInfo>)));
        QObject::connect(d->controller, SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)),
                         formatter, SLOT(onSatellitesInView(QList<QGeoSatelliteInfo>)));
    }

    d->loggers.insert(logger, formatter);
}

void LoggingManager::removeLogger(AbstractLoggingPerformer *logger)
{
    AbstractGeoLoggingFormatter *formatter = d->loggers.value(logger);

    QObject::disconnect(formatter, SIGNAL(dataReady(QByteArray)), logger, SLOT(onDataReady(QByteArray)));

    d->loggers.remove(logger);

    // If this is the last reference, disconnect telemetry sources.
    if(!d->loggers.values().contains(formatter))
    {
        qDebug() << "LoggingManager: Disconnecting formatter" << formatter->id() << "from telemetry sources.";
        QObject::disconnect(d->controller, SIGNAL(positionUpdated(QGeoPositionInfo)),
                            formatter, SLOT(onPositionUpdate(QGeoPositionInfo)));
        QObject::disconnect(d->controller, SIGNAL(satellitesInUseUpdated(QList<QGeoSatelliteInfo>)),
                            formatter, SLOT(onSatellitesInUse(QList<QGeoSatelliteInfo>)));
        QObject::disconnect(d->controller, SIGNAL(satellitesInViewUpdated(QList<QGeoSatelliteInfo>)),
                            formatter, SLOT(onSatellitesInView(QList<QGeoSatelliteInfo>)));
    }
}
