#ifndef COLUMBUSMODEL_H
#define COLUMBUSMODEL_H

#include <QObject>
#include <QGeoPositionInfo>

QTM_USE_NAMESPACE

class ColumbusModelPrivate;

class ColumbusModel : public QObject
{
    Q_OBJECT

    Q_ENUMS(DisplayUnits)

    Q_PROPERTY(QString name READ name WRITE setName NOTIFY nameChanged)
    Q_PROPERTY(QString comment READ comment WRITE setComment NOTIFY commentChanged)

    Q_PROPERTY(DisplayUnits displayUnits READ displayUnits WRITE setDisplayUnits NOTIFY displayUnitsChanged)

    Q_PROPERTY(qreal   distance READ distance NOTIFY distanceUpdated)

    Q_PROPERTY(qreal   currentSpeed READ currentSpeed NOTIFY currentSpeedUpdated)
    Q_PROPERTY(qreal   averageSpeed READ averageSpeed NOTIFY averageSpeedUpdated)
    Q_PROPERTY(qreal   maximumSpeed READ maximumSpeed NOTIFY maximumSpeedUpdated)

    Q_PROPERTY(qreal   currentAltitude READ currentAltitude NOTIFY currentAltitudeUpdated)
    Q_PROPERTY(qreal   minimumAltitude READ minimumAltitude NOTIFY minimumAltitudeUpdated)
    Q_PROPERTY(qreal   maximumAltitude READ maximumAltitude NOTIFY maximumAltitudeUpdated)

    Q_PROPERTY(qreal   currentHeading  READ currentHeading NOTIFY currentHeadingUpdated)

    Q_PROPERTY(qreal   currentHorizontalAccuracy READ currentHorizontalAccuracy NOTIFY currentHorizontalAccuracyUpdated)
    Q_PROPERTY(qreal   currentVerticalAccuracy READ currentVerticalAccuracy NOTIFY currentVerticalAccuracyUpdated)

    Q_PROPERTY(int     travelTime READ travelTime NOTIFY travelTimeUpdated)
    Q_PROPERTY(int     elapsedTime READ elapsedTime NOTIFY elapsedTimeUpdated)

public:
    typedef enum
    {
        Metric,
        Imperial,
        Nautical
    } DisplayUnits;

    explicit ColumbusModel(QObject *parent = 0);
            ~ColumbusModel();

    QString name() const;
    QString comment() const;

    static qreal convertDistanceToUnits(DisplayUnits units, qreal distance);
    static qreal convertSpeedToUnits(DisplayUnits units, qreal speed);
    static qreal convertAltitudeToUnits(DisplayUnits units, qreal altitude);

    DisplayUnits displayUnits() const;

    QString speedUnits() const;
    QString distanceUnits() const;
    QString altitudeUnits() const;
    QGeoCoordinate::CoordinateFormat positionFormat() const;

    qreal distance() const;

    qreal currentSpeed() const;
    qreal averageSpeed() const;
    qreal maximumSpeed() const;

    qreal currentAltitude() const;
    qreal minimumAltitude() const;
    qreal maximumAltitude() const;

    qreal currentHeading() const;

    QGeoCoordinate currentPosition() const;

    QString currentLatitude() const;
    QString currentLongitude() const;

    qreal currentHorizontalAccuracy() const;
    qreal currentVerticalAccuracy() const;

    int travelTime() const;
    int elapsedTime() const;

    const QList<QGeoPositionInfo>* datalog() const;
    const QList<QGeoCoordinate>* tracklog() const;

signals:
    void updated();

    void nameChanged(const QString &name);
    void commentChanged(const QString &comment);

    void displayUnitsChanged(DisplayUnits units);
    void positionFormatChanged(QGeoCoordinate::CoordinateFormat format);

    void distanceUpdated(qreal distance);

    void currentSpeedUpdated(qreal speed);
    void averageSpeedUpdated(qreal speed);
    void maximumSpeedUpdated(qreal speed);

    void currentAltitudeUpdated(qreal altitude);
    void minimumAltitudeUpdated(qreal altitude);
    void maximumAltitudeUpdated(qreal altitude);

    void currentHeadingUpdated(qreal heading);

    void currentHorizontalAccuracyUpdated(qreal hacc);
    void currentVerticalAccuracyUpdated(qreal vacc);

    void travelTimeUpdated(int time);
    void elapsedTimeUpdated(int time);

public slots:
    void setName(const QString &name);
    void setComment(const QString &comment);

    void setDisplayUnits(DisplayUnits units);
    void setPositionFormat(QGeoCoordinate::CoordinateFormat);

    void onPositionUpdated(const QGeoPositionInfo &position);

private:
    ColumbusModelPrivate *d;
};

#endif // COLUMBUSMODEL_H
