/*
 *  Copyright (C) 2013 Nicolai Hess
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "ledpulse-home-plugin.h"
#include "ledpulse-led-files.h"
#include "ledpulse-settings-dialog.h"
#include "ledpulse-pulseaudio.h"
#include <gtk/gtk.h>
#include <errno.h>
#include <string.h>
#include <math.h>
	
HD_DEFINE_PLUGIN_MODULE(LedPulseHomePlugin, ledpulse_home_plugin, HD_TYPE_HOME_PLUGIN_ITEM);

#define LEDPULSE_HOME_PLUGIN_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), LEDPULSE_TYPE_HOME_PLUGIN, LedPulseHomePluginPrivate))

#define WIDGET_HEIGHT 96
#define WIDGET_WIDTH 96

#define IMAGES_DIR                   "/etc/hildon/theme/images/"
#define BACKGROUND_IMAGE_FILE        IMAGES_DIR "ApplicationShortcutApplet.png"
#define BACKGROUND_ACTIVE_IMAGE_FILE IMAGES_DIR "ApplicationShortcutAppletPressed.png"
#define SOURCE_MONITOR "sink.hw0.monitor"
#define SOURCE_MICROPHONE "source.hw0"
#define SOURCE_BLUETOOTH "source.hw1"

struct _LedPulseHomePluginPrivate {
  
  gboolean enabled;
  gboolean pressed;
  LedPulseAudio* pulse_audio;
  LedFileDescriptors* led_files;
};

static const gchar* ledpulse_home_plugin_source_for(LedPulseHomePlugin *self)
{
  if(g_strcmp0(self->settings->source, "Microphone")==0)
    return SOURCE_MICROPHONE;
  if(g_strcmp0(self->settings->source, "Bluetooth")==0)
    return SOURCE_BLUETOOTH;
  return SOURCE_MONITOR;
}

static inline gboolean is_visible(LedPulseHomePlugin *self)
{
  gboolean is_on_current_desktop;
  g_object_get(G_OBJECT(self),
	       "is-on-current-desktop", &is_on_current_desktop, NULL);
  return is_on_current_desktop;
}

static void ledpulse_home_plugin_visibility_change_callback(GObject *object, GParamSpec *pspec, gpointer data)
{
  LedPulseHomePlugin *self = LEDPULSE_HOME_PLUGIN(data);
  gboolean visible = is_visible(self);
}

static void
ledpulse_home_plugin_disable(LedPulseHomePlugin* self)
{
  self->priv->enabled = FALSE;
  ledpulse_pulseaudio_disconnect_stream(self->priv->pulse_audio);
}

static void
ledpulse_home_plugin_enable(LedPulseHomePlugin* self)
{
  self->priv->enabled = TRUE;
  ledpulse_pulseaudio_enable_streaming(self->priv->pulse_audio, ledpulse_home_plugin_source_for(self), self);
}

static gboolean
ledpulse_home_plugin_button_release(GtkWidget* widget, GdkEventButton *event, gpointer userdata)
{
  LedPulseHomePlugin* self = LEDPULSE_HOME_PLUGIN(userdata);
  LedPulseHomePluginPrivate* priv = self->priv;
  priv->pressed = FALSE;
  if(priv->enabled)
    ledpulse_home_plugin_disable(self);
  else
    ledpulse_home_plugin_enable(self);
  gtk_widget_queue_draw(GTK_WIDGET(self));
  return FALSE;
}

static gboolean
ledpulse_home_plugin_button_press(GtkWidget* widget, GdkEventButton *event, gpointer userdata)
{
  LedPulseHomePlugin* self = LEDPULSE_HOME_PLUGIN(userdata);
  LedPulseHomePluginPrivate* priv = self->priv;
  priv->pressed = TRUE;
  gtk_widget_queue_draw(GTK_WIDGET(self));
  return FALSE;
}

static void ledpulse_home_plugin_init_ui(LedPulseHomePlugin* self)
{
  GtkWidget *contents = gtk_event_box_new();
  gtk_event_box_set_visible_window(GTK_EVENT_BOX(contents), FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(contents), 0);
  g_signal_connect(GTK_CONTAINER(contents), "button-release-event",  G_CALLBACK(ledpulse_home_plugin_button_release), self);
  g_signal_connect(GTK_CONTAINER(contents), "button-press-event", G_CALLBACK(ledpulse_home_plugin_button_press), self);
  gtk_widget_set_size_request(contents, WIDGET_WIDTH, WIDGET_HEIGHT);
  gtk_container_add(GTK_CONTAINER(self), contents);
  gtk_widget_show_all(GTK_WIDGET(contents));
}

void ledpulse_home_plugin_process_audio_data(LedPulseHomePlugin* self, const void* data, size_t nbytes)
{
  if(nbytes<1)
    return;
  float v = fabs(((const float*)data)[0]);
  int idata = (int)(v*255.0);
  ledpulse_led_files_write_to_leds(self->priv->led_files, self->settings, idata);
}

void ledpulse_home_plugin_reload_settings(LedPulseHomePlugin* self) {
  gboolean was_enabled = self->priv->enabled;
  if(was_enabled)
    ledpulse_home_plugin_disable(self);
  ledpulse_settings_load(self->settings);
  ledpulse_led_files_close_leds(self->priv->led_files);
  ledpulse_led_files_open_from(self->priv->led_files, self->settings);
  if(was_enabled)
    ledpulse_home_plugin_enable(self);
}

static void ledpulse_home_plugin_dispose(GObject *object)
{
  LedPulseHomePlugin *self = LEDPULSE_HOME_PLUGIN(object);
  G_OBJECT_CLASS(ledpulse_home_plugin_parent_class)->dispose(object);
}

static void ledpulse_home_plugin_finalize(GObject *object)
{
  LedPulseHomePlugin *self = LEDPULSE_HOME_PLUGIN(object);
  ledpulse_pulseaudio_dispose(&self->priv->pulse_audio);
  if(self->priv->led_files)
  {
    ledpulse_led_files_close_leds(self->priv->led_files);
    g_free(self->priv->led_files);
  }
  if(self->settings)
    g_free(self->settings);
  G_OBJECT_CLASS(ledpulse_home_plugin_parent_class)->finalize(object);
}

static void ledpulse_home_plugin_realize(GtkWidget *widget)
{
  GdkScreen *screen = gtk_widget_get_screen(widget);
  gtk_widget_set_colormap(widget, gdk_screen_get_rgba_colormap(screen));
  gtk_widget_set_app_paintable(widget, TRUE);
  GTK_WIDGET_CLASS(ledpulse_home_plugin_parent_class)->realize(widget);
}

static void ledpulse_home_plugin_size_request(GtkWidget *widget, GtkRequisition *requisition)
{
  requisition->width = WIDGET_WIDTH;
  requisition->height = WIDGET_HEIGHT;
}

static void draw_background(LedPulseHomePlugin* self, cairo_t* cr)
{ 
  cairo_set_operator(cr, CAIRO_OPERATOR_SOURCE);
  GdkPixbuf* pixbuf_background = NULL;
  if(!self->priv->pressed)
    pixbuf_background = gdk_pixbuf_new_from_file(BACKGROUND_IMAGE_FILE, NULL);
  else
    pixbuf_background = gdk_pixbuf_new_from_file(BACKGROUND_ACTIVE_IMAGE_FILE, NULL);
  gdk_cairo_set_source_pixbuf(cr, pixbuf_background, 0, 0);
  cairo_paint(cr);
  g_object_unref(pixbuf_background);
}

static void draw_icon(LedPulseHomePlugin* self, cairo_t* cr)
{
  GdkPixbuf* pixbuf =  gdk_pixbuf_new_from_file("/usr/share/icons/hicolor/48x48/hildon/control_led.png", NULL);
  cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
  gdk_cairo_set_source_pixbuf(cr, pixbuf, 24, 24);
  if(self->priv->enabled)
    cairo_set_operator(cr, CAIRO_OPERATOR_OVER);
  else
    cairo_set_operator(cr, CAIRO_OPERATOR_SATURATE);
  cairo_paint(cr);
  g_object_unref(pixbuf);
}

static gboolean ledpulse_home_plugin_expose(GtkWidget *widget, GdkEventExpose *event)
{
  LedPulseHomePlugin *self = LEDPULSE_HOME_PLUGIN(widget);
  cairo_t *cr = gdk_cairo_create(widget->window);
  cairo_rectangle(cr, event->area.x, event->area.y, event->area.width, event->area.height);
  cairo_clip(cr);
  draw_background(self, cr);
  draw_icon(self, cr);
  cairo_destroy(cr);
  return GTK_WIDGET_CLASS(ledpulse_home_plugin_parent_class)->expose_event(widget, event);
}

static void ledpulse_home_plugin_init(LedPulseHomePlugin *self)
{
  LedPulseHomePluginPrivate *priv;
  self->priv = priv = LEDPULSE_HOME_PLUGIN_GET_PRIVATE(self);	
  self->priv->enabled = FALSE;
  self->priv->led_files = NULL;
  system("sudo /usr/bin/ledpulse-chmod");
  self->settings = ledpulse_settings_create();
  self->priv->led_files = ledpulse_led_files_create();
  self->priv->pulse_audio = ledpulse_pulseaudio_create();
  
  ledpulse_settings_load(self->settings);
  ledpulse_led_files_open_from(self->priv->led_files, self->settings);
  ledpulse_pulseaudio_init(self->priv->pulse_audio);

  hd_home_plugin_item_set_settings(HD_HOME_PLUGIN_ITEM(self), TRUE);
  g_signal_connect(G_OBJECT(self), "show-settings", G_CALLBACK(ledpulse_settings_show), self);
  g_signal_connect(G_OBJECT(self), "notify::is-on-current-desktop",
		   G_CALLBACK(ledpulse_home_plugin_visibility_change_callback), self);
  ledpulse_home_plugin_init_ui(self);
}

static void ledpulse_home_plugin_class_init(LedPulseHomePluginClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

  gobject_class->dispose = ledpulse_home_plugin_dispose;
  gobject_class->finalize = ledpulse_home_plugin_finalize;

  widget_class->realize = ledpulse_home_plugin_realize;
  widget_class->size_request = ledpulse_home_plugin_size_request;
  widget_class->expose_event = ledpulse_home_plugin_expose;

  g_type_class_add_private(klass, sizeof(LedPulseHomePluginPrivate));
}

static void ledpulse_home_plugin_class_finalize(LedPulseHomePluginClass *klass)
{
	
}
