#include "viewer.h"
#include "editwidget.h"

Viewer::Viewer(QWidget *parent, Qt::WFlags flags): QMainWindow(parent, flags)
{
	currentImageNumber = -1;
	initMenu();
	initToolBar();
	initViewer();
	initEditor();
	initStatusBar();
	scaleFactor = 1.0;
	editScaleFactor = 1.0;
	getConfig(cfg);
	fitToScreen();
}

Viewer::~Viewer()
{

}


void Viewer::Error(QString err)
{
	QMessageBox mb(this);
	mb.setIcon(QMessageBox::Warning);
	mb.setText(err);
	mb.exec();
}


void Viewer::showMessage(QString message)
{
	QMessageBox mbx(this);
	mbx.setIcon(QMessageBox::Information);
	mbx.setText(message);
	mbx.exec();
}

void Viewer::ScaleImage(double factor)
{
	
	if(currentImageNumber > -1)
	{
		scaleFactor *= factor;
		imageLabel->resize(scaleFactor * imageLabel->pixmap()->size());
	}
}
void Viewer::initToolBar()
{
	mainToolBar = this->addToolBar(tr("Main ToolBar"));
        mainToolBar->setIconSize(QSize(40,40));
	QAction *actNext = new QAction(tr("Next"), this);
	QAction *actPrev = new QAction(tr("Prev"), this);
	QAction *actZoomIn = new QAction(tr("ZoomIn"), this);
	QAction *actZoomOut = new QAction(tr("ZoomOut"), this);
	QAction *actEdit = new QAction(tr("Edit"), this);
	if(currentImageNumber > -1)
	{
		actCounter->setText(QString::number(currentImageNumber+1) + "/" + QString::number(imageFileNames.size()));
	}
	else
	{
		actCounter = new QAction("--/--", this);
	}
	actNext->setIcon(QIcon("Resources/next.png"));
	actPrev->setIcon(QIcon("Resources/prev.png"));
	actZoomIn->setIcon(QIcon("Resources/zoomIn.png"));
	actZoomOut->setIcon(QIcon("Resources/zoomOut.png"));
	actEdit->setIcon(QIcon("Resources/edit.png"));
	mainToolBar->setOrientation(Qt::Vertical);
	mainToolBar->addAction(actPrev);	
	mainToolBar->addAction(actNext);
	mainToolBar->addAction(actZoomIn);
	mainToolBar->addAction(actZoomOut);
	mainToolBar->addAction(actEdit);
	mainToolBar->addAction(actCounter);
	mainToolBar->setAllowedAreas(Qt::LeftToolBarArea);
	connect(actNext, SIGNAL(triggered()), this, SLOT(showNextImage()));
	connect(actPrev, SIGNAL(triggered()), this, SLOT(showPreviousImage()));
	connect(actZoomIn, SIGNAL(triggered()), this, SLOT(zoomIn()));
	connect(actZoomOut, SIGNAL(triggered()), this, SLOT(zoomOut()));
	connect(actEdit, SIGNAL(triggered()), this, SLOT(showEditor()));
}
void Viewer::initStatusBar()
{
	statusBar = new QStatusBar(this);
	this->setStatusBar(statusBar);
	statusMessage = new QLabel("", statusBar);
	sizeStatusString = new QLabel("", statusBar);
	resolutionStatusString = new QLabel("", statusBar);
	statusBar->addWidget(statusMessage);
	statusBar->addWidget(resolutionStatusString);
	statusBar->addWidget(sizeStatusString);
}
void Viewer::initEditToolBar()
{
        editToolBar = this->addToolBar(tr("Main ToolBar"));
	editToolBar->setIconSize(QSize(40,40));
	actPen = new QAction(tr("Pen"), this);
        QAction *actSave = new QAction(tr("Save"), this);
	QAction *actCrop = new QAction(tr("Crop"), this);
	QAction *actSelect = new QAction(tr("Select"), this);
	QAction *actColor = new QAction(tr("Color"), this);
        QAction *actCancel = new QAction(tr("Cancel"), this);
	QAction *actUndo = new QAction(tr("Undo"), this);
	QAction *actRedo = new QAction(tr("Redo"), this);
	QAction *actRotateR = new QAction(tr("Rotate right"), this);
	QAction *actRotateL = new QAction(tr("Rotate left"), this);
	QAction *actBandW = new QAction(tr("Black and white"), this);
        QComboBox *combo = new QComboBox(this);
        editToolBar->setAllowedAreas(Qt::LeftToolBarArea);
        actCancel->setIcon(QIcon("Resources/cancel.png"));
	actUndo->setIcon(QIcon("Resources/undo.png"));
	actRedo->setIcon(QIcon("Resources/redo.png"));
	actPen->setIcon(QIcon("Resources/brush.png"));
        actSave->setIcon(QIcon("Resources/save.png"));
        actSelect->setIcon(QIcon("Resources/select.png"));
	actCrop->setIcon(QIcon("Resources/crop.png"));
	actColor->setIcon(QIcon("Resources/palette.png"));
	actRotateL->setIcon(QIcon("Resources/rotateL.png"));
	actRotateR->setIcon(QIcon("Resources/rotateR.png"));
	actBandW->setIcon(QIcon("Resources/blackandwhite.png"));
	editToolBar2 = this->addToolBar(tr("Main ToolBar"));
	editToolBar2->setIconSize(QSize(40,40));
	editToolBar->addAction(actSave);
	editToolBar->addAction(actPen);
        editToolBar->addAction(actSelect);	
	editToolBar->addAction(actCrop);
	editToolBar->addAction(actColor);
	editToolBar->addWidget(combo);
        editToolBar2->addAction(actRotateL);
	editToolBar2->addAction(actRotateR);
	editToolBar2->addAction(actCancel);
	editToolBar2->addAction(actUndo);
	editToolBar2->addAction(actRedo);
	editToolBar2->addAction(actBandW);
	int i;
	for(i = 1; i < 30; i++)
	{
		combo->addItem(QString::number(i));
	}
	editToolBar2->setAllowedAreas(Qt::RightToolBarArea);
        connect(actSave, SIGNAL(triggered()), this, SLOT(SaveImage()));
	connect(actPen, SIGNAL(triggered()), this, SLOT(drawPen()));
        connect(actSelect, SIGNAL(triggered()), this, SLOT(drawSelect()));
	connect(actColor, SIGNAL(triggered()), this, SLOT(chooseColor()));
	connect(actCrop, SIGNAL(triggered()), this, SLOT(drawCrop()));
	connect(combo, SIGNAL(currentIndexChanged(int)), this, SLOT(currentIndexChanged(int)));
        connect(actUndo, SIGNAL(triggered()), editImageLabel, SLOT(undo()));
	connect(actRedo, SIGNAL(triggered()), editImageLabel, SLOT(redo()));
	connect(actRotateL, SIGNAL(triggered()), editImageLabel, SLOT(rotateLeft()));
	connect(actRotateR, SIGNAL(triggered()), editImageLabel, SLOT(rotateRight()));
	connect(actCancel, SIGNAL(triggered()), this, SLOT(showEditor()));
	connect(actBandW, SIGNAL(triggered()), editImageLabel, SLOT(makeBlackAndWhite()));
	actPen->setCheckable(1);
	editToolBar->setOrientation(Qt::Vertical);
	editToolBar2->setOrientation(Qt::Vertical);
}
void Viewer::initMenu()
{	QMenuBar *menuBar = new QMenuBar(this);
	QMenu *m1 = new QMenu(tr("File"));
	QMenu *m2 = new QMenu(tr("Help"));
	QAction *openAct = new QAction(tr("O&pen"),m1);
        //act22 = new QAction(tr("Resize"),m1);
	QAction *act12 = new QAction(tr("Save"),m1);
	QAction *act21 = new QAction(tr("About"),m2);
	QAction *exitAct = new QAction(tr("E&xit"),m1);
	exitAct->setShortcut(tr("Ctrl+Q"));	

	
	m1->addAction(openAct);
	m1->addAction(act22);
	m1->addAction(act12);
	m1->addAction(exitAct);
	m2->addAction(act21);
	act22->setDisabled(1);
	
	connect(act12, SIGNAL(triggered()), this, SLOT(SaveImage()));
	connect(openAct, SIGNAL(triggered()), this, SLOT(LoadImages()));
	connect(act21, SIGNAL(triggered()), this, SLOT(About()));
	connect(act22, SIGNAL(triggered()), this, SLOT(Resize()));
	connect(exitAct, SIGNAL(triggered()), this, SLOT(close()));

	//menuBar->addMenu(m1);
	//menuBar->addMenu(m2);
	this->setMenuBar(menuBar);
}

void Viewer::initViewer()
{
	widgetSet = new QStackedWidget(this);
	QWidget *viewerWidget = new QWidget(widgetSet);
	vbl = new QVBoxLayout();
	imageLabel = new QLabel();
	scrollArea = new QScrollArea();

	imageLabel->setBackgroundRole(QPalette::Base);
	imageLabel->setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Ignored);
	imageLabel->setScaledContents(true);

	scrollArea->setWidget(imageLabel);
	viewerWidget->setLayout(vbl);
	viewerWidget->layout()->addWidget(scrollArea);
	widgetSet->addWidget(viewerWidget);
	this->setCentralWidget(widgetSet);
}
void Viewer::initEditor()
{
	QWidget *mainWidget = new QWidget(this);
	QVBoxLayout *vbl = new QVBoxLayout();
	editImageLabel = new EditWidget(this);
	scrollArea = new QScrollArea();
		
	scrollArea->setWidget(editImageLabel);
	mainWidget->setLayout(vbl);
	mainWidget->layout()->addWidget(scrollArea);
	widgetSet->addWidget(mainWidget);
}

void Viewer::showViewer(int imageNumber)
{
	qint64 fileSize;
	if(imageNumber > -1 && !imageFileNames.empty())
	{
		QFile imageFile(imageFileNames.at(imageNumber));
		QImage image(imageFileNames.at(imageNumber));
		if(!image.isNull())
		{
			imageLabel->setPixmap(QPixmap::fromImage(image));
			fitImageToLayout();
			//imageLabel->adjustSize();
			currentImageNumber = imageNumber;
			actCounter->setText(QString::number(currentImageNumber+1) + "/" + QString::number(imageFileNames.size()));
			setWindowTitle(imageFileNames.at(imageNumber));
			scaleFactor = 1.0;
			resolutionStatusString->setText(QString(QString::number(image.width()) 
									       + "x" + QString::number(image.height()) 
									       + " (" + QString::number(image.depth()) + "bit)"));
			fileSize = imageFile.size()/1024;
			if(fileSize/1024 > 0)
			{
				sizeStatusString->setText(QString::number(fileSize/1024) 
							 + "." + QString::number((fileSize%1024)/100) + "mb");
			}
			else
			{
				sizeStatusString->setText(QString::number(fileSize) + "kb");
			}
		}
	}
}

void Viewer::showStatusMessage(QString message)
{
	statusMessage->setText(message);
}
void Viewer::fitImageToLayout()
{
	int layoutWidth = vbl->geometry().size().width()-20;
	int layoutHeight = vbl->geometry().size().height()-20;
	int pixmapWidth = imageLabel->pixmap()->size().width();
	int pixmapHeight = imageLabel->pixmap()->size().height();
	qreal resizeValue=1.0; //float
	
	resizeValue = (float)layoutWidth/pixmapWidth;
	if(!((int)(pixmapWidth*resizeValue) <= layoutWidth && (int)(pixmapHeight*resizeValue) <= layoutHeight))
	{
		resizeValue = (float)layoutHeight/pixmapHeight;
	}
	
	imageLabel->resize((int)(pixmapWidth*resizeValue), (int)(pixmapHeight*resizeValue));
	scrollArea->setAlignment(Qt::AlignHCenter);
}
void Viewer::resizeEvent(QResizeEvent* event)
{
	if(imageLabel != NULL)
	{
		if(imageLabel->pixmap() != NULL)
		{
			fitImageToLayout();
		}
	}
}
void Viewer::getConfig(QSettings *cfg)
{
	cfg = NULL;
	cfg = new QSettings("settings.ini", QSettings::IniFormat, this);
	if(cfg == NULL)
	{
		QFile file("settings.ini");
		file.open(QIODevice::ReadWrite);
	}

}
/*******/
/*slots*/
/*******/
void Viewer::showPreviousImage()
{
	if(currentImageNumber-1 > -1 && currentImageNumber-1 < imageFileNames.size())
	{
		showViewer(currentImageNumber-1);
	}
	else
	{
		showViewer(imageFileNames.size()-1);
	}
}

void Viewer::showNextImage()
{
	if(currentImageNumber+1 > -1 && currentImageNumber+1 < imageFileNames.size())
	{
		showViewer(currentImageNumber+1);
	}
	else
	{
		showViewer(0);
	}
}

void Viewer::showEditor()
{
	int count = widgetSet->currentIndex();
	if(count == 0)
	{
		if(!imageFileNames.isEmpty())
		{
			QString fname = imageFileNames.at(currentImageNumber);
			QImage image(fname);
			if(!image.isNull())
			{
				editImageLabel->pixmap = new QPixmap(fname);
                                //for save before selecting
                                //editImageLabel->saveSelectPix = new QPixmap(fname);
				editImageLabel->saveSelectPix = NULL;
				editImageLabel->saveToHistory();
				editImageLabel->setPixmap(*(editImageLabel->pixmap));
				editImageLabel->adjustSize();
				//editImageLabel->setImage(imageFileNames.at(currentImageNumber));
				widgetSet->setCurrentIndex(1);
				//this->addToolBar(editToolBar);
				editScaleFactor = 1.0;
				this->removeToolBar(mainToolBar);
				initEditToolBar();
                                act22->setDisabled(0);
			}
		}
		else
		{
			Error(tr("Load images first"));
		}
	}
	else
	{
		widgetSet->setCurrentIndex(0);
		this->removeToolBar(editToolBar);
		this->removeToolBar(editToolBar2);
		initToolBar();
                act22->setDisabled(1);
	}
	emit update();
}

void Viewer::zoomIn()
{
	ScaleImage(1.25);
	emit update();
}

void Viewer::zoomOut()
{
	ScaleImage(0.8);
}

void Viewer::fitToScreen()
{
	//scrollArea->setWidgetResizable(false);
	//imageLabel->adjustSize();
        //scaleFactor = 1.0;
	QSize size = widgetSet->size();
	QSize size1 = this->size();
	QSize size2 = scrollArea->size();
	//imageLabel->resize(size);
}
void Viewer::drawPen()
{
	editImageLabel->drawPenSelect();
        if(actPen->isChecked())
            actPen->setChecked(0);
        else
            actPen->setChecked(1);
}
void Viewer::drawCrop()
{
        editImageLabel->drawCropSelect(); 
}
void Viewer::drawSelect()
{
	editImageLabel->drawRectSelect();
}
void Viewer::chooseColor()
{
	QColor c = QColorDialog::getColor(Qt::white, this );
	editImageLabel->editChooseColor(c);
}
void Viewer::currentIndexChanged(int index)
{
	
	editImageLabel->editChangeSizeBrush(index);
}
void Viewer::SaveImage()
{
	if(widgetSet->currentIndex() == 1)
	{
		editImageLabel->removeSelect();
		QString fileName = QFileDialog::getSaveFileName(this, tr("Save File"), "untitled.jpg", tr("Images (*.png *.xpm *.jpg)"));
		if(!fileName.isEmpty())
		{	
			if(editImageLabel->pixmap->save(fileName))
				{
					showMessage(tr("Saved"));
				}
				else
				{
					showMessage(tr("failed to save"));
				}
			
		}
		
		emit update();
	}
	else
	{
		showMessage(tr("Saving image works only in editing mode"));
	}
}


void Viewer::LoadImages()
{
	QStringList fileNames = QFileDialog::getOpenFileNames(this, tr("Select one or more files to open"), "/home", tr("Images (*.png *.jpg)"));
	if(!fileNames.empty())
	{
		imageFileNames = fileNames;
		showViewer(0);
		emit update();
	}
}
void Viewer::About()
{
	
	QMessageBox mb(this);
	mb.setIcon(QMessageBox::Information);
	mb.setText(tr("KolibriLook - application for viewing images and simple operations with its. <br /> Version: 0.3"));
	mb.exec();
}
void Viewer::Resize()
{
    if(widgetSet->currentIndex() == 1)
	{
		editImageLabel->drawResize();
	}
	else
	{
		showMessage(tr("Resizing image works only in editing mode"));
	}
}
