/*
* Copyright (C) 2010  Koos Vriezen <koos.vriezen@gmail.com>
*
* This library is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 2 of the License, or (at your option) any later version.
*
* This library is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this library; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef _KMPLAYER_STRING_H_
#define _KMPLAYER_STRING_H_

#include <unistd.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#include <glib.h>

#undef KDE_NO_CDTOR_EXPORT
#undef KDE_NO_EXPORT
#if __GNUC__ - 0 > 3 || (__GNUC__ - 0 == 3 && __GNUC_MINOR__ - 0 > 3)
  #define KMPLAYER_NO_CDTOR_EXPORT __attribute__ ((visibility("hidden")))
  #define KMPLAYER_NO_EXPORT __attribute__ ((visibility("hidden")))
  #define KMPLAYER_EXPORT __attribute__ ((visibility("default")))
#elif __GNUC__ - 0 > 3 || (__GNUC__ - 0 == 3 && __GNUC_MINOR__ - 0 > 2)
  #define KMPLAYER_NO_CDTOR_EXPORT
  #define KMPLAYER_NO_EXPORT __attribute__ ((visibility("hidden")))
  #define KMPLAYER_EXPORT
#else
  #define KMPLAYER_NO_CDTOR_EXPORT
  #define KMPLAYER_NO_EXPORT
  #define KMPLAYER_EXPORT
#endif
#ifndef KDE_EXPORT
  #define KDE_EXPORT KMPLAYER_EXPORT
  #define KDE_NO_EXPORT KMPLAYER_NO_EXPORT
  #define KDE_NO_CDTOR_EXPORT KMPLAYER_NO_CDTOR_EXPORT
#endif


#ifndef ASSERT
#define ASSERT g_assert
//#define ASSERT Q_ASSERT
#endif

namespace KMPlayer {

template <class T>
class KMPLAYER_EXPORT CharTmpl {
    T m_unichar;
public:
    CharTmpl ();
    CharTmpl (const T & char_impl);
    CharTmpl (const char ch);
    CharTmpl (const int unichar);
    CharTmpl (const CharTmpl <T> & other);
    ~CharTmpl ();
    bool operator == (const CharTmpl <T> & other) const;
    bool operator == (const T & char_impl) const;
    bool operator == (const char ch) const;
    bool operator != (const CharTmpl <T> & other) const;
    bool operator != (const T & char_impl) const;
    bool operator != (const char ch) const;
    CharTmpl <T> & operator = (const CharTmpl <T> & other);
    CharTmpl <T> & operator = (const T & char_impl);
    CharTmpl <T> & operator = (const char ch);
    operator T ();
    operator T () const;
    operator char () const;
    int unicode () const;
    // these isXX should be specialized
    bool isSpace () const;
    bool isLetterOrNumber () const;
};

typedef CharTmpl <gunichar> Char;


class ByteArrayImpl {
    char * m_data;
    unsigned m_size;
    unsigned m_capacity;
    bool data_owner;
public:
    ByteArrayImpl ();
    //ByteArrayImpl (unsigned sz);
    ByteArrayImpl (const ByteArrayImpl & other);
    ByteArrayImpl (const char * data, unsigned size);
    ~ByteArrayImpl ();
    unsigned size () const;
    void resize (unsigned ns);
    char * data () const;
    bool isEmpty () const;
    ByteArrayImpl & operator = (const ByteArrayImpl & other);
    ByteArrayImpl & operator = (const char *str);
};

typedef ByteArrayImpl ByteArray;


class FixedSizeAllocator {
    void **pool;
    size_t size;
    int count;
public:
    FixedSizeAllocator (size_t s)
        : pool ((void**) malloc (10 * sizeof (void *))), size (s), count (0) {}

    void *alloc ();
    void dealloc (void *p);
};

template <class T>
class Shared {
public:
    ~Shared ();
    void clear ();
    bool isNull () const;
protected:
    T *data;
    Shared ();
    void attach (const Shared<T> &other);
    void detach ();
};

struct SharedChars {
    SharedChars (gchar *p, int len) : ptr (p), ref_count (1), buf_len (len) {}

    // Shared requirements
    SharedChars (const SharedChars &s)
        : ptr(g_strdup (s.ptr)), ref_count (1), buf_len (s.buf_len) {}
    ~SharedChars () { g_free (ptr); }
    void *operator new (size_t);
    void operator delete (void *);
    void ref () { ref_count++; }
    void unref () { if (--ref_count <= 0) delete this; }
    int ref_count;

    gchar *ptr;
    int buf_len; // excluding zero terminator
};

template <> void Shared<SharedChars>::attach (const Shared<SharedChars> &other);

class KMPLAYER_NO_EXPORT GLibString : public Shared <SharedChars> {
public:
    GLibString ();
    GLibString (const GLibString & str);
    GLibString (const char * str, int maxlen = -1);
    GLibString (const Char & str);
    ~GLibString ();
    void append (const char *p, int length);  // p must be valid utf8
    void takeUtf8 (char *p, int buf_len=-1);  // p must be valid utf8
    int compare (const char *str) const;
    int compare (const GLibString & str) const;
    bool operator == (const GLibString &other) const;
    bool operator == (const char * str) const;
    bool operator != (const GLibString &other) const;
    bool operator != (const char * str) const;
    GLibString & operator = (const GLibString & str);
    GLibString & operator = (const char * s);
    GLibString & operator = (const Char & ch);
    GLibString & operator << (const GLibString &other);
    GLibString & operator << (const char * str);
    GLibString operator + (const char * str) const;
    GLibString operator + (const GLibString &other) const;
    GLibString operator + (const Char & ch) const;
    GLibString & operator += (const GLibString & str);
    GLibString & operator += (const char * str);
    GLibString & operator += (const Char & ch);
    Char operator [] (int index) const;
    operator const char * () const;
    bool operator < (const GLibString &other) const;
    void take (char * s);
    GLibString lower () const;
    GLibString upper () const;
    GLibString stripWhiteSpace () const;
    static GLibString fill (Char ch, int length);
    static GLibString number (int nr);
    static GLibString number (double nr);
    static GLibString number (off_t nr);
    bool startsWith (const GLibString & prefix) const;
    bool endsWith (const GLibString & suffix) const;
    int indexOf (const GLibString &needle, int offset=-1) const;
    int lastIndexOf (const GLibString & needle) const;
    GLibString replace (Char c1, Char c2) const;
    bool isEmpty () const;
    GLibString mid (const int index, unsigned length=(unsigned)-1) const;
    GLibString left (int index) const;
    int toInt (bool *valid=0L, int base=10) const;
    double toDouble (bool *valid=0L) const;
    int length () const;
    void truncate (int pos);
};

template < class U>
GLibString operator + (const CharTmpl <U> &c, const GLibString &s) {
    GLibString str (c);
    return str += s;
}

template <class U>
GLibString operator + (const GLibString &s, const CharTmpl <U> &c) {
    GLibString str (s);
    return str += c;
}

typedef GLibString String;

//-----------------------------------------------------------------------------

template <class T> inline CharTmpl<T>::CharTmpl () : m_unichar (0) {}

template <class T>
inline CharTmpl<T>::CharTmpl (const T & char_impl) : m_unichar (char_impl) {}

template <class T>
inline CharTmpl<T>::CharTmpl (const char ch) : m_unichar (ch) {}

template <class T>
inline CharTmpl<T>::CharTmpl (const int unichar) : m_unichar (unichar) {}

template <class T>
inline CharTmpl<T>::CharTmpl (const CharTmpl <T> & other) 
    :  m_unichar (other.m_unichar) {}

template <class T> inline CharTmpl<T>::~CharTmpl () {}

template <class T> inline bool CharTmpl<T>::operator == (const CharTmpl <T> & other) const {
    return m_unichar == other.m_unichar;
}

template <class T> inline bool CharTmpl<T>::operator == (const T & char_impl) const {
    return m_unichar == char_impl;
}

template <class T> inline bool CharTmpl<T>::operator == (const char ch) const {
    return m_unichar == ch;
}

template <class T> inline bool CharTmpl<T>::operator != (const CharTmpl <T> & other) const {
    return m_unichar != other.m_unichar;
}

template <class T> inline bool CharTmpl<T>::operator != (const T & char_impl) const {
    return m_unichar != char_impl;
}

template <class T> inline  bool CharTmpl<T>::operator != (const char ch) const {
    return m_unichar != ch;
}

template <class T>
inline CharTmpl <T> & CharTmpl<T>::operator = (const CharTmpl <T> & other) {
    m_unichar = other.m_unichar;
    return *this;
}

template <class T>
inline CharTmpl <T> & CharTmpl<T>::operator = (const T & char_impl) {
    m_unichar = char_impl;
    return *this;
}

template <class T>
inline CharTmpl <T> & CharTmpl<T>::operator = (const char ch) {
    m_unichar = ch;
    return *this;
}

template <class T> inline CharTmpl<T>::operator T () { return m_unichar; }

template <class T> inline CharTmpl<T>::operator T () const { return m_unichar; }

template <class T> inline CharTmpl<T>::operator char () const {
    gchar buf [8];
    int nr = g_unichar_to_utf8 (m_unichar, buf);
    return (char) buf [nr-1];
}

template <class T> inline int CharTmpl<T>::unicode () const {
    return (int) m_unichar;
}

template <class T> inline bool CharTmpl<T>::isSpace () const {
    return false;
}

template <class T> inline bool CharTmpl<T>::isLetterOrNumber () const {
    return false;
}

//-----------------------------------------------------------------------------

template <>
inline bool CharTmpl<gunichar>::isSpace () const {
    return g_unichar_isspace (m_unichar);
}

template <>
inline bool CharTmpl<gunichar>::isLetterOrNumber () const {
    return g_unichar_isalnum (m_unichar);
}

template <>
inline CharTmpl <gunichar> & CharTmpl<gunichar>::operator = (const char ch) {
    char buf [] = { ch, 0 };
    const gchar *end;
    if (!g_utf8_validate (buf, -1, &end)) {
        gsize bytes_read, bytes_written;
        gchar *utf8 =g_locale_to_utf8(buf, -1, &bytes_read, &bytes_written, 0L);
        m_unichar = g_utf8_get_char (utf8);
        g_free (utf8);
    } else
        m_unichar = g_utf8_get_char (buf);
    return *this;
}

template <>
inline CharTmpl<gunichar>::CharTmpl (const char ch) {
    operator = (ch);
}

template <>
inline  bool CharTmpl<gunichar>::operator == (const char ch) const {
    return m_unichar == Char (ch). m_unichar;
}

template <>
inline  bool CharTmpl<gunichar>::operator != (const char ch) const {
    return m_unichar != Char (ch).m_unichar;
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline ByteArrayImpl::ByteArrayImpl ()
 : m_data (0L), m_size (0), m_capacity (0), data_owner (true) {}

/*KMPLAYER_NO_CDTOR_EXPORT inline ByteArrayImpl::ByteArrayImpl (unsigned sz)
 : m_data ((char *) malloc (sz + 1)),
   m_size (sz),
   m_capacity (sz),
   data_owner (true) { m_data[sz] = 0; }*/

inline ByteArrayImpl::ByteArrayImpl (const char * d, unsigned sz)
 : m_data ((char *)d), m_size (sz), m_capacity (sz), data_owner (false) {}

KMPLAYER_NO_CDTOR_EXPORT inline ByteArrayImpl::~ByteArrayImpl () {
    if (m_data && data_owner)
        free (m_data);
}

KMPLAYER_NO_EXPORT inline unsigned ByteArrayImpl::size () const {
    return m_size;
}

KMPLAYER_NO_EXPORT inline char * ByteArrayImpl::data () const {
    return m_data;
}

KMPLAYER_NO_EXPORT inline bool ByteArrayImpl::isEmpty () const {
    return (!m_data || m_size ==0);
}

//-----------------------------------------------------------------------------

template <class T>
inline Shared<T>::Shared () : data (0) {}

template <class T>
inline Shared<T>::~Shared () {
    clear ();
}

template <class T>
inline bool Shared<T>::isNull () const {
    return !data;
}

template <class T>
inline void Shared<T>::clear () {
    if (data)
        data->unref ();
    data = 0L;
}

template <class T>
inline void Shared<T>::detach () {
    if (data && data->ref_count > 1) {
        T *tmp = new T (data);
        if (data)
            data->unref ();
        data = tmp;
    }
}

//-----------------------------------------------------------------------------

KMPLAYER_NO_CDTOR_EXPORT inline GLibString::GLibString () {}

KMPLAYER_NO_CDTOR_EXPORT inline GLibString::GLibString (const GLibString &s) {
    attach (s);
}

KMPLAYER_NO_CDTOR_EXPORT
inline GLibString::GLibString (const char * str, int maxlen) {
    if (str)
        take (maxlen > -1 ? g_strndup (str, maxlen) : g_strdup (str));
}

KMPLAYER_NO_EXPORT inline GLibString::~GLibString () {}

KMPLAYER_NO_EXPORT
inline GLibString & GLibString::operator = (const GLibString &str) {
    attach (str);
    return *this;
}

KMPLAYER_NO_EXPORT inline GLibString::operator const char * () const {
    return data ? data->ptr : "";
}

KMPLAYER_NO_EXPORT inline Char GLibString::operator [] (int index) const {
    // unchecked pointer, TODO add a modifying operator []
    gchar * off = g_utf8_offset_to_pointer (data->ptr, index);
    return Char (g_utf8_get_char (off));
}

KMPLAYER_NO_EXPORT inline GLibString GLibString::number (int nr) {
    GLibString str;
    str.takeUtf8 (g_strdup_printf ("%d", nr));
    return str;
}

KMPLAYER_NO_EXPORT inline GLibString GLibString::number (double nr) {
    GLibString str;
    str.takeUtf8 (g_strdup_printf ("%f", nr));
    return str;
}

KMPLAYER_NO_EXPORT inline GLibString GLibString::number (off_t nr) {
    GLibString str;
    str.takeUtf8 (g_strdup_printf ("%lu", nr));
    return str;
}

KMPLAYER_NO_EXPORT inline bool GLibString::isEmpty () const {
    return !data || !data->ptr [0];
}

KMPLAYER_NO_EXPORT inline int GLibString::length () const {
    return data ? g_utf8_strlen (data->ptr, -1) : 0;
}

KMPLAYER_NO_EXPORT
inline bool GLibString::operator == (const GLibString &other) const {
    return !compare (other);
}

KMPLAYER_NO_EXPORT inline bool GLibString::operator == (const char *str) const {
    return !compare (str);
}

KMPLAYER_NO_EXPORT
inline bool GLibString::operator != (const GLibString &other) const {
    return compare (other);
}

KMPLAYER_NO_EXPORT inline bool GLibString::operator != (const char *str) const {
    return compare (str);
}

KMPLAYER_NO_EXPORT
inline GLibString GLibString::operator + (const GLibString &other) const {
    GLibString str (*this);
    return str += other;
}

KMPLAYER_NO_EXPORT
inline GLibString GLibString::operator + (const char *str) const {
    GLibString str1 (*this);
    return str1 += str;
}

KMPLAYER_NO_EXPORT
inline GLibString GLibString::operator + (const Char & ch) const {
    GLibString str (*this);
    return str += ch;
}


KMPLAYER_NO_EXPORT
inline GLibString & GLibString::operator << (const GLibString &other) {
    return *this += other;
}

KMPLAYER_NO_EXPORT
inline GLibString & GLibString::operator << (const char *str) {
    return *this += str;
}

KMPLAYER_NO_EXPORT
inline bool GLibString::operator < (const GLibString &other) const {
    return compare (other) < 0;
}

KMPLAYER_NO_EXPORT inline GLibString GLibString::lower () const {
    GLibString str;
    if (data)
        str.take (g_utf8_strdown (data->ptr, -1));
    return str;
}

KMPLAYER_NO_EXPORT inline GLibString GLibString::upper () const {
    GLibString str;
    if (data)
        str.take (g_utf8_strup (data->ptr, -1));
    return str;
}

} // namespace KMPlayer

#endif

