#include "kimi.h"
#include "settings.h"
#include <stdlib.h>
#include <string.h>

void kimi_conf_initialize(Kimi* kimi, const char* dir)
{
    char buf[BUF_SIZE];
    sprintf(buf, "%s/.kimi", dir);
    g_mkdir(buf, 0700);
    char* arr[2];
    arr[0] = buf;
    arr[1] = 0;
    
    kimi->config_modules = g_ptr_array_new();
    kimi->config = g_key_file_new();
   
    GError* err = NULL;

    gboolean boo = g_key_file_load_from_dirs(kimi->config, "kimiconfig", arr, &kimi->config_file, G_KEY_FILE_NONE, &err);
    
    if (err) {
        kimi->config_file = g_strdup_printf("%s/kimiconfig", buf);
        g_debug("%s", err->message);
        g_clear_error(&err);
    }
}

void kimi_conf_deinitialize(Kimi* kimi)
{
    int i;

    /* Save config */
    unsigned int length;
    char* config_data = g_key_file_to_data(kimi->config, &length, NULL);
    if (length != 0) {
        g_debug("%s", kimi->config_file);
        FILE* file = fopen(kimi->config_file, "w");
        if (file) {
            fwrite(config_data, length, 1, file);
            fclose(file);
            free(config_data);
            return;
        }
    }


    GPtrArray* config_modules = kimi->config_modules;
    
    /* For each config module free all options */
    for (i = 0; i < config_modules->len; i++) {
        ConfigModule* cm = g_ptr_array_index(config_modules, i);
        
        if (cm->name)
            free(cm->name);
        
        GHashTable* options = cm->options;
        
        GHashTableIter iter;
        g_hash_table_iter_init(&iter, options);
        gpointer iter_id;
        gpointer iter_val;
        
        while (g_hash_table_iter_next(&iter, &iter_id, &iter_val)) {
            Option* opt = (Option*) iter_val;
            free(opt->id);
            free(opt->string);
            /* If this option is string, free it */
            if (opt->type == OT_PASSWORD || opt->type == OT_STRING)
                free(opt->value.v_text);
        }
        g_hash_table_destroy(options);        
    }
    
    if (kimi->config_file)
        free(kimi->config_file);

    if (kimi->config)
        g_key_file_free(kimi->config);

    g_ptr_array_free(config_modules, TRUE);
}

ConfigModule* kimi_conf_create_config_module(Kimi* kimi, const char* id, const char* name, GError** error)
{
    ConfigModule* cm = calloc(1, sizeof(ConfigModule));
    CHECK_ALLOC(cm, error);
    cm->kimi = kimi;

    cm->id = strdup(id);
    CHECK_ALLOC(cm->id, error);

    cm->name = strdup(name);
    CHECK_ALLOC(cm->name, error, free(cm), free(cm->id));
    
    cm->options = g_hash_table_new_full(g_str_hash, g_str_equal, NULL, NULL);
    CHECK_ALLOC(cm->options, error, free(cm), free(cm->name));
    
    g_ptr_array_add(kimi->config_modules, cm);
    
    g_message("New config module \"%s\" registered", name);
    return cm;
}

Option* kimi_conf_register_option(ConfigModule* cm,
                              const char* id,
                              const char* str,
                              OptionType type,
                              int flags,
                              GError** error)
{
    if (g_hash_table_lookup(cm->options, id)) {
        g_warning("Config module %s register option %s twice", cm->name, id);
        return NULL;
    }
    
    Option* opt = calloc(1, sizeof(Option));
    CHECK_ALLOC(opt, error);
    /* Set fields of option */ 
    opt->type = type;
    opt->value.state = OS_UNSET;
    opt->string = strdup(str);
    opt->flags = flags;
    opt->cm = cm;   
    CHECK_ALLOC(opt->string, error, free(opt));
    
    char* id_dup = strdup(id);
    CHECK_ALLOC(id_dup, error, free(opt), free(opt->string));
    opt->id = id_dup;

    g_hash_table_insert(cm->options, id_dup, opt);
    g_message("Module \"%s\" have registered new option \"%s\"", cm->name, id);
    
    /* Load option of it is in config file*/
    if (opt->flags & OF_STORABLE) {
        GError* err = NULL;
        
        switch (opt->type) {
        case OT_STRING:
        case OT_PASSWORD:
            opt->value.v_text = g_key_file_get_string(cm->kimi->config, cm->id, id, &err);
            break;
        case OT_INTEGER:
            opt->value.v_int = g_key_file_get_integer(cm->kimi->config, cm->id, id, &err);
            break;
        case OT_BOOLEAN:
            opt->value.v_boolean = g_key_file_get_boolean(cm->kimi->config, cm->id, id, &err);
            break;
        }

        if (err) {
            opt->value.state = OS_UNSET;
            g_clear_error(&err);
        } else {
            opt->value.state = OS_SET;
        }
    }
    
    return opt;
}

static void kimi_conf_store_option(ConfigModule* cm, Option* opt) {
    if (opt->flags & OF_STORABLE) {
        switch (opt->type) {
        case OT_STRING:
        case OT_PASSWORD:
            g_key_file_set_string(cm->kimi->config, cm->id, opt->id, opt->value.v_text);
            break;
        case OT_INTEGER:
            g_key_file_set_integer(cm->kimi->config, cm->id, opt->id, opt->value.v_int);
            break;
        case OT_BOOLEAN:
            g_key_file_set_boolean(cm->kimi->config, cm->id, opt->id, opt->value.v_boolean);
        }
    }
}

int kimi_conf_set_option(ConfigModule* cm, const char* id, OptionValue optv, GError** error)
{
    Option* opt;
    if (!(opt = g_hash_table_lookup(cm->options, id))) {
        g_warning("Config module \"%s\" sets option \"%s\", that doesn't exist", cm->name, id);
        return 0;
    }
    
    /* Free last value if option has text type */
    if ((opt->type == OT_PASSWORD || opt->type == OT_STRING) && opt->value.v_text != NULL) {
        free(opt->value.v_text);
    }
    
    opt->value = optv;
    opt->value.state = OS_SET;

    kimi_conf_store_option(cm, opt);
    return 0;
}

Option* kimi_conf_get_option(ConfigModule* cm, const char* id, GError** error)
{
    Option* opt;
    if (!(opt = g_hash_table_lookup(cm->options, id))) {
        g_warning("Config module %s gets option %s, that doesn't exist", cm->name, id);
        g_set_error(error, KIMI_CONF_ERROR, KIMI_CONF_ERROR_NO_SUCH_OPTION, "Option %s doesn't exist", id);
        
        return NULL;
    }

    return opt;
}

int kimi_conf_show_all_options_to_user(ConfigModule* cm, const char* message, GError** error)
{
    GPtrArray* options = g_ptr_array_new();
    
    GHashTableIter iter;
    g_hash_table_iter_init(&iter, cm->options);
    gpointer key, value;
    
    while (g_hash_table_iter_next(&iter, &key, &value)) {
        Option* opt = (Option*) value;
        if (opt->flags & OF_DISPLAYABLE)
            g_ptr_array_add(options, value);
    }
    
    kimi_conf_show_all_options_to_user_array(cm, message, options, error);
    
    return 0;
}

int kimi_conf_show_all_options_to_user_array(ConfigModule* cm, const char* message, GPtrArray* options, GError** error)
{
    cm->kimi->ui_callbacks.show_options(cm, cm->name, message, options);
    /*for (i = 0; i < options->len; i++) {
        Option* opt = g_ptr_array_index(options, i);
        if (opt->value.state == OS_SET)
            kimi_conf_store_option(cm, opt);        
    }
*/
    return 0;
}

OptionState kimi_conf_option_get_state(Option* opt)
{
    return opt ? opt->value.state : OS_INVALID;
}

int kimi_conf_option_get_int(Option* opt)
{   
    if (!opt) {
        g_warning("kimi_conf_option_get called with NULL parameter");
        return 0;
    }
    return opt->value.v_int;
}

const char* kimi_conf_option_get_string(Option* opt)
{
    if (!opt) {
        g_warning("kimi_conf_option_get called with NULL parameter");
        return NULL;
    }   

    return opt->value.v_text;
}

int kimi_conf_option_get_boolean(Option* opt)
{
    if (!opt) {
        g_warning("kimi_conf_option_get called with NULL parameter");
        return 0;
    }   

    return opt->value.v_boolean;
}


void kimi_conf_option_sel_init(Option* opt)
{
    int i;
    if (opt->value.v_sel_list == NULL) {
        opt->value.v_sel_list = g_ptr_array_new();
    }

    GPtrArray* list = opt->value.v_sel_list;
    
    for (i = 0; i < list->len; i++) {
        free(list->pdata[i]);
    }
}

void kimi_conf_option_sel_add(Option* opt, const char* string)
{
    g_ptr_array_add(opt->value.v_sel_list, strdup(string));
}



void kimi_conf_option_set_int(Option* opt, int val)
{
    if (!opt) {  
        g_warning("kimi_conf_option_set called with NULL parameter");
        return;           
    }                           

    if (opt->type != OT_INTEGER) {
        g_warning("Setting integer value to non-integer option");
    }
    opt->value.state = OS_SET;
    opt->value.v_int = val;
    kimi_conf_store_option(opt->cm, opt);
}

void kimi_conf_option_set_string(Option* opt, const char* val)
{
    if (!opt) {   
        g_warning("kimi_conf_option_set called with NULL parameter");
        return;           
    }                           

    if (opt->type != OT_STRING && opt->type != OT_PASSWORD)
        g_warning("Setting string value to non-string option");
    else if (opt->value.v_text) {
        free(opt->value.v_text);
    }
    opt->value.state = OS_SET;
    opt->value.v_text = strdup(val);
    
    kimi_conf_store_option(opt->cm, opt);
}

void kimi_conf_option_set_boolean(Option* opt, int val)
{
    if (!opt) {   
        g_warning("kimi_conf_option_set called with NULL parameter");
        return;           
    }                           

    if (opt->type != OT_BOOLEAN) {
        g_warning("Setting boolean value to non-boolean option");
    }

    opt->value.state = OS_SET;
    opt->value.v_boolean = val;
    kimi_conf_store_option(opt->cm, opt);
}

const GPtrArray* kimi_conf_get_config_modules(Kimi* kimi)
{
    return kimi->config_modules;
}

GPtrArray* kimi_conf_get_options(ConfigModule* cm)
{
    GPtrArray* options = g_ptr_array_new();

    GHashTableIter iter;
    g_hash_table_iter_init(&iter, cm->options);
    gpointer key, value;

    while (g_hash_table_iter_next(&iter, &key, &value)) {
        Option* opt = (Option*) value;
        if (opt->flags & OF_DISPLAYABLE)
            g_ptr_array_add(options, value);
    }

    return options;
}


