#ifndef PERIOD_H
#define PERIOD_H

#include <time.h>
#include <glib.h>
#include <libical/ical.h>

//#define GLIB_MODE

/**
 * Recurrence types
 */
typedef enum 
{
    REC_SECOND,
    REC_MINUTE,
    REC_HOUR,
    REC_DAY,
    REC_WEEK,
    REC_MONTH,
    REC_YEAR,
    REC_NO
} PerType;

/**
 * Struct that describes recurrence rule
 */
typedef struct {
    int type; /**< Type of rule (0 - Recurrence rule, 1 - exception rule (not used now)) */
    //int cb_rul_freed; /**< Not used */
    
    //char* ical;
    
    int count; /**< How many instances of event will be generated */
    PerType freq; /**< Frequency unit of event */
    time_t until; /**< Time to which the event will be occur */
    short interval; /**< How often event will be occu */
#ifdef GLIB_MODE
    GArray* seconds; /**< BYDAY array */
    GArray* minutes;
    GArray* hours;
    GArray* week_days;
    GArray* month_days;
    GArray* year_days;
    GArray* week_number;
    GArray* months;
    GArray* set_pos;
#else
    short* seconds; /**< BYDAY array */
    short* minutes;
    short* hours;
    short* week_days;
    short* month_days;
    short* year_days;
    short* week_number;
    short* months;
    short* set_pos;
#endif
    short wkst; /**< Week start */
} Rule;

/**
 * Structure that describes period. Period is array of rules 
 */
typedef GPtrArray Period;

/**
 * @brief Create rule from iCalendar string
 * @param ical Rule in ical format
 * @param type Type of rule (RRule(0), ERule(1))
 * @param wkst Week start day (0 = 7 = sunday, 1 = monday)
 * @return Rule
 */
Rule* kimi_per_create_rule_from_ical(const char* ical, int type, int wkst);

/**
 * @brief Create empty recurrence rules
 * @param freq Frequency unit (daily, monthly, etc.)
 * @param wkst Week start day (0 = 7 = sunday, 1 = monday)
 * @return Rule
 */
Rule* kimi_per_create_rule(int type, PerType freq, int wkst);

/**
 * @brief Convert rule to iCalendar string
 * @param rul Rule to convert
 * @return Ical string
 */
const char* kimi_per_get_ical(Rule* rul);

/**
 * @brief Fill rule fields from iCalendar string
 * @param rul Rule
 * @param ical iCalendar string
 */
void kimi_per_set_ical(Rule* rul, const char* ical);

/**
 * @brief Free rule
 * @param rule Rule to free
 */
void kimi_per_free_rule(Rule* rul);

/**
 * @param Clean fields of rule, but don't free struct
 * @param rule Rule to clean
 */
void kimi_per_clear_rule(Rule* rul);

/**
 * @brief Create new period
 */
#define kimi_per_create_period() g_ptr_array_new()

/**
 * @brief Add rule to period
 */
#define kimi_per_add_rule(p, r) g_ptr_array_add(p, r)

Period* kimi_per_dup(Period* per);

/**
 * @brief Free period
 * This function frees all rules of period, then frees period itself
 * @param per Period to free
 */
void kimi_per_free_period(Period* per);

/**
 * @brief Merge two arrays of iCalendar recurrence rules
 * dest = dest OR src (as sets)
 * @param dest
 * @param src
 */
void kimi_per_merge_arrays(GArray* dest, GArray* src);

/**
 * @brief Generate times when rule occurs
 * @param rule Rule
 * @param begin begin of view interval
 * @param end begin of view interval
 * @param event_time Time when event occurs itself
 * @return GArray of time_ts
 */
GArray* kimi_per_generate_rule_instance_times(Rule* rul, time_t begin, time_t end, 
                                       time_t event_time);

/**
 * @brief Generate times when period occurs
 * @param per Period
 * @param begin begin of view interval
 * @param end begin of view interval
 * @param event_time Time when event occurs itself
 * @return GArray of time_ts
 */
GArray* kimi_per_generate_instance_times(Period* per, time_t begin, time_t end, 
                                       time_t event_time);

/**
 * @brief Convert kimi Rule to icalrecurrencetype
 * @param libical_rule icalrecurrencetype
 * @param kimi_rule Rule
 */
void kimi_per_convert_kimi_to_libical_rule(struct icalrecurrencetype* libical_rule,
                                           Rule* kimi_rule);
/**
 * @brief Convert icalrecurrencetype to kimi Rule
 * @param kimi_rule Rule
 * @param libical_rule icalrecurrencetype
 */
void kimi_per_convert_libical_to_kimi_rule(Rule* kimi_rule,
                                           struct icalrecurrencetype* libical_rule);
/**
 * @brief Convert time in ical format to time_t
 * @param str time in ical format
 * @return result of conversion
 */
time_t kimi_per_get_time_t(const char* str);
#endif /* PERIOD_H */

