/*
 * chnl_sm.h
 *
 * DSP-BIOS Bridge driver support functions for TI OMAP processors.
 *
 * Copyright (C) 2005-2006 Texas Instruments, Inc.
 *
 * This package is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * THIS PACKAGE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED
 * WARRANTIES OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */


/*
 *  ======== chnl_sm.h ========
 *  Description:
 *      Prototypes for channel lower edge functions for a WinBRIDGE mini driver
 *      implementing data transfer via shared memory.
 *
 *  Public Functions:
 *      CHNLSM_DisableInterrupt;
 *      CHNLSM_EnableInterrupt;
 *      CHNLSM_ISR;
 *      CHNLSM_Read;
 *      CHNLSM_UpdateSHMLength;
 *      CHNLSM_Write;
 *
 *  Notes:
 *      These lower edge functions must be implemented by the WMD writer.
 *      Currently, CHNLSM_Read() and CHNLSM_Write() are not called, but must
 *      be defined to link.
 *
 */

#ifndef CHNLSM_
#define CHNLSM_

#include <dspbridge/wmd.h>

/*
 *  ======== CHNLSM_DisableInterrupt ========
 *  Purpose:
 *      Disable interrupts from the DSP board to the PC.
 *  Parameters:
 *      hDevContext:    Handle to mini-driver defined device info.
 *  Returns:
 *  Requires:
 *  Ensures:
 */
       extern DSP_STATUS CHNLSM_DisableInterrupt(struct WMD_DEV_CONTEXT*
							hDevContext);

/*
 *  ======== CHNLSM_EnableInterrupt ========
 *  Purpose:
 *      Enable interrupts from the DSP board to the PC.
 *  Parameters:
 *      hDevContext:    Handle to mini-driver defined device info.
 *  Returns:
 *  Requires:
 *  Ensures:
 */
       extern DSP_STATUS CHNLSM_EnableInterrupt(struct WMD_DEV_CONTEXT*
						       hDevContext);

/*
 *  ======== CHNLSM_InterruptDSP2 ========
 *  Purpose:
 *      Set interrupt value & send an interrupt to the DSP processor(s).
 *      This is typicaly used when mailbox interrupt mechanisms allow data
 *      to be associated with interrupt such as for OMAP's CMD/DATA regs.
 *  Parameters:
 *      hDevContext:    Handle to mini-driver defined device info.
 *      wMbVal:         Value associated with interrupt(e.g. mailbox value).
 *  Returns:
 *      DSP_SOK:        Interrupt sent;
 *      else:           Unable to send interrupt.
 *  Requires:
 *  Ensures:
 */
       extern DSP_STATUS CHNLSM_InterruptDSP2(struct WMD_DEV_CONTEXT*
						     hDevContext, u16 wMbVal);

/*
 *  ======== CHNLSM_ISR ========
 *  Purpose:
 *      Mini-driver's ISR, called by WCD when the board interrupts the host.
 *  Parameters:
 *      hDevContext:    Handle to the mini-driver defined device info.
 *      pfSchedDPC:     Set to TRUE to schedule a deferred procedure call
 *                      to advance the channel protocol.  The channel class
 *                      library will call the WMD's CHNLSM_DPC routine during
 *                      its own DPC, before dispatching I/O.
 *                      The channel class library should ignore *pfSchedDPC when
 *                      CHNLSM_ISR returns FALSE.
 *      pwMBRegVal:     Value of mailbox register.
 *  Returns:
 *      TRUE if this interrupt is was generated by the DSP board.
 *      FALSE otherwise.
 *  Requires:
 *      Interrupts to the host processor are disabled on entry.
 *      Must only call functions which are in page locked memory.
 *      Must only call asynchronous OS services.
 *      The EOI for this interrupt has already been sent to the PIC.
 *  Ensures:
 *      If the interrupt is *not* shared, this routine must return TRUE.
 */
       extern bool CHNLSM_ISR(struct WMD_DEV_CONTEXT *hDevContext,
				     OUT bool *pfSchedDPC,
				     OUT u16 *pwIntrVal);

/*
 *  ======== CHNLSM_Read ========
 *  Purpose:
 *      Read data from DSP board memory into a Host buffer.
 *  Parameters:
 *      hDevContext:    Handle to mini-driver defined device info.
 *      pHostBuf:       Pointer to host buffer (Destination).
 *      dwDSPAddr:      Address on DSP board (Source).
 *      ulNumBytes:     Number of bytes to transfer.
 *  Returns:
 *  Requires:
 *  Ensures:
 */
       extern DSP_STATUS CHNLSM_Read(struct WMD_DEV_CONTEXT *hDevContext,
					    OUT u8 *pHostBuf,
					    u32 dwDSPAddr, u32 ulNumBytes);

/*
 *  ======== CHNLSM_UpdateSHMLength ========
 *  Purpose:
 *      Allow the minidriver a chance to override the SHM length as reported
 *      to the mini driver (chnl_sm.lib) by Windows Plug and Play.
 *  Parameters:
 *      hDevContext:    Handle to mini-driver defined device info.
 *      pSHMLength:     Pointer to size of SHM window (in DSP words).
 *  Returns:
 *      TRUE if pSHMLength updated; FALSE otherwise.
 *  Requires:
 *      pSHMLength != NULL.
 *  Ensures:
 *      No more than sizeof(u32) bytes written to *pSHMLength
 */
	extern bool CHNLSM_UpdateSHMLength(struct WMD_DEV_CONTEXT *hDevContext,
					   IN OUT u32 *pSHMLength);

/*
 *  ======== CHNLSM_Write ========
 *  Purpose:
 *      Write data from a Host buffer to DSP board memory.
 *  Parameters:
 *      hDevContext:    Handle to mini-driver defined device info.
 *      pHostBuf:       Pointer to host buffer (Source).
 *      dwDSPAddr:      Address on DSP board (Destination).
 *      ulNumBytes:     Number of bytes to transfer.
 *  Returns:
 *  Requires:
 *  Ensures:
 */
       extern DSP_STATUS CHNLSM_Write(struct WMD_DEV_CONTEXT *hDevContext,
					    IN u8 *pHostBuf,
					    u32 dwDSPAddr, u32 ulNumBytes);

#endif				/* CHNLSM_ */
