/*
 * linux/arch/arm/mach-omap2/usb-musb.c
 *
 * This file will contain the board specific details for the
 * MENTOR USB OTG controller on OMAP3430
 *
 * Copyright (C) 2007-2008 Texas Instruments
 * Copyright (C) 2008 Nokia Corporation
 * Author: Vikram Pandita
 *
 * Generalization by:
 * Felipe Balbi <felipe.balbi@nokia.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#include <linux/types.h>
#include <linux/errno.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/clk.h>
#include <linux/dma-mapping.h>

#include <asm/io.h>

#include <linux/usb/musb.h>

#include <mach/hardware.h>
#include <mach/pm.h>
#include <mach/mux.h>
#include <mach/usb.h>

#define SIDLEMODE	3
#define SMARTIDLE	(2 << SIDLEMODE)
#define AUTOIDLE        (1 << 0)
#define FORCESTDBY	(1 << 0)
#define OTG_SYSCONFIG	(OMAP34XX_HSUSB_OTG_BASE + 0x404)
#define OTG_FORCESTDBY	(OMAP34XX_HSUSB_OTG_BASE + 0x414)

#ifdef CONFIG_USB_MUSB_SOC
static struct resource musb_resources[] = {
	[0] = {
		.start	= cpu_is_omap34xx()
			? OMAP34XX_HSUSB_OTG_BASE
			: OMAP243X_HS_BASE,
		.end	= cpu_is_omap34xx()
			? OMAP34XX_HSUSB_OTG_BASE + SZ_8K - 1
			: OMAP243X_HS_BASE + SZ_8K - 1,
		.flags	= IORESOURCE_MEM,
	},
	[1] = {	/* general IRQ */
		.start	= INT_243X_HS_USB_MC,
		.flags	= IORESOURCE_IRQ,
	},
	[2] = {	/* DMA IRQ */
		.start	= INT_243X_HS_USB_DMA,
		.flags	= IORESOURCE_IRQ,
	},
};

static int clk_on;

static int musb_set_clock(struct clk *clk, int state)
{
	if (state) {
		if (clk_on > 0)
			return -ENODEV;

		clk_enable(clk);
		clk_on = 1;
	} else {
		if (clk_on == 0)
			return -ENODEV;

		clk_disable(clk);
		clk_on = 0;
	}

	return 0;
}

static struct musb_hdrc_eps_bits musb_eps[] = {
	{	"ep1_tx", 10,	},
	{	"ep1_rx", 10,	},
	{	"ep2_tx", 9,	},
	{	"ep2_rx", 9,	},
	{	"ep3_tx", 3,	},
	{	"ep3_rx", 3,	},
	{	"ep4_tx", 3,	},
	{	"ep4_rx", 3,	},
	{	"ep5_tx", 3,	},
	{	"ep5_rx", 3,	},
	{	"ep6_tx", 3,	},
	{	"ep6_rx", 3,	},
	{	"ep7_tx", 3,	},
	{	"ep7_rx", 3,	},
	{	"ep8_tx", 2,	},
	{	"ep8_rx", 2,	},
	{	"ep9_tx", 2,	},
	{	"ep9_rx", 2,	},
	{	"ep10_tx", 2,	},
	{	"ep10_rx", 2,	},
	{	"ep11_tx", 2,	},
	{	"ep11_rx", 2,	},
	{	"ep12_tx", 2,	},
	{	"ep12_rx", 2,	},
	{	"ep13_tx", 2,	},
	{	"ep13_rx", 2,	},
	{	"ep14_tx", 2,	},
	{	"ep14_rx", 2,	},
	{	"ep15_tx", 2,	},
	{	"ep15_rx", 2,	},
};

static struct musb_hdrc_config musb_config = {
	.multipoint	= 1,
	.dyn_fifo	= 1,
	.soft_con	= 1,
	.dma		= 1,
	.num_eps	= 16,
	.dma_channels	= 7,
	.dma_req_chan	= (1 << 0) | (1 << 1) | (1 << 2) | (1 << 3),
	.ram_bits	= 12,
	.eps_bits	= musb_eps,
};

static struct musb_hdrc_platform_data musb_plat = {
#ifdef CONFIG_USB_MUSB_OTG
	.mode		= MUSB_OTG,
#elif defined(CONFIG_USB_MUSB_HDRC_HCD)
	.mode		= MUSB_HOST,
#elif defined(CONFIG_USB_GADGET_MUSB_HDRC)
	.mode		= MUSB_PERIPHERAL,
#endif
	.clock		= cpu_is_omap34xx()
			? "hsotgusb_ick"
			: "usbhs_ick",
	.set_clock	= musb_set_clock,
	.config		= &musb_config,

	/* REVISIT charge pump on TWL4030 can supply up to
	 * 200 mA ... but this value is board-specific, like
	 * "mode", and should be passed to usb_musb_init().
         *
         * Since the power can come from a Y-cable, let the user
         * decide on power constraints and not limit anything here.
	 */
	.power		= 0,			/* use default of 500 mA */
};

static u64 musb_dmamask = DMA_32BIT_MASK;

static struct platform_device musb_device = {
	.name		= "musb_hdrc",
	.id		= -1,
	.dev = {
		.dma_mask		= &musb_dmamask,
		.coherent_dma_mask	= DMA_32BIT_MASK,
		.platform_data		= &musb_plat,
	},
	.num_resources	= ARRAY_SIZE(musb_resources),
	.resource	= musb_resources,
};
#endif


void __init usb_musb_init(struct musb_board_data *board)
{
	musb_plat.board = board;

#ifdef CONFIG_USB_MUSB_SOC
	if (platform_device_register(&musb_device) < 0) {
		printk(KERN_ERR "Unable to register HS-USB (MUSB) device\n");
		return;
	}
#endif
#if !defined(CONFIG_USB) || defined(CONFIG_USB_MODULE)
	/* Force MUSB to standby if not used */
	if (cpu_is_omap34xx()) {
		omap_writel(AUTOIDLE, OTG_SYSCONFIG);
		omap_writel(FORCESTDBY, OTG_FORCESTDBY);
	}
#endif
}

