/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QString>
#include <QtCore/QDebug>
#include <QtGui/QApplication>
#include <QtGui/QDesktopWidget>
#include <QtXml/QDomNode>
#include "widgetscreen.h"
#include "themescreen.h"

WidgetScreen::WidgetScreen(QWidget* parent): QStackedWidget(parent), AbstractScreen(),
currentOrientation_(LANDSCAPE)
{
}

WidgetScreen::~WidgetScreen()
{
    clear();
}

bool WidgetScreen::load(Orientation orientation, QDomNode const& data, Reader* reader)
{
    if(screens_.find(orientation) == screens_.end())
    {
        qDebug() << "Orientation " << orientation << " not set";
        return false;
    }

    bool ret = screens_[orientation]->load(data, reader);

    if(ret)
    {
        loadedScreens_.insert(orientation);

        if(loadedScreens_.size() == 1)
        {
            currentOrientation_ = orientation;
            setCurrentWidget(screens_[orientation]);
        }
    }

    return ret;
}

bool WidgetScreen::load(QDomNode const& data, Reader* reader)
{
    return load(LANDSCAPE, data, reader) && load(PORTRAIT, data, reader);
}

void WidgetScreen::setColor(Orientation orientation, QString const& color)
{
    if(screens_.find(orientation) != screens_.end())
    {
        screens_[orientation]->setColor(color);
    }
}

void WidgetScreen::setColor(QString const& color)
{
    setColor(LANDSCAPE, color);
    setColor(PORTRAIT, color);
}

void WidgetScreen::addScreen(ThemeScreen* screen, Orientation orientation)
{
    if(screens_.find(orientation) != screens_.end())
    {
        removeWidget(screens_[orientation]);
        delete screens_[orientation];
        screens_.remove(orientation);
    }

    screens_[orientation] = screen;
    addWidget(screen);
    connect(screen, SIGNAL(minimizePressed()), this, SIGNAL(minimizePressed()));
    connect(screen, SIGNAL(settingsPressed()), this, SIGNAL(settingsPressed()));
    connect(screen, SIGNAL(closePressed()), this, SIGNAL(closePressed()));
    connect(screen, SIGNAL(clicked()), this, SIGNAL(clicked()));
}

bool WidgetScreen::orientationEnabled(Orientation orientation) const
{
    return screens_.find(orientation) != screens_.end();
}

bool WidgetScreen::orientationLoaded(Orientation orientation) const
{
    return loadedScreens_.find(orientation) != loadedScreens_.end();
}

void WidgetScreen::reArrange()
{
    QRect rect = QApplication::desktop()->screenGeometry();

    Orientation o = LANDSCAPE;

    if(rect.height() > rect.width())
    {
        o = PORTRAIT;
    }

    if(o != currentOrientation_)
    {
        if(screens_.find(o) != screens_.end())
        {
            currentOrientation_ = o;
            setCurrentWidget(screens_[o]);
            screens_[o]->reArrange();

            if(o == LANDSCAPE)
            {
                screens_[o]->forceRepaint();
            }
        }
    }
}

void WidgetScreen::flip()
{
    for(QMap<Orientation, ThemeScreen*>::iterator it = screens_.begin();
        it != screens_.end(); it++)
    {
        it.value()->flip();
    }
}

void WidgetScreen::clear()
{
    for(QMap<Orientation, ThemeScreen*>::iterator it = screens_.begin();
    it != screens_.end(); it++)
    {
        removeWidget(it.value());
        delete it.value();
    }

    screens_.clear();
    loadedScreens_.clear();
}

void WidgetScreen::removeUnloaded()
{
    for(QMap<Orientation, ThemeScreen*>::iterator it = screens_.begin();
    it != screens_.end(); it++)
    {
        if(loadedScreens_.find(it.key()) == loadedScreens_.end())
        {
            removeWidget(it.value());
            delete it.value();
            screens_.remove(it.key());
        }
    }
}
