/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtGui/QApplication>
#include <QtGui/QWidget>
#include <QtGui/QGraphicsItem>
#include <QtCore/QString>
#include <QtCore/QDebug>
#include <QtXml/QDomNode>
#include <QtXml/QDomNodeList>
#include "themescreen.h"
#include "graphicselement.h"
#include "reader.h"
#include "graphicsscene.h"

ThemeScreen::ThemeScreen(QWidget* parent): GraphicsScreen(parent)
{
}

bool ThemeScreen::load(QDomNode const& data, Reader* reader)
{
    QDomNodeList childNodes = data.childNodes();

    if(childNodes.size() == 0)
    {
        error_ = "No elements specified for " + data.nodeName();
        return false;
    }

    for(int i = 0; i < childNodes.size(); i++)
    {
        GraphicsElement* element = GraphicsElement::getElement(childNodes.at(i).nodeName(), reader);

        if(element == 0)
        {
            qDebug() << "Warning: invalid element: " << childNodes.at(i).nodeName();
            continue;
        }

        elements_.push_back(element);

        QDomNodeList options = childNodes.at(i).childNodes();

        for(int i = 0; i < options.size(); i++)
        {
            QString nodeName = options.at(i).nodeName();



            if(nodeName == "effect")
            {
                QDomNode effectName = options.at(i).attributes().namedItem("name");

                if(effectName.isNull())
                {
                    qDebug() << "Missing name for effect";
                }
                else
                {
                    if(!element->setEffect(effectName.toAttr().value()))
                    {
                        qDebug() << "Invalid effect: " << effectName.toAttr().value();
                    }
                    else
                    {
                        QDomNodeList themeOptions = options.at(i).childNodes();

                        for(int i = 0; i < themeOptions.size(); i++)
                        {
                            if(!element->setEffectAttribute(themeOptions.at(i).nodeName(), themeOptions.at(i).toElement().text()))
                            {
                                qDebug() << "Warning: invalid effect option: " << themeOptions.at(i).nodeName();
                            }
                        }

                        element->applyEffect();
                    }
                }
            }
            else
            {
                if(!element->setAttribute(nodeName, options.at(i).toElement().text()))
                {
                    qDebug() << "Warning: invalid option: " << options.at(i).nodeName();
                }
            }
        }

        element->addToScene(getScene());

    }

    forceRepaint();

    return true;
}

void ThemeScreen::removeElements()
{
    for(int i = 0; i < elements_.size(); i++)
    {
        QGraphicsItem* item = elements_.at(i)->getElement();

        if(item)
        {
            getScene()->removeItem(item);
            delete item;
        }

        delete elements_.at(i);
    }

    elements_.clear();
}

void ThemeScreen::update()
{
    for(int i = 0; i < elements_.size(); i++)
    {
        elements_.at(i)->update();
    }
}

void ThemeScreen::setColor(QString const& color)
{
    Q_UNUSED(color);
}
