/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QString>
#include <QtCore/QDebug>
#include <QtCore/QTime>
#include <QtGui/QGraphicsScene>
#include <QtGui/QGraphicsTextItem>
#include "textelement.h"
#include "reader.h"
#include "odometer.h"
#include "graphicsscene.h"

namespace
{
    const GraphicsElement::AttributeDetails ATTRIBUTES[TextElement::ATTRIBUTE_COUNT] =
    {
     {"xpos", true},
     {"ypos", true},
     {"zpos", true},
     {"data", false},
     {"format", false},
     {"width", true},
     {"align", false},
     {"color", false},
     {"size", true},
     {"font", false},
     {"bold", false},
     {"italic", false},
     {"uppercase", false},
     {"letterspacing", true},
     {"timeformat", false},
     {"dateformat", false}
    };

    const QString FIELDS[TextElement::FIELD_COUNT] =  {"TRIP", "TOTAL", "SPEED",
                                                       "MAXSPEED", "AVGSPEED",
                                                       "UNIT", "SPEEDUNIT", "TIME", "DATE"};
}

TextElement::TextElement(Reader* reader): GraphicsElement(reader),
data_(""), format_(""), align_("left"), timeFormat_("hh:mm"), dateFormat_("dd.MM.yyyy"), fontSize_(16), bold_(false), italic_(false),
uppercase_(false), letterSpacing_(0)
{
    element_ = new QGraphicsTextItem();
}

bool TextElement::setAttribute(QString const& name, QString const& value)
{
    int intVal = 0;
    int attrId = -1;

    if((attrId = getAttribute(name, value, ATTRIBUTES, ATTRIBUTE_COUNT, intVal)) != -1)
    {
        Attribute attr = static_cast<Attribute>(attrId);

        bool fontChanged = false;

        switch(attr)
        {
        case XPOS:
            element_->setX(intVal);
            break;
        case YPOS:
            element_->setY(intVal);
            break;
        case ZPOS:
            element_->setZValue(intVal);
            break;
        case DATA:
            data_ = value;
            break;
        case FORMAT:
            format_ = value;
            break;
        case WIDTH:
            element_->setTextWidth(intVal);
            break;
        case ALIGN:
            align_ = value;
            break;
        case COLOR:
            element_->setDefaultTextColor(QColor(value));
            break;
        case SIZE:
            fontSize_ = intVal;
            fontChanged = true;
            break;
        case FONT:
            fontFile_ = value;
            fontChanged = true;
            break;
        case BOLD:
            if(value != "false" && value != "0")
            {
                bold_ = true;
                fontChanged = true;
            }
            break;
        case ITALIC:
            if(value != "false" && value != "0")
            {
                italic_ = true;
                fontChanged = true;
            }
            break;
        case UPPERCASE:
            if(value != "false" && value != "0")
            {
                uppercase_ = true;
            }
            break;
        case LETTERSPACING:
            letterSpacing_ = intVal;
            fontChanged = true;
            break;
        case TIMEFORMAT:
            timeFormat_ = value;
            break;
        case DATEFORMAT:
            dateFormat_ = value;
            break;
        default:
            qDebug() << "Unknown attribute: " << attr;
            return false;
        }

        if(fontChanged)
        {
            return applyFont();
        }

        return true;

    }
    else
    {
        return false;
    }
}

bool TextElement::applyFont()
{
    QFont font;

    if(!fontFile_.isEmpty())
    {
        if(!getFont(fontFile_, font))
        {
            return false;
        }
    }

    font.setPointSize(fontSize_);

    if(bold_)
    {
        font.setBold(true);
    }

    if(italic_)
    {
        font.setItalic(true);
    }

    if(letterSpacing_)
    {
        font.setLetterSpacing(QFont::PercentageSpacing, letterSpacing_);
    }

    element_->setFont(font);

    return true;
}

void TextElement::addToScene(GraphicsScene* scene)
{
    specialFields_.clear();

    for(int i = 0; i < FIELD_COUNT; i++)
    {
        if(data_.indexOf("{" + FIELDS[i] + "}") != -1)
        {
            specialFields_.push_back(static_cast<Field>(i));
        }

    }

    QString replaced = data_;
    replaceSpecialFields(replaced);
    updateHtml(replaced);

    scene->addItem(element_);
}

void TextElement::update()
{
    if(specialFields_.isEmpty())
    {
        return;
    }

    QString replaced = data_;
    replaceSpecialFields(replaced);

    updateHtml(replaced);
}

void TextElement::updateHtml(QString data)
{
    if(uppercase_)
    {
        data = data.toUpper();
    }

    element_->setHtml("<div align='"+align_+"'>"+data+"</div>");
}

void TextElement::replaceSpecialFields(QString& value)
{
    for(int i = 0; i < specialFields_.size(); i++)
    {
        Field f = specialFields_.at(i);
        Odometer* o = &(Odometer::instance());

        switch(f)
        {
        case TRIP:
            replaceValue(value, f, formatString(o->getTrip()));
            break;
        case TOTAL:
            replaceValue(value, f, formatString(o->getTotal()));
            break;
        case SPEED:
            replaceValue(value, f, formatString(o->getLatestFix().speed));
            break;
        case MAXSPEED:
            replaceValue(value, f, formatString(o->getMaxSpeed()));
            break;
        case AVGSPEED:
            replaceValue(value, f, formatString(o->getAverageSpeed()));
            break;
        case UNIT:
            replaceValue(value, f, o->getUnit());
            break;
        case SPEEDUNIT:
            replaceValue(value, f, o->getSpeedUnit());
            break;
        case TIME:
            replaceValue(value, f, QTime::currentTime().toString(timeFormat_));
            break;
        case DATE:
            replaceValue(value, f, QDate::currentDate().toString(dateFormat_));
            break;
        default:
            qDebug() << "Unknown field: " << f;
        }
    }
}

void TextElement::replaceValue(QString& value,
                               TextElement::Field field,
                               QString const& replace)
{
    value = value.replace("{" + FIELDS[field] + "}", replace);
}

QString TextElement::formatString(double val)
{
    QString format = format_;

    if(format.isEmpty())
    {
        format = "%.1lf";
    }

    QString result;
    result.sprintf(format.toLatin1().data(), val);

    return result;
}


QGraphicsItem* TextElement::getElement() const
{
    return element_;
}
