/*
 * This file is part of jSpeed.
 *
 * jSpeed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * jSpeed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with jSpeed.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <QtCore/QDir>
#include <QtCore/QDebug>
#include <QtCore/QString>
#include <QtCore/QStringList>
#include <QtGui/QFileDialog>
#include <QtGui/QDialogButtonBox>
#include <QtGui/QPushButton>
#include <QtGui/QHBoxLayout>
#include <QtGui/QVBoxLayout>
#include <QtGui/QMessageBox>
#include "themeselector.h"
#include "buttonselector.h"
#include "themeloader.h"
#include "settings.h"

ThemeSelector::ThemeSelector(QWidget* parent): QDialog(parent)
{
    setWindowTitle(tr("Select theme"));

    QHBoxLayout* layout = new QHBoxLayout;
    QHBoxLayout* left = new QHBoxLayout;

    QPushButton* saveButton = new QPushButton(tr("Save"));
    connect(saveButton, SIGNAL(clicked(bool)), this, SLOT(saveTheme()));
    QDialogButtonBox* buttons = new QDialogButtonBox;
    buttons->setCenterButtons(false);
    buttons->addButton(saveButton, QDialogButtonBox::AcceptRole);

    selector_ = new ButtonSelector(tr("Theme"), this);
    theme_ = Settings::instance().value("theme", "default").toString();

    QPushButton* loadButton = new QPushButton(tr("Import"));
    connect(loadButton, SIGNAL(clicked(bool)), this, SLOT(loadFromFile()));

    left->addWidget(selector_, Qt::AlignLeft);
    left->addWidget(loadButton);

    layout->addLayout(left, Qt::AlignLeft);
    layout->addWidget(buttons);

    setLayout(layout);

}

void ThemeSelector::saveTheme()
{
    QString theme = selector_->value().toString();

    if(theme == theme_)
    {
        hide();
        return;
    }

    Settings::instance().setValue("theme", theme);
    hide();
    theme_ = theme;
    emit themeChanged();
}

void ThemeSelector::loadFromFile()
{
    QString filename = QFileDialog::getOpenFileName(this, tr("Open file"),
                                                    QDir::home().path(),
                                                    tr("Theme files") + "(*" + ThemeLoader::getThemeSuffix() + ")");

    if(filename.isEmpty())
    {
        return;
    }

    qDebug() << filename;

    QString basename;
    int i = filename.lastIndexOf(QDir::separator());

    if(i == -1)
    {
        basename = filename;
    }
    else
    {
        basename = filename.mid(i + 1);
    }

    QString targetFile = ThemeLoader::getThemeDir() + basename;

    qDebug() << targetFile;

    bool ok = true;

    if(QFile::exists(targetFile))
    {
        ok = confirmCopy(basename);

        if(ok)
        {
            QFile::remove(targetFile);
        }
    }

    if(ok)
    {
        QFile::copy(filename, targetFile);
        QString name;
        QString id;
        getThemeDetails(basename, name, id);
        theme_ = id;
        loadThemes();
    }
}

void ThemeSelector::loadThemes()
{
    selector_->clear();
    selector_->addItem(tr("Default"), "default");
    QDir themeDir(ThemeLoader::getThemeDir());

    if(!themeDir.exists() || !themeDir.isReadable())
    {
        qDebug() << "Warning: theme dir (" + ThemeLoader::getThemeDir() + ") doesn't exist or is read protected";
        return;
    }

    QStringList filters;
    filters << "*" + ThemeLoader::getThemeSuffix();
    themeDir.setNameFilters(filters);
    themeDir.setFilter(QDir::Files);
    QStringList files = themeDir.entryList();

    for(int i = 0; i < files.size(); i++)
    {
        QString name;
        QString id;
        getThemeDetails(files.at(i), name, id);
        selector_->addItem(name, id);

        if(id == theme_)
        {
            selector_->setCurrentIndex(i + 1);
        }
    }

}

void ThemeSelector::getThemeDetails(QString filename,
                                    QString& name,
                                    QString& id)
{
    static QRegExp cleaner(QRegExp::escape(ThemeLoader::getThemeSuffix()) + "$");
    filename = filename.replace(cleaner, "");

    id = filename;
    name = filename.at(0).toUpper() + filename.mid(1);
}

bool ThemeSelector::confirmCopy(QString const& filename)
{
    QMessageBox::StandardButton result =  QMessageBox::question(this, tr("File exists"),
             tr("Theme file %1 already exists in theme directory, overwrite?").arg(filename),
             QMessageBox::Yes | QMessageBox::No);

    return (result == QMessageBox::Yes);
}

void ThemeSelector::setVisible(bool visible)
{
    if(visible)
    {
        loadThemes();
    }

    QDialog::setVisible(visible);
}
