/*
 * jammo-note-button.c
 *
 * This file is part of JamMo.
 *
 * This is box visualizing note. It always need start and stop events (jammo-midi-event).
 * It is draggable and can be dropped to miditrack-view.
 * (c) 2010 University of Oulu
 *
 * Authors: Aapo Rantalainen
 */

#include "jammo-note-button.h"
#include "jammo-miditrack-view.h"
#include <string.h>
#include "../meam/jammo-midi.h"

G_DEFINE_TYPE(JammoNoteButton, jammo_note_button, TANGLE_TYPE_DRAGGABLE_ACTOR);

enum {
	PROP_0,
	PROP_START_EVENT,
	PROP_STOP_EVENT,
};

struct _JammoNoteButtonPrivate {
	JammoMidiEvent* start_event;
	JammoMidiEvent* stop_event;
};

static gboolean on_clicked(TangleButton* button, gpointer user_data);

static const ClutterColor default_note_color = { 255, 0, 0, 128 };

//This is only for debugging
ClutterActor* jammo_note_button_new_without_events() {
	ClutterActor* note =  CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_NOTE_BUTTON, NULL));
	ClutterActor* box =  clutter_rectangle_new_with_color(&default_note_color);

	tangle_button_set_normal_background_actor(TANGLE_BUTTON(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(note))), box);
	return note;
}

/*
width,height
and start+stop events
*/
ClutterActor* jammo_note_button_new(guint width, guint height, JammoMidiEvent* start, JammoMidiEvent* stop) {
	//printf("making note with x=%d and y=%d\n",x,y);
	ClutterActor* note =  CLUTTER_ACTOR(g_object_new(JAMMO_TYPE_NOTE_BUTTON, "start-event",start,"stop-event",stop,NULL));
	ClutterActor* box =  clutter_rectangle_new_with_color(&default_note_color);
	clutter_actor_set_size(box,(gfloat)width,(gfloat)height);
	tangle_button_set_normal_background_actor(TANGLE_BUTTON(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(note))), box);

	return note;
}



static gboolean jammo_note_button_end_dragging(TangleDraggableActor* draggable_actor, TangleDragging* dragging) {
	gboolean retvalue = FALSE;
	JammoNoteButton* note_button;
	ClutterActor* parent;
	ClutterActor* grand_parent;
	
	note_button = JAMMO_NOTE_BUTTON(draggable_actor);
	
	parent = clutter_actor_get_parent(CLUTTER_ACTOR(note_button));
	grand_parent = clutter_actor_get_parent(parent);
	if (JAMMO_IS_MIDITRACK_VIEW(grand_parent)) {
		clutter_container_remove_actor(CLUTTER_CONTAINER(parent), CLUTTER_ACTOR(note_button));
	} else {
		retvalue = TANGLE_DRAGGABLE_ACTOR_CLASS(jammo_note_button_parent_class)->end_dragging(draggable_actor, dragging);
	}
	
	return retvalue;
}

static GObject* jammo_note_button_constructor(GType type, guint n_construct_properties, GObjectConstructParam* construct_properties) {
	GObject* object;
	ClutterActor* button;

	button = tangle_button_new();
	tangle_widget_set_prefer_background_size(TANGLE_WIDGET(button), TRUE);

	object = tangle_construct_with_extra_properties(G_OBJECT_CLASS(jammo_note_button_parent_class)->constructor, type, n_construct_properties, construct_properties, "wrapped", button, NULL);

	g_signal_connect(button, "clicked", G_CALLBACK(on_clicked), object);

	return object;
}

static void jammo_note_button_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoNoteButton* note_button;
	
	note_button = JAMMO_NOTE_BUTTON(object);

	switch (prop_id) {
		case PROP_START_EVENT:
			note_button->priv->start_event = (JammoMidiEvent*)(g_value_get_pointer(value));
			break;
		case PROP_STOP_EVENT:
			note_button->priv->stop_event = (JammoMidiEvent*)(g_value_get_pointer(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_note_button_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
	JammoNoteButton* note_button;
	note_button = JAMMO_NOTE_BUTTON(object);

	switch (prop_id) {
		case PROP_START_EVENT:
			g_value_set_pointer (value, note_button->priv->start_event);
			break;
		case PROP_STOP_EVENT:
			g_value_set_pointer (value, note_button->priv->stop_event);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
        }
}

static void jammo_note_button_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_note_button_parent_class)->finalize(object);
}

static void jammo_note_button_dispose(GObject* object) {
	/*
	JammoNoteButton* note_button = JAMMO_NOTE_BUTTON(object);
	printf("dispose note_button\n");
	if (note_button->priv->start_event) {
		g_free(note_button->priv->start_event);
		note_button->priv->start_event = NULL;
	}
	if (note_button->priv->stop_event) {
		g_free(note_button->priv->stop_event);
		note_button->priv->stop_event = NULL;
	}
*/
	G_OBJECT_CLASS(jammo_note_button_parent_class)->dispose(object);
}

static void jammo_note_button_class_init(JammoNoteButtonClass* note_button_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(note_button_class);
	TangleDraggableActorClass* draggable_actor_class = TANGLE_DRAGGABLE_ACTOR_CLASS(note_button_class);

	gobject_class->constructor = jammo_note_button_constructor;
	gobject_class->finalize = jammo_note_button_finalize;
	gobject_class->dispose = jammo_note_button_dispose;
	gobject_class->set_property = jammo_note_button_set_property;
	gobject_class->get_property = jammo_note_button_get_property;

	draggable_actor_class->end_dragging = jammo_note_button_end_dragging;

	/**
	 * JammoNoteButton:start-event:
	 */
	g_object_class_install_property(gobject_class, PROP_START_EVENT,
	                                g_param_spec_pointer ("start-event",
	                                "Start Event",
	                                "",
	                                //NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoNoteButton:stop-event:
	 */
	g_object_class_install_property(gobject_class, PROP_STOP_EVENT,
	                                g_param_spec_pointer ("stop-event",
	                                "Stop Event",
	                                "",
	                                //NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(JammoNoteButtonPrivate));
}


static void jammo_note_button_init(JammoNoteButton* note_button) {
	note_button->priv = G_TYPE_INSTANCE_GET_PRIVATE(note_button, JAMMO_TYPE_NOTE_BUTTON, JammoNoteButtonPrivate);
	//printf("jammo-note-button-init\n");
}


static gboolean on_clicked(TangleButton* button, gpointer user_data) {
	JammoNoteButton* note_button;
	note_button = JAMMO_NOTE_BUTTON(user_data);

	GstClockTime start=0;
	GstClockTime stop=0;
	if (note_button->priv->start_event) {
		jammomidi_print_event(note_button->priv->start_event);
		start = note_button->priv->start_event->timestamp;
	}
	if (note_button->priv->stop_event) {
		jammomidi_print_event(note_button->priv->stop_event);
		stop = note_button->priv->stop_event->timestamp;
	}
	printf("duration of this note-button: %f s\n",((long long unsigned)(stop-start)*1.0)/1000000000LLU);

	return FALSE;
}

