/*
 * jammo-game.c
 *
 * This file is part of JamMo.
 *
 * (c) 2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-game.h"
#include "jammo-chum.h"
#include <string.h>

static void clutter_scriptable_iface_init(ClutterScriptableIface* iface);

G_DEFINE_TYPE_WITH_CODE(JammoGame, jammo_game, TANGLE_TYPE_OBJECT,
                        G_IMPLEMENT_INTERFACE(CLUTTER_TYPE_SCRIPTABLE, clutter_scriptable_iface_init););

enum {
	PROP_0,
	PROP_VIEW,
	PROP_TASKS_TO_COMPLETE,
	PROP_TASKS_COMPLETED,
	PROP_STARTED_ACTION,
	PROP_TASK_COMPLETED_ACTION,
	PROP_TASK_REDONE_ACTION,
	PROP_COMPLETED_ACTION
};

enum {
	STARTED,
	COMPLETED,
	TASK_COMPLETED,
	TASK_REDONE,
	LAST_SIGNAL
};

typedef struct _Task {
	gchar* name;
	
	guint completed : 1;
} Task;

struct _JammoGamePrivate {
	TangleView* view;
	GList* tasks;
	guint tasks_to_complete;
	
	TangleAction* started_action;
	TangleAction* task_completed_action;
	TangleAction* task_redone_action;
	TangleAction* completed_action;
	
	GSList* enabled_actors;
	GSList* disabled_actors;
	
	guint completed : 1;
};

static guint signals[LAST_SIGNAL] = { 0 };
static ClutterScriptableIface* parent_scriptable_iface = NULL;

static void hide_actor(ClutterActor* actor, gpointer user_data);
static gint task_compare_name(Task* task, const gchar* name);

static void marshal_VOID__STRING_UINT (GClosure     *closure,
                                       GValue       *return_value G_GNUC_UNUSED,
                                       guint         n_param_values,
                                       const GValue *param_values,
                                       gpointer      invocation_hint G_GNUC_UNUSED,
                                       gpointer      marshal_data);
				
JammoGame* jammo_game_new(TangleView* view) {

	return JAMMO_GAME(g_object_new(JAMMO_TYPE_GAME, "view", view, NULL));
}

void jammo_game_start(JammoGame* game) {
	ClutterActor* parent;
	GSList* slist_item;
	ClutterActor* actor;
	
	parent = clutter_actor_get_parent(CLUTTER_ACTOR(game->priv->view));
	g_warn_if_fail(parent != NULL);
		
	if (CLUTTER_IS_CONTAINER(parent)) {
		clutter_container_foreach(CLUTTER_CONTAINER(parent), hide_actor, game->priv->view);
	}

	clutter_container_foreach(CLUTTER_CONTAINER(game->priv->view), CLUTTER_CALLBACK(clutter_actor_hide), NULL);
	for (slist_item = game->priv->enabled_actors; slist_item; slist_item = slist_item->next) {
		actor = CLUTTER_ACTOR(slist_item->data);
		clutter_actor_show(actor);
	}
	for (slist_item = game->priv->disabled_actors; slist_item; slist_item = slist_item->next) {
		actor = CLUTTER_ACTOR(slist_item->data);
		clutter_actor_show(actor);
		jammo_chum_disable_all_actors(actor);
	}
	
	tangle_actor_show(TANGLE_ACTOR(game->priv->view));

	g_signal_emit(game, signals[STARTED], 0);
}

void jammo_game_stop(JammoGame* game) {
	GSList* slist_item;
	ClutterActor* actor;

	clutter_container_foreach(CLUTTER_CONTAINER(game->priv->view), CLUTTER_CALLBACK(clutter_actor_show), NULL);
	for (slist_item = game->priv->disabled_actors; slist_item; slist_item = slist_item->next) {
		actor = CLUTTER_ACTOR(slist_item->data);
		jammo_chum_enable_all_actors(actor);
	}
}

GList* jammo_game_get_tasks(JammoGame* game) {
	GList* task_names = NULL;
	GList* task_in_list;
	Task* task;
	
	for (task_in_list = game->priv->tasks; task_in_list; task_in_list = task_in_list->next) {
		task = (Task*)task_in_list->data;
		task_names = g_list_prepend(task_names, task->name);
	}
	
	return task_names;
}

void jammo_game_add_task(JammoGame* game, const gchar* task_name) {
	Task* task;
	
	if (g_list_find_custom(game->priv->tasks, task_name, (GCompareFunc)task_compare_name)) {
		g_warning("A task '%s' already exists in the game.\n", task_name);
	} else {
		task = g_new0(Task, 1);
		task->name = g_strdup(task_name);
		game->priv->tasks = g_list_prepend(game->priv->tasks, task);
	}
}

gboolean jammo_game_get_task_completed(JammoGame* game, const gchar* task_name) {
	gboolean completed = FALSE;
	GList* task_in_list;
	Task* task;
	
	if (!(task_in_list = g_list_find_custom(game->priv->tasks, task_name, (GCompareFunc)task_compare_name))) {
		g_warning("A task '%s' does not exists in the game.\n", task_name);
	} else {
		task = (Task*)task_in_list->data;
		completed = task->completed;
	}
	
	return completed;
}

void jammo_game_set_task_completed(JammoGame* game, const gchar* task_name, gboolean is_completed) {
	GList* task_in_list;
	Task* task;
	guint tasks_completed;
	
	if (!(task_in_list = g_list_find_custom(game->priv->tasks, task_name, (GCompareFunc)task_compare_name))) {
		g_warning("A task '%s' does not exists in the game.\n", task_name);
	} else {
		task = (Task*)task_in_list->data;
		if (task->completed && is_completed &&
		    (tasks_completed = jammo_game_get_tasks_completed(game)) < game->priv->tasks_to_complete) {
			g_signal_emit(game, signals[TASK_REDONE], 0, task_name, tasks_completed);
		} else if (task->completed != is_completed) {
			task->completed = is_completed;

			if (task->completed && !game->priv->completed) {
				tasks_completed = jammo_game_get_tasks_completed(game);
				g_signal_emit(game, signals[TASK_COMPLETED], 0, task_name, tasks_completed);

				if (tasks_completed >= game->priv->tasks_to_complete) {
					g_signal_emit(game, signals[COMPLETED], 0);
				}
			}
		}
	}
}

guint jammo_game_get_tasks_to_complete(JammoGame* game) {

	return game->priv->tasks_to_complete;
}

void jammo_game_set_tasks_to_complete(JammoGame* game, guint number_of_tasks) {
	if (game->priv->tasks_to_complete != number_of_tasks) {
		game->priv->tasks_to_complete = number_of_tasks;
		g_object_notify(G_OBJECT(game), "tasks-to-complete");
	}
}

guint jammo_game_get_tasks_completed(JammoGame* game) {
	guint number_of_completed_tasks = 0;
	GList* task_in_list;
	Task* task;
	
	for (task_in_list = game->priv->tasks; task_in_list; task_in_list = task_in_list->next) {
		task = (Task*)task_in_list->data;
		if (task->completed) {
			number_of_completed_tasks++;
		}
	}
	
	
	return number_of_completed_tasks;
}

#define SET_ACTION(a) \
	if (game->priv->a) { \
		g_object_unref(game->priv->a); \
	} \
	game->priv->a = TANGLE_ACTION(g_value_get_object(value)); \
	g_object_ref(game->priv->a);

static void jammo_game_started(JammoGame* game) {
	if (game->priv->started_action) {
		tangle_action_execute_full_list(game->priv->started_action, G_OBJECT(game), "started", NULL);
	}
}

static void jammo_game_task_completed(JammoGame* game, const gchar* task_name, guint tasks_completed) {
	if (game->priv->task_completed_action) {
		tangle_action_execute_full_list(game->priv->task_completed_action, G_OBJECT(game), "task-completed", "task-name", task_name, "tasks-completed", tasks_completed, NULL);
	}
}

static void jammo_game_task_redone(JammoGame* game, const gchar* task_name, guint tasks_completed) {
	if (game->priv->task_redone_action) {
		tangle_action_execute_full_list(game->priv->task_redone_action, G_OBJECT(game), "task-redone", "task-name", task_name, "tasks-completed", tasks_completed, NULL);
	}
}

static void jammo_game_completed(JammoGame* game) {
	if (game->priv->completed_action) {
		tangle_action_execute_full_list(game->priv->completed_action, G_OBJECT(game), "completed", NULL);
	}
}

static void jammo_game_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoGame* game;
	
	game = JAMMO_GAME(object);

	switch (prop_id) {
		case PROP_VIEW:
			g_assert(!game->priv->view);
			game->priv->view = TANGLE_VIEW(g_value_get_object(value));
			break;
		case PROP_TASKS_TO_COMPLETE:
			jammo_game_set_tasks_to_complete(game, g_value_get_uint(value));
			break;
		case PROP_STARTED_ACTION:
			SET_ACTION(started_action);
			break;
		case PROP_TASK_COMPLETED_ACTION:
			SET_ACTION(task_completed_action);
			break;
		case PROP_TASK_REDONE_ACTION:
			SET_ACTION(task_redone_action);
			break;
		case PROP_COMPLETED_ACTION:
			SET_ACTION(completed_action);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_game_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoGame* game;

	game = JAMMO_GAME(object);

        switch (prop_id) {
		case PROP_VIEW:
			g_value_set_object(value, game->priv->view);
			break;
		case PROP_TASKS_TO_COMPLETE:
			g_value_set_uint(value, game->priv->tasks_to_complete);
			break;
		case PROP_TASKS_COMPLETED:
			g_value_set_uint(value, jammo_game_get_tasks_completed(game));
			break;
		case PROP_STARTED_ACTION:
			g_value_set_object(value, game->priv->started_action);
			break;
		case PROP_TASK_COMPLETED_ACTION:
			g_value_set_object(value, game->priv->task_completed_action);
			break;
		case PROP_TASK_REDONE_ACTION:
			g_value_set_object(value, game->priv->task_redone_action);
			break;
		case PROP_COMPLETED_ACTION:
			g_value_set_object(value, game->priv->completed_action);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_game_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_game_parent_class)->finalize(object);
}

static void jammo_game_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_game_parent_class)->dispose(object);
}

static void jammo_game_class_init(JammoGameClass* game_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(game_class);

	gobject_class->finalize = jammo_game_finalize;
	gobject_class->dispose = jammo_game_dispose;
	gobject_class->set_property = jammo_game_set_property;
	gobject_class->get_property = jammo_game_get_property;

	game_class->started = jammo_game_started;
	game_class->task_completed = jammo_game_task_completed;
	game_class->task_redone = jammo_game_task_redone;
	game_class->completed = jammo_game_completed;

	/**
	 * JammoGame:view:
	 *
	 * The TangleView that contains this game.
	 */
	g_object_class_install_property(gobject_class, PROP_VIEW,
	                                g_param_spec_object("view",
	                                "View",
	                                "The TangleView that contains this game",
	                                TANGLE_TYPE_VIEW,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoGame:tasks-to-complete:
	 *
	 * The number of tasks that has been completed.
	 */
	g_object_class_install_property(gobject_class, PROP_TASKS_TO_COMPLETE,
	                                g_param_spec_uint("tasks-to-complete",
	                                "Tasks to complete",
	                                "The number of tasks that has to be completed",
	                                0, G_MAXUINT, 0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoGame:tasks-completed:
	 *
	 * The number of tasks that has been completed.
	 */
	g_object_class_install_property(gobject_class, PROP_TASKS_COMPLETED,
	                                g_param_spec_uint("tasks-completed",
	                                "Tasks completed",
	                                "The number of tasks that has been completed",
	                                0, G_MAXUINT, 0,
	                                G_PARAM_READABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoGame:started-action:
	 *
	 * The action that correspond to the started signal.
	 */
	g_object_class_install_property(gobject_class, PROP_STARTED_ACTION,
	                                g_param_spec_object("started-action",
	                                "Started action",
	                                "The action that correspond to the started signal",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoGame:task-completed-action:
	 *
	 * The action that correspond to the task-completed signal.
	 */
	g_object_class_install_property(gobject_class, PROP_TASK_COMPLETED_ACTION,
	                                g_param_spec_object("task-completed-action",
	                                "Task completed action",
	                                "The action that correspond to the task-completed signal",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoGame:task-redone-action:
	 *
	 * The action that correspond to the task-redone signal.
	 */
	g_object_class_install_property(gobject_class, PROP_TASK_REDONE_ACTION,
	                                g_param_spec_object("task-redone-action",
	                                "Taskredone action",
	                                "The action that correspond to the task-redone signal",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoGame:completed-action:
	 *
	 * The action that correspond to the completed signal.
	 */
	g_object_class_install_property(gobject_class, PROP_COMPLETED_ACTION,
	                                g_param_spec_object("completed-action",
	                                "Completed action",
	                                "The action that correspond to the completed signal",
	                                TANGLE_TYPE_ACTION,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	/**
	 * JammoGame::started:
	 * @game: the object which received the signal
	 *
	 * The ::started signal is emitted when the game has been started by calling jammo_game_start().
	 */
	signals[STARTED] = g_signal_new("started", G_TYPE_FROM_CLASS(gobject_class),
	                                G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(JammoGameClass, started),
					NULL, NULL,
					g_cclosure_marshal_VOID__VOID,
					G_TYPE_NONE, 0);
	/**
	 * JammoGame::task-completed:
	 * @game: the object which received the signal
	 * @task_name: the name of the task that has been completed
	 * @tasks_completed: the number of tasks that has been already completed
	 *
	 * The ::task-completed signal is emitted when a task has been completed by calling jammo_game_set_task_completed().
	 */
	signals[TASK_COMPLETED] = g_signal_new("task-completed", G_TYPE_FROM_CLASS(gobject_class),
	                                       G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(JammoGameClass, task_completed),
					       NULL, NULL,
					       marshal_VOID__STRING_UINT,
					       G_TYPE_NONE, 2,
					       G_TYPE_STRING,
					       G_TYPE_UINT);
	/**
	 * JammoGame::task-completed:
	 * @game: the object which received the signal
	 * @task_name: the name of the task that has been completed
	 * @tasks_completed: the number of tasks that has been already completed
	 *
	 * The ::task-redone signal is emitted when a task has been already completed is completed again by calling jammo_game_set_task_completed().
	 */
	signals[TASK_REDONE] = g_signal_new("task-redone", G_TYPE_FROM_CLASS(gobject_class),
	                                       G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(JammoGameClass, task_redone),
					       NULL, NULL,
					       marshal_VOID__STRING_UINT,
					       G_TYPE_NONE, 2,
					       G_TYPE_STRING,
					       G_TYPE_UINT);
	/**
	 * JammoGame::completed:
	 * @game: the object which received the signal
	 *
	 * The ::completed signal is emitted when the game has been completed by calling jammo_game_complete().
	 */
	signals[COMPLETED] = g_signal_new("completed", G_TYPE_FROM_CLASS(gobject_class),
	                                  G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET(JammoGameClass, completed),
					  NULL, NULL,
				  	  g_cclosure_marshal_VOID__VOID,
				 	  G_TYPE_NONE, 0);

	g_type_class_add_private (gobject_class, sizeof (JammoGamePrivate));
}

static void jammo_game_init(JammoGame* game) {
	game->priv = G_TYPE_INSTANCE_GET_PRIVATE(game, JAMMO_TYPE_GAME, JammoGamePrivate);
}

static gboolean jammo_game_parse_custom_node(ClutterScriptable* scriptable, ClutterScript* script, GValue* value, const gchar* name, JsonNode* node) {
	gboolean retvalue = FALSE;
	JammoGame* game;
	GSList* slist = NULL;
	JsonArray* array;
	gint i;
	JsonNode* n;
	GObject* object;
	
	game = JAMMO_GAME(scriptable);

	if (!strcmp(name, "enabled-actors") || !strcmp(name, "disabled-actors")) {
		if (JSON_NODE_TYPE(node) != JSON_NODE_ARRAY) {
			g_warning("Expected JSON array for '%s', skipped.", name);
		} else {
			array = json_node_get_array(node);
			for (i = json_array_get_length(array) - 1; i >= 0; i--) {
				n = json_array_get_element(array, i);
				if (JSON_NODE_TYPE(n) != JSON_NODE_VALUE || json_node_get_value_type(n) != G_TYPE_STRING) {
					g_warning("Expected a string value in '%s' array, skipped.", name);
				} else if (!(object = clutter_script_get_object(script, json_node_get_string(n)))) {
					g_warning("A reference to non-existing object '%s' in '%s' array, skipped.", json_node_get_string(n), name);
				} else {
					slist = g_slist_prepend(slist, object);
				}
			}
					
			g_value_init(value, G_TYPE_POINTER);
			g_value_set_pointer(value, slist);

			retvalue = TRUE;
		}
	} else if (!strcmp(name, "tasks")) {
		if (JSON_NODE_TYPE(node) != JSON_NODE_ARRAY) {
			g_warning("Expected JSON array for '%s', skipped.", name);
		} else {
			array = json_node_get_array(node);
			for (i = json_array_get_length(array) - 1; i >= 0; i--) {
				n = json_array_get_element(array, i);
				if (JSON_NODE_TYPE(n) != JSON_NODE_VALUE || json_node_get_value_type(n) != G_TYPE_STRING) {
					g_warning("Expected a string value in '%s' array, skipped.", name);
				} else {
					slist = g_slist_prepend(slist,json_node_dup_string(n));
				}
			}
					
			g_value_init(value, G_TYPE_POINTER);
			g_value_set_pointer(value, slist);
			
			retvalue = TRUE;
		}	
	} else if (parent_scriptable_iface->parse_custom_node) {
		retvalue = parent_scriptable_iface->parse_custom_node(scriptable, script, value, name, node);
	}
	
	return retvalue;
}

static void jammo_game_set_custom_property(ClutterScriptable* scriptable, ClutterScript* script, const gchar* name, const GValue* value) {
	JammoGame* game;
	GSList* slist;
	GSList* slist_item;
	
	game = JAMMO_GAME(scriptable);
	
	if (!strcmp(name, "enabled-actors")) {
		g_return_if_fail(!game->priv->enabled_actors);
		g_return_if_fail(G_VALUE_HOLDS(value, G_TYPE_POINTER));

		game->priv->enabled_actors = (GSList*)g_value_get_pointer(value);
	} else if (!strcmp(name, "disabled-actors")) {
		g_return_if_fail(!game->priv->disabled_actors);
		g_return_if_fail(G_VALUE_HOLDS(value, G_TYPE_POINTER));

		game->priv->disabled_actors = (GSList*)g_value_get_pointer(value);
	} else if (!strcmp(name, "tasks")) {
		g_return_if_fail(G_VALUE_HOLDS(value, G_TYPE_POINTER));

		slist = (GSList*)g_value_get_pointer(value);
		for (slist_item = slist; slist_item; slist_item = slist_item->next) {
			jammo_game_add_task(game, (const gchar*)slist_item->data);
			g_free(slist_item->data);
		}
		g_slist_free(slist);	
	} else if (parent_scriptable_iface->set_custom_property) {
		parent_scriptable_iface->set_custom_property(scriptable, script, name, value);
	} else {
		g_object_set_property(G_OBJECT(scriptable), name, value);
	}
}

static void clutter_scriptable_iface_init(ClutterScriptableIface* iface) {
	if (!(parent_scriptable_iface = g_type_interface_peek_parent (iface))) {
		parent_scriptable_iface = g_type_default_interface_peek(CLUTTER_TYPE_SCRIPTABLE);
	}
	
	iface->parse_custom_node = jammo_game_parse_custom_node;
	iface->set_custom_property = jammo_game_set_custom_property;
}


static void hide_actor(ClutterActor* actor, gpointer user_data) {
	if (actor != user_data) {
		if (TANGLE_IS_ACTOR(actor)) {
			tangle_actor_hide_animated(TANGLE_ACTOR(actor));
		} else {
			clutter_actor_hide(actor);
		}
	}
}

static gint task_compare_name(Task* task, const gchar* name) {

	return strcmp(task->name, name);
}

/* Generated marshaler */

#ifdef G_ENABLE_DEBUG
#define g_marshal_value_peek_uint(v)     g_value_get_uint (v)
#define g_marshal_value_peek_string(v)   (char*) g_value_get_string (v)
#else /* !G_ENABLE_DEBUG */
/* WARNING: This code accesses GValues directly, which is UNSUPPORTED API.
 *          Do not access GValues directly in your code. Instead, use the
 *          g_value_get_*() functions
 */
#define g_marshal_value_peek_uint(v)     (v)->data[0].v_uint
#define g_marshal_value_peek_string(v)   (v)->data[0].v_pointer
#endif /* !G_ENABLE_DEBUG */

static void marshal_VOID__STRING_UINT (GClosure     *closure,
                                       GValue       *return_value G_GNUC_UNUSED,
                                       guint         n_param_values,
                                       const GValue *param_values,
                                       gpointer      invocation_hint G_GNUC_UNUSED,
                                       gpointer      marshal_data)
{
  typedef void (*GMarshalFunc_VOID__STRING_UINT) (gpointer     data1,
                                                  gpointer     arg_1,
                                                  guint        arg_2,
                                                  gpointer     data2);
  register GMarshalFunc_VOID__STRING_UINT callback;
  register GCClosure *cc = (GCClosure*) closure;
  register gpointer data1, data2;

  g_return_if_fail (n_param_values == 3);

  if (G_CCLOSURE_SWAP_DATA (closure))
    {
      data1 = closure->data;
      data2 = g_value_peek_pointer (param_values + 0);
    }
  else
    {
      data1 = g_value_peek_pointer (param_values + 0);
      data2 = closure->data;
    }
  callback = (GMarshalFunc_VOID__STRING_UINT) (marshal_data ? marshal_data : cc->callback);

  callback (data1,
            g_marshal_value_peek_string (param_values + 1),
            g_marshal_value_peek_uint (param_values + 2),
            data2);
}

