/*
 * jammo-playing-track.h
 *
 * This file is part of JamMo.
 *
 * (c) 2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-playing-track.h"

G_DEFINE_TYPE(JammoPlayingTrack, jammo_playing_track, JAMMO_TYPE_TRACK);

enum {
	PROP_0,
	PROP_MUTED,
	PROP_VOLUME,
	PROP_PANNING,
	PROP_ECHO_DELAY,
	PROP_ECHO_INTENSITY,
	PROP_ECHO_FEEDBACK
};

struct _JammoPlayingTrackPrivate {
	GstElement* bin;
	GstElement* audioecho;
	GstElement* volume;
	GstElement* audiopanorama;
};

#define RETURN_OBJECT_PROPERTY(o,n,t) t v; g_object_get(o, n, &v, NULL); return v;

/**
 * jammo_playing_track_get_muted:
 * @playing_track: a #JammoPlayingTrack
 *
 * Returns whether the track is muted or audible.
 *
 * Return value: whether the track is muted or not
 */
gboolean jammo_playing_track_get_muted(JammoPlayingTrack* playing_track) {

	RETURN_OBJECT_PROPERTY(playing_track->priv->volume, "muted", gboolean);
}

/**
 * jammo_playing_track_set_muted:
 * @playing_track: a #JammoPlayingTrack
 * @muted: whether the track is muted or not
 *
 * Sets whether the track is muted or audible.
 */
void jammo_playing_track_set_muted(JammoPlayingTrack* playing_track, gboolean muted) {
	if (jammo_playing_track_get_muted(playing_track) != muted) {
		g_object_set(playing_track->priv->volume, "mute", muted, NULL);
		g_object_notify(G_OBJECT(playing_track), "muted");
	}
}

/**
 * jammo_playing_track_get_volume:
 * @playing_track: a #JammoPlayingTrack
 *
 * Returns the individual volume of the track.
 *
 * Return value: a volume between 0.0 - 1.0
 */
gfloat jammo_playing_track_get_volume(JammoPlayingTrack* playing_track) {
	
	RETURN_OBJECT_PROPERTY(playing_track->priv->volume, "volume", gdouble);
}

/**
 * jammo_playing_track_set_volume:
 * @playing_track: a #JammoPlayingTrack
 * @volume: volume for the track: 0.0(min) - 1.0(max)
 *
 * Sets the individual volume of the track.
 */
void jammo_playing_track_set_volume(JammoPlayingTrack* playing_track, gfloat volume) {
	if (jammo_playing_track_get_volume(playing_track) != volume) {
		g_object_set(playing_track->priv->volume, "volume", (double)volume, NULL);
		g_object_notify(G_OBJECT(playing_track), "volume");
	}
}

/**
 * jammo_playing_track_get_panning:
 * @playing_track: a #JammoPlayingTrack
 *
 * Returns the audio position in stereo panorama (left - right).
 *
 * Return value: a volume between -1.0 - 1.0
 */
gfloat jammo_playing_track_get_panning(JammoPlayingTrack* playing_track) {

	RETURN_OBJECT_PROPERTY(playing_track->priv->audiopanorama, "panorama", gfloat);
}

/**
 * jammo_playing_track_set_panning:
 * @playing_track: a #JammoPlayingTrack
 * @panning: the position in stereo panorama: -1.0 (left) - 1.0 (right)
 *
 * Sets the audio position in stereo panorama (left, center, right).
 */
void jammo_playing_track_set_panning(JammoPlayingTrack* playing_track, gfloat panning) {
	if (jammo_playing_track_get_panning(playing_track) != panning) {
		g_object_set(playing_track->priv->audiopanorama, "panorama", panning, NULL);
		g_object_notify(G_OBJECT(playing_track), "panning");
	}
}

guint64 jammo_playing_track_get_echo_delay(JammoPlayingTrack* playing_track) {
	
	RETURN_OBJECT_PROPERTY(playing_track->priv->audioecho, "delay", guint64);
}

void jammo_playing_track_set_echo_delay(JammoPlayingTrack* playing_track, guint64 echo_delay) {
	if (jammo_playing_track_get_echo_delay(playing_track) != echo_delay) {
		g_object_set(playing_track->priv->audioecho, "delay", echo_delay, NULL);
		g_object_notify(G_OBJECT(playing_track), "echo-delay");
	}
}

gfloat jammo_playing_track_get_echo_intensity(JammoPlayingTrack* playing_track) {

	RETURN_OBJECT_PROPERTY(playing_track->priv->audioecho, "intensity", guint64);
}

void jammo_playing_track_set_echo_intensity(JammoPlayingTrack* playing_track, gfloat echo_intensity) {
	if (jammo_playing_track_get_echo_intensity(playing_track) != echo_intensity) {
		g_object_set(playing_track->priv->audioecho, "intensity", echo_intensity, NULL);
		g_object_notify(G_OBJECT(playing_track), "echo-intensity");
	}
}

gfloat jammo_playing_track_get_echo_feedback(JammoPlayingTrack* playing_track) {
	
	RETURN_OBJECT_PROPERTY(playing_track->priv->audioecho, "feedback", gfloat);
}

void jammo_playing_track_set_echo_feedback(JammoPlayingTrack* playing_track, gfloat echo_feedback) {
	if (jammo_playing_track_get_echo_feedback(playing_track) != echo_feedback) {
		g_object_set(playing_track->priv->audioecho, "feedback", echo_feedback, NULL);
		g_object_notify(G_OBJECT(playing_track), "echo-feedback");
	}
}

static GstElement* jammo_playing_track_get_element(JammoTrack* playing_track) {

	return JAMMO_PLAYING_TRACK(playing_track)->priv->bin;
}

static void jammo_playing_track_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoPlayingTrack* playing_track;
	
	playing_track = JAMMO_PLAYING_TRACK(object);

	switch (prop_id) {
		case PROP_MUTED:
			jammo_playing_track_set_muted(playing_track, g_value_get_boolean(value));
			break;
		case PROP_VOLUME:
			jammo_playing_track_set_volume(playing_track, g_value_get_float(value));
			break;
		case PROP_PANNING:
			jammo_playing_track_set_panning(playing_track, g_value_get_float(value));
			break;
		case PROP_ECHO_DELAY:
			jammo_playing_track_set_echo_delay(playing_track, g_value_get_uint64(value));
			break;
		case PROP_ECHO_INTENSITY:
			jammo_playing_track_set_echo_intensity(playing_track, g_value_get_float(value));
			break;
		case PROP_ECHO_FEEDBACK:
			jammo_playing_track_set_echo_feedback(playing_track, g_value_get_float(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_playing_track_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoPlayingTrack* playing_track;

	playing_track = JAMMO_PLAYING_TRACK(object);

        switch (prop_id) {
		case PROP_MUTED:
			g_value_set_boolean(value, jammo_playing_track_get_muted(playing_track));
			break;
		case PROP_VOLUME:
			g_value_set_float(value, jammo_playing_track_get_volume(playing_track));
			break;
		case PROP_PANNING:
			g_value_set_float(value, jammo_playing_track_get_panning(playing_track));
			break;
		case PROP_ECHO_DELAY:
			g_value_set_uint64(value, jammo_playing_track_get_echo_delay(playing_track));
			break;
		case PROP_ECHO_INTENSITY:
			g_value_set_float(value, jammo_playing_track_get_echo_intensity(playing_track));
			break;
		case PROP_ECHO_FEEDBACK:
			g_value_set_float(value, jammo_playing_track_get_echo_feedback(playing_track));
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static void jammo_playing_track_finalize(GObject* object) {
	G_OBJECT_CLASS(jammo_playing_track_parent_class)->finalize(object);
}

static void jammo_playing_track_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_playing_track_parent_class)->dispose(object);
}

static void jammo_playing_track_constructed(GObject* object) {
	JammoPlayingTrack* playing_track;
	JammoPlayingTrackClass* playing_track_class;
	GstElement* element;
	GstPad* pad;
	
	playing_track = JAMMO_PLAYING_TRACK(object);
	playing_track_class = JAMMO_PLAYING_TRACK_GET_CLASS(playing_track);
	playing_track->priv->bin = gst_bin_new(NULL);
	element = playing_track_class->get_playing_element(playing_track);
	playing_track->priv->audioecho = gst_element_factory_make("audioecho", NULL);
	playing_track->priv->volume = gst_element_factory_make("volume", NULL);
	playing_track->priv->audiopanorama = gst_element_factory_make("audiopanorama", NULL);
	gst_bin_add_many(GST_BIN(playing_track->priv->bin),
	                 element,
	                 playing_track->priv->audioecho,
	                 playing_track->priv->volume,
	                 playing_track->priv->audiopanorama,
	                 NULL);
	gst_element_link_many(element,
	                      playing_track->priv->audioecho,
	                      playing_track->priv->volume,
	                      playing_track->priv->audiopanorama,
	                      NULL);

	pad = gst_element_get_pad(playing_track->priv->audiopanorama, "src");
	gst_element_add_pad(playing_track->priv->bin, gst_ghost_pad_new("src", pad));
	gst_object_unref(pad);
}
	
static void jammo_playing_track_class_init(JammoPlayingTrackClass* playing_track_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(playing_track_class);
	JammoTrackClass* track_class = JAMMO_TRACK_CLASS(playing_track_class);

	gobject_class->constructed = jammo_playing_track_constructed;
	gobject_class->finalize = jammo_playing_track_finalize;
	gobject_class->dispose = jammo_playing_track_dispose;
	gobject_class->set_property = jammo_playing_track_set_property;
	gobject_class->get_property = jammo_playing_track_get_property;

	track_class->get_element = jammo_playing_track_get_element;

	/**
	 * JammoPlayingTrack:muted:
	 *
	 * Whether the track is muted or audible.
	 */
	g_object_class_install_property(gobject_class, PROP_MUTED,
	                                g_param_spec_boolean("muted",
	                                "Muted",
	                                "Whether the track is muted or audible",
	                                FALSE,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoPlayingTrack:volume:
	 *
	 * The volume of the track.
	 */
	g_object_class_install_property(gobject_class, PROP_VOLUME,
	                                g_param_spec_float("volume",
	                                "Volume",
	                                "The volume of the track",
	                                0.0, 1.0, 1.0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoPlayingTrack:panning:
	 *
	 * The audio position in stereo panorama.
	 */
	g_object_class_install_property(gobject_class, PROP_PANNING,
	                                g_param_spec_float("panning",
	                                "Panning",
	                                "The audio position in stereo panorama",
	                                -1.0, 1.0, 0.0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoPlayingTrack:echo-delay:
	 *
	 * Delay of the echo in nanoseconds.
	 */
	g_object_class_install_property(gobject_class, PROP_ECHO_DELAY,
	                                g_param_spec_uint64("echo-delay",
	                                "Echo delay",
	                                "The delay of the echo in nanoseconds",
	                                1, G_MAXUINT64, 1,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoPlayingTrack:echo-intensity:
	 *
	 * The intensity of the echo.
	 */
	g_object_class_install_property(gobject_class, PROP_ECHO_INTENSITY,
	                                g_param_spec_float("echo-intensity",
	                                "Echo intensity",
	                                "The intensity of the echo",
	                                -0.0, 1.0, 0.0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));
	/**
	 * JammoPlayingTrack:echo-feedback:
	 *
	 * Amount of feedback in percentage.
	 */
	g_object_class_install_property(gobject_class, PROP_ECHO_FEEDBACK,
	                                g_param_spec_float("echo-feedback",
	                                "Echo feedback",
	                                "Amount of feedback in percentage",
	                                0.0, 1.0, 0.0,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private (gobject_class, sizeof (JammoPlayingTrackPrivate));
}

static void jammo_playing_track_init(JammoPlayingTrack* playing_track) {
	playing_track->priv = G_TYPE_INSTANCE_GET_PRIVATE(playing_track, JAMMO_TYPE_PLAYING_TRACK, JammoPlayingTrackPrivate);
}

