/**sequencer_loop.c is part of JamMo.
 License: GPLv2, read more from COPYING

 This file is for clutter based gui.
 This is part of the sequencer.
 */
#include <glib-object.h>
#include <clutter/clutter.h>
#include <string.h>
#include <stdlib.h>
#include <tangle.h>

#include "sequencer_loop.h"
#include "sequencer.h"
#include "../jammo-sample-button.h"
#include "../jammo.h" //get_id


//Static pointers for loop rolls
static ClutterActor* rhytmical_sample_looper;
static ClutterActor* melodical_sample_looper;
static ClutterActor* harmonical_sample_looper;
static ClutterActor* effect_sample_looper;



gboolean sequencer_lock_pressed (TangleButton* tanglebutton, gpointer data){

	const gchar* name = clutter_actor_get_name(CLUTTER_ACTOR(tanglebutton));
	printf("lock '%s' pressed\n",name);
	ClutterActor* scroller = NULL;
	if (strncmp(name,"rhytm",5)==0)
		scroller = jammo_get_actor_by_id("fullsequencer-loop-rhytmical-wheel-scroller");
	else if (strncmp(name,"melod",5)==0)
		scroller = jammo_get_actor_by_id("fullsequencer-loop-melodical-wheel-scroller");
	else if (strncmp(name,"harmo",5)==0)
		scroller = jammo_get_actor_by_id("fullsequencer-loop-harmonical-wheel-scroller");
	else if (strncmp(name,"effec",5)==0)
		scroller = jammo_get_actor_by_id("fullsequencer-loop-effect-wheel-scroller");

	else
		printf("can't detect which lock pressed\n");

	if (scroller)  {
		gfloat y = tangle_scrolling_actor_get_scrolling_threshold_y(TANGLE_SCROLLING_ACTOR(scroller));
		if (y==10000.0)
			tangle_scrolling_actor_set_scrolling_threshold_y(TANGLE_SCROLLING_ACTOR(scroller),30.0);//TODO: use original default value
		else
			tangle_scrolling_actor_set_scrolling_threshold_y(TANGLE_SCROLLING_ACTOR(scroller),10000.0);
	}
	return TRUE;
}


/*
Set dragging_threshold_y  of SampleButtons to MAXFLOAT. TODO: use template for them.
ScrollingActor doesn't have clampping. This will set it.
*/
static void fine_tune_wheel (ClutterActor* sample_looper) {

	GList* children;
	for (children = tangle_widget_get_children_readonly(TANGLE_WIDGET(sample_looper)); children; children = children->next) {
	tangle_draggable_actor_set_dragging_threshold_y (TANGLE_DRAGGABLE_ACTOR(children->data), G_MAXFLOAT); //DISABLE starting dragging vertically
	}


	ClutterActor* scrolling_actor = clutter_actor_get_parent(sample_looper);
	g_signal_connect_swapped(scrolling_actor, "clamp-scrolling-offset-y", G_CALLBACK(tangle_widget_clamp_child_boundaries), sample_looper);
}


#define ABS_F(x) ((x) < 0 ? -(x) : (x))
/*
Returns JammoSample closest to center of wheel. If wheel is rolling and two sample are equal
some of them are returned.
(This function can be used when wheel is running but there are now reason to do that).
Will return NULL in error cases
*/
static JammoSample* get_centermost_sample_from_loop_roll(ClutterActor* wheel) {
	gfloat y;
	gfloat offset=0; //How much scroller is scrolled
	gfloat height;    //Height of the scroller (we take this from color_box behind samples)

	//Wheels first actor is scroller
	ClutterActor* scroller = clutter_actor_get_parent(wheel);
	if (TANGLE_IS_SCROLLING_ACTOR(scroller)){
		offset = tangle_scrolling_actor_get_scrolling_offset_y(TANGLE_SCROLLING_ACTOR(scroller));
		offset+=95; //height of the sample buttons. (we are looking sample in SECOND slot)
		//printf("offset = %f\n",offset);
	}
	else {
		printf("scroller was not parent of wheel\n");
		return NULL;
	}

	TangleWidget* container = TANGLE_WIDGET(tangle_wrapper_actor_get_wrapped(TANGLE_WRAPPER_ACTOR(scroller)));

	//Check height of the wheel
	ClutterActor* color_box = tangle_widget_get_nth_child(container,0);
	height = clutter_actor_get_height(color_box);
	//printf("height = %f\n",height);

	//Check all sample button which one is closest to center (='offset')
	float best_value = height;
	JammoSampleButton* closest = NULL;

	GList* children;
	for (children = tangle_widget_get_children_readonly(container); children; children = children->next) {
		if (!JAMMO_IS_SAMPLE_BUTTON(children->data)){
			//printf("this is not sample_button\n");  //Color_box triggers this at least
			continue;
		}

		y = clutter_actor_get_y (children->data);
		if (y==0) y+=height; //actor in first slot should be consider be in after very last (+height)

		if (ABS_F(y-offset) < best_value) {
			best_value = ABS_F(y-offset);
			closest = JAMMO_SAMPLE_BUTTON(children->data);
		}

	}
	if (closest==NULL) return NULL;
	return jammo_sample_button_get_sample(closest);
}


/***************************
Functions for json
**************************/


/*
 * plays all samples from centerline of loop rolls
 */
void fullsequencer_loop_arrow_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	//get_centermost_sample_from_loop_roll can return NULL (in error cases)
	JammoSample* r = get_centermost_sample_from_loop_roll(rhytmical_sample_looper);
	JammoSample* e = get_centermost_sample_from_loop_roll(effect_sample_looper);
	JammoSample* m = get_centermost_sample_from_loop_roll(melodical_sample_looper);
	JammoSample* h = get_centermost_sample_from_loop_roll(harmonical_sample_looper);

	//These play very same time even without adhoc sequencer and tracks.
	if (r!=NULL)   jammo_sample_play(r);
	if (e!=NULL)   jammo_sample_play(e);
	if (m!=NULL)   jammo_sample_play(m);
	if (h!=NULL)   jammo_sample_play(h);
}

/*
amount=how many slots.
If loop is already rolling, this will abort it and will make own.
Loop will still snap to grid always.
*/
static void rotate_loop_for_amount(ClutterActor* wheel, gint amount) {
	ClutterActor* scroller = clutter_actor_get_parent(wheel);
	if (TANGLE_IS_SCROLLING_ACTOR(scroller)){
		//Check if lock is pressed
		gfloat threshold_y = tangle_scrolling_actor_get_scrolling_threshold_y(TANGLE_SCROLLING_ACTOR(scroller));
		if (threshold_y==10000.0) //means rolling is disabled for this wheel 
			return;
		gfloat offset = tangle_scrolling_actor_get_scrolling_offset_y(TANGLE_SCROLLING_ACTOR(scroller));
		int current_slot = (int)(offset)/95; //round to lower slot, if rolling already
		gfloat target = (current_slot+amount) * 95.0;
		//printf("current_slot = %d, target pixel='%f'\n",current_slot,target);
		clutter_actor_animate(CLUTTER_ACTOR(scroller), CLUTTER_EASE_IN_OUT_QUAD, 550, "scrolling-offset-y", target, NULL);
	}
	else {
		printf("scroller was not parent of wheel\n");
	}
}


/*
There are not yet logic for 'Cannot Fail'. It just turn them one slot forward. TODO
*/
void fullsequencer_loop_cannot_fail_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	rotate_loop_for_amount(rhytmical_sample_looper, 1);
	rotate_loop_for_amount(melodical_sample_looper, 1);
	rotate_loop_for_amount(harmonical_sample_looper, 1);
	rotate_loop_for_amount(effect_sample_looper, 1);
}



void fullsequencer_loop_crazy_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	GRand *random_gen = g_rand_new();

	gint rot_rhyt = g_rand_int_range(random_gen, -30, 30);
	gint rot_melo = g_rand_int_range(random_gen, -30, 30);
	gint rot_harm = g_rand_int_range(random_gen, -30, 30);
	gint rot_effe = g_rand_int_range(random_gen, -30, 30);

	rotate_loop_for_amount(rhytmical_sample_looper, rot_rhyt);
	rotate_loop_for_amount(melodical_sample_looper, rot_melo);
	rotate_loop_for_amount(harmonical_sample_looper, rot_harm);
	rotate_loop_for_amount(effect_sample_looper, rot_effe);

	g_rand_free(random_gen);
}

void fullsequencer_loop_handle_clicked (TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties){
	printf("fullsequencer_loop_handle_clicked: not implemented yet\n");
}

//////////////////////////////////////

//Bottom of the loop-view there are dropping-area, and when samplebutton is dragged here, view is changed.
void sequencer_handle_dragging (TangleDroppableActor *droppable_actor, TangleDragging *dragging){
	sequencer_change_to_sequencer_view(NULL,NULL);
}


void sequencer_loop_tune_wheels() {
	//Disable scrolling, use only arrow-buttons
	g_object_set(jammo_get_actor_by_id("fullsequencer-scroller"), "scrolling-threshold-y" , G_MAXFLOAT, "scrolling-threshold-x" , G_MAXFLOAT, NULL);

	//Static
	rhytmical_sample_looper  =  jammo_get_actor_by_id("fullsequencer-loop-rhytmical-samples");
	melodical_sample_looper  =  jammo_get_actor_by_id("fullsequencer-loop-melodical-samples");
	harmonical_sample_looper =  jammo_get_actor_by_id("fullsequencer-loop-harmonical-samples");
	effect_sample_looper     =  jammo_get_actor_by_id("fullsequencer-loop-effect-samples");

	fine_tune_wheel(rhytmical_sample_looper);
	fine_tune_wheel(melodical_sample_looper);
	fine_tune_wheel(harmonical_sample_looper);
	fine_tune_wheel(effect_sample_looper);
}
