/*
 * jammo-playing-track.c
 *
 * This file is part of JamMo.
 *
 * (c) 2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */
 
#include "jammo-playing-track.h"
#include "jammo-meam-private.h"
#include <string.h>

G_DEFINE_TYPE(JammoPlayingTrack, jammo_playing_track, JAMMO_TYPE_TRACK);

enum {
	PROP_0,
	PROP_FILENAME
};

struct _JammoPlayingTrackPrivate {
	GstElement* bin;
	gchar* filename;
};

static void on_new_decoded_pad(GstElement* element, GstPad* pad, gboolean last, gpointer data);

JammoPlayingTrack* jammo_playing_track_new(const gchar* filename) {

	return JAMMO_PLAYING_TRACK(g_object_new(JAMMO_TYPE_PLAYING_TRACK, "filename", filename, NULL));
}


static void jammo_playing_track_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* pspec) {
	JammoPlayingTrack* playing_track;
	
	playing_track = JAMMO_PLAYING_TRACK(object);

	switch (prop_id) {
		case PROP_FILENAME:
			g_assert(!playing_track->priv->filename);
			playing_track->priv->filename = g_strdup(g_value_get_string(value));
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
			break;
	}
}

static void jammo_playing_track_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* pspec) {
        JammoPlayingTrack* playing_track;

	playing_track = JAMMO_PLAYING_TRACK(object);

        switch (prop_id) {
		case PROP_FILENAME:
			g_value_set_string(value, playing_track->priv->filename);
			break;
	        default:
		        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		        break;
        }
}

static guint64 jammo_playing_track_get_duration(JammoTrack* track) {
	guint64 duration = 0;

	/* TODO: Get real duration. */

	return duration;
}

static GstElement* jammo_playing_track_get_element(JammoTrack* track) {

	return JAMMO_PLAYING_TRACK(track)->priv->bin;
}

static GObject* jammo_playing_track_constructor(GType type, guint n_properties, GObjectConstructParam* properties) {
	GObject* object;
	JammoPlayingTrack* playing_track;
	GstElement* source;
	GstElement* decodebin;
	GstElement* convert;
	GstPad* pad;
	
	object = G_OBJECT_CLASS(jammo_playing_track_parent_class)->constructor(type, n_properties, properties);

	playing_track = JAMMO_PLAYING_TRACK(object);

	g_assert(playing_track->priv->filename);

	playing_track->priv->bin = gst_bin_new(NULL);

	source = gst_element_factory_make("filesrc", NULL);
	g_object_set(source, "location", playing_track->priv->filename, NULL);

	decodebin = gst_element_factory_make("decodebin", NULL);
	convert = gst_element_factory_make("audioconvert", NULL);
	g_signal_connect(decodebin, "new-decoded-pad", G_CALLBACK(on_new_decoded_pad), convert);

	gst_bin_add_many(GST_BIN(playing_track->priv->bin), source, decodebin, convert, NULL);
	gst_element_link(source, decodebin);

	pad = gst_element_get_pad(convert, "src");
	gst_element_add_pad(playing_track->priv->bin, gst_ghost_pad_new("src", pad));
	gst_object_unref(pad);

	return object;
}


static void jammo_playing_track_finalize(GObject* object) {
	JammoPlayingTrack* playing_track;
	
	playing_track = JAMMO_PLAYING_TRACK(object);
	g_free(playing_track->priv->filename);

	G_OBJECT_CLASS(jammo_playing_track_parent_class)->finalize(object);
}

static void jammo_playing_track_dispose(GObject* object) {
	G_OBJECT_CLASS(jammo_playing_track_parent_class)->dispose(object);
}

static void jammo_playing_track_class_init(JammoPlayingTrackClass* playing_track_class) {
	GObjectClass* gobject_class = G_OBJECT_CLASS(playing_track_class);
	JammoTrackClass* track_class = JAMMO_TRACK_CLASS(playing_track_class);

	gobject_class->constructor = jammo_playing_track_constructor;
	gobject_class->finalize = jammo_playing_track_finalize;
	gobject_class->dispose = jammo_playing_track_dispose;
	gobject_class->set_property = jammo_playing_track_set_property;
	gobject_class->get_property = jammo_playing_track_get_property;

	track_class->get_duration = jammo_playing_track_get_duration;
	track_class->get_element = jammo_playing_track_get_element;

	/**
	 * JammoPlayingTrack:filename:
	 */
	g_object_class_install_property(gobject_class, PROP_FILENAME,
	                                g_param_spec_string("filename",
	                                "File name",
	                                "The file name of the played audio",
	                                NULL,
	                                G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK |G_PARAM_STATIC_BLURB));

	g_type_class_add_private(gobject_class, sizeof(JammoPlayingTrackPrivate));
}

static void jammo_playing_track_init(JammoPlayingTrack* playing_track) {
	playing_track->priv = G_TYPE_INSTANCE_GET_PRIVATE(playing_track, JAMMO_TYPE_PLAYING_TRACK, JammoPlayingTrackPrivate);
}

static void on_new_decoded_pad(GstElement* element, GstPad* pad, gboolean last, gpointer data) {
	GstElement* sink;
	GstPad* sinkpad;
	GstCaps* caps;

	sink = GST_ELEMENT(data);
	sinkpad = gst_element_get_pad(sink, "sink");
	if (!GST_PAD_IS_LINKED(sinkpad)) {
		caps = gst_pad_get_caps(pad);
		if (g_strrstr(gst_structure_get_name(gst_caps_get_structure(caps, 0)), "audio")) {
			gst_pad_link(pad, sinkpad);
		}
		gst_caps_unref(caps);
	}
	g_object_unref(sinkpad);
}
