/*
 * jammo.c (=> main.c)
 *
 * This file is part of JamMo.
 *
 * (c) 2009-2010 University of Oulu
 *
 * Authors: Henrik Hedberg <henrik.hedberg@oulu.fi>
 */

#include "jammo-chum.h"
#include "../meam/jammo-meam.h"
#include "jammo-mentor.h"
#include "jammo-mentor-action.h"
#include <tangle.h>
#include <clutter/clutter.h>
#include <stdlib.h> 

#include "jammo-mentor.h"
#include "chum.h" //e.g set_easy_game
#include <fcntl.h> //we check some files

#include <string.h>


#include "../gui_clutter/midi_editor.h"

#include "../gui_clutter/sequencer.h"
#include "../gui_clutter/startmenu.h"
#include "../gui_clutter/communitymenu.h"
#include "../gui_clutter/gamesmenu.h"

#ifdef N900
#include <libosso.h>
#endif


static const char* search_path = "/opt/jammo";

/*
 * Note! You need Tangle Toolkit that is available here:
 *
 * http://gitorious.org/tangle
 *
 * To compile:
 * gcc -rdynamic -o jammo jammo.c jammo-*.c $(pkg-config --cflags --libs clutter-1.0 tangle gstreamer-0.10 gstreamer-base-0.10 gstreamer-controller-0.10) ../meam/libmeam.a
 *
 * Note that -rdynamic is important in order to dynamic signal handler binding to work!
 */

static ClutterScript* script;

ClutterActor* jammo_get_actor_by_id(const char* id) {
	ClutterActor* actor = NULL;

	if (!(actor = CLUTTER_ACTOR(clutter_script_get_object(script, id)))) {
		g_warning("Actor '%s' not found.", id);
	}
	
	return actor;
}

/* A TangleStylesheet directory is tried to load from the following locations:
 * 	/opt/jammo/stylesheet
 * 	<current_working_directory>/data/stylesheet
 */
static TangleStylesheet* get_jammo_stylesheet(void) {
	TangleStylesheet* stylesheet;
	gchar* cwd;
	gchar* filename;
	
	cwd = g_get_current_dir();
	filename = g_build_filename(cwd, "data", "stylesheet", NULL);

	if (g_file_test(filename, G_FILE_TEST_EXISTS)) {
		stylesheet = tangle_stylesheet_new_from_file(filename);
	} else {
		stylesheet = tangle_stylesheet_new_from_file("/opt/jammo/stylesheet");
	}

	g_free(cwd);
	g_free(filename);

	return stylesheet;
}

/* JSON files and content within those are searched from the following locations:
 * 	/opt/jammo
 * 	<current_working_directory>/data/jammo
 */
static void add_search_paths(ClutterScript* script) {
	const gchar* default_search_path = "/opt/jammo";
	gchar* development_search_path;
	gchar* cwd;
	const gchar* search_paths[2];

	cwd = g_get_current_dir();
	development_search_path = g_build_filename(cwd, "data", "jammo", NULL);

	search_paths[0] = development_search_path;
	search_paths[1] = default_search_path;
	clutter_script_add_search_paths(script, search_paths, G_N_ELEMENTS(search_paths));

	g_free(cwd);
	g_free(development_search_path);
}

#ifdef N900
static gboolean pause_display_blanking(gpointer user_data) {
	osso_context_t* osso_context;
	
	osso_context = (osso_context_t*)user_data;
	
	g_return_val_if_fail(osso_display_blanking_pause(osso_context) == OSSO_OK, FALSE);
	
	return TRUE;
}
#endif

/* Be sure there are needed place to save compositions and singings.
 */
static void create_directories(void) {
	gchar* cmd; 
	
	cmd= g_strdup_printf("mkdir -p %s",COMPOSITIONS_DIRECTORY);
	if (system(cmd)) {
		printf("Error, can't call '%s'\n", cmd);
	}
	g_free(cmd);
	
	cmd = g_strdup_printf("mkdir -p %s",SINGINGS_DIRECTORY);
	if (system(cmd)) {
		printf("Error, can't call '%s'\n", cmd);
	}
	g_free(cmd);
}

int main(int argc, char** argv) {
#ifdef N900
	osso_context_t* osso_context;
#endif

	TangleStylesheet* stylesheet;
	gchar* filename;
	ClutterActor* stage;
	GError* error = NULL;
	GObject* object;
	ClutterActor* actor;
	ClutterColor black = { 0, 0, 0, 255 };
	
	jammo_meam_init(&argc, &argv);
	jammo_chum_init(&argc, &argv);
	srand (time (NULL)); //We will use rand()

#ifdef N900
	osso_context = osso_initialize("jammo", "git", 0, NULL);
	g_assert(osso_context != NULL);
	
	osso_display_state_on(osso_context);
	osso_display_blanking_pause(osso_context);
	
	g_timeout_add(55000, pause_display_blanking, osso_context);
#endif

	create_directories();

	/* TODO: Temporary: make sure Mentor classes exists. */
	printf("%p %p\n", JAMMO_TYPE_MENTOR, JAMMO_TYPE_MENTOR_ACTION);

	if ((stylesheet = get_jammo_stylesheet())) {
		tangle_stylesheet_set_default(stylesheet);
	}

	script = clutter_script_new();
	add_search_paths(script);
	if (!(filename = clutter_script_lookup_filename(script, "jammo.json"))) {
		g_critical("File 'jammo.json' not found.");
		
		return 1;
	}
	if (!clutter_script_load_from_file(script, filename, &error)) {
		g_warning("Could not load file '%s': %s", filename, error->message);

		return 2;
	}
	
	if (!(object = clutter_script_get_object(script, "main"))) {
		g_warning("Object 'main' not found from the file 'jammo.json'.");
		
		return 3;
	}
	
	if (!CLUTTER_IS_ACTOR(object)) {
		g_warning("Object 'main' is not ClutterActor in the file 'jammo.json'.");
		
		return 4;
	}
	
	actor = CLUTTER_ACTOR(object);
	
	stage = clutter_stage_get_default();
	clutter_stage_set_title(CLUTTER_STAGE(stage), "JamMo");
	clutter_stage_set_color(CLUTTER_STAGE(stage), &black);
	clutter_stage_set_key_focus(CLUTTER_STAGE(stage), NULL);

#ifdef N900
	clutter_stage_set_fullscreen(CLUTTER_STAGE(stage), TRUE);
#else
	clutter_actor_set_size(stage, 800, 480);
#endif

	clutter_container_add_actor(CLUTTER_CONTAINER(stage), actor);
	
	clutter_actor_show(stage);
	
 if (argc==2 && strcmp(argv[1],"editor")==0) { 
	start_midi_editor();
	}
	else if (argc==2 && strcmp(argv[1],"sequencer")==0) {
		start_sequencer();
	}
	else if (argc==2 && strcmp(argv[1],"startmenu")==0) {
		start_startmenu();
	}
	else if (argc==2 && strcmp(argv[1],"gamesmenu")==0) {
		start_gamesmenu();
	}
	else if (argc==2 && strcmp(argv[1],"communitymenu")==0) {
		start_communitymenu();
	}
	
	clutter_main();

	return 0;
}

void jammo_easy_changed(TangleAction *action, GObject *source, const gchar *trigger, TangleProperties *properties) {
	ClutterActor* ct =CLUTTER_ACTOR(source);
	const gchar* name = clutter_actor_get_name(ct);
	printf("easy/advanced clicked. get_name: '%s'\n",name);

	if (strcmp(name,"easy")==0)
		{
		chum_set_easy_game(TRUE);
		jammo_mentor_speak(jammo_mentor_get_default(), "/opt/jammo/mentor_speech/mentor_easy_selected_fi.ogg");
		}
	else if (strcmp(name,"advanced")==0)
		{
		chum_set_easy_game(FALSE);
		jammo_mentor_speak(jammo_mentor_get_default(), "/opt/jammo/mentor_speech/mentor_advanced_selected_fi.ogg");
		}

	ClutterActor* button;
	ClutterActor* texture;
	gchar *new_name = g_strdup_printf("main-menu-easy-selection-%s-texture", name);
 
	button = jammo_get_actor_by_id("main-menu-easy-selection");
	texture = jammo_get_actor_by_id(new_name);
	tangle_button_set_normal_background_actor(TANGLE_BUTTON(button), texture);

}


