/**sample.c is part of JamMo.
License: GPLv2, read more from COPYING

Sample is one sound, e.g. one wav-file. It must be in some track.
*/

#include "sample.h"
#include "sequencer.h"

//private functions
static void update_pipelines_end_time(guint64 sample_end);
static void put_sample_to_slot(int slot,Sample *sample, guint track_number);
static Sample *remove_sample_from_slot(int slot, guint track_number);
/**
 *  add_new_sample - function for adding sample to the specific track and specific slot
 *  @param track_number - a track in which the sample will be added
 *  @param srcLocation - Location (path) of the sample
 *  @param slot - Slot for this sample
 *  @return Sample - Returns that just created sample. Or NULL if failed.
 */ 
Sample *add_new_sample_to_track_nth_slot(int track_number, const gchar srcLocation[], guint slot) 
{
//slot must be on boundaries.
if (slot<0) slot=0;
if (slot>SLOT_LIMIT-1) slot=SLOT_LIMIT-1;


//Count starting point of this sample:
//60s / tempo * meter = lenght of one slot.
//because we are interested in times in nanoseconds, we multiple it by 1000000000
//and divide should be last operation because we are handling integers
//guint64 starting_time = slot * 60000000000 * GST_NSECOND * get_nth_track_from_pipeline(track_number)->meter /get_nth_track_from_pipeline(track_number)->tempo;
//-> On maemo: warning: integer constant is too large for "long" type

guint64 a0 = 60000;
guint64 a1= 1000000;
guint64 a2=a0*a1;
guint64 starting_time = slot * a2 * GST_NSECOND * get_nth_track_from_pipeline(track_number)->meter /get_nth_track_from_pipeline(track_number)->tempo;



/* //this is good way to debug how many zerous are missing
int i=1;
for (i=1;i<10000000;i=i*10)
  printf("to_track %d (%d,%s,%" GST_TIME_FORMAT ")\n",i,track_number, srcLocation, GST_TIME_ARGS (starting_time*i));
*/

Sample *new_sample= add_new_sample_to_track( track_number, srcLocation, starting_time);
put_sample_to_slot(slot,new_sample,track_number);

return new_sample;
}

/*This is removing-function which is visible to outside*/
void empty_track_from_slot(guint track_number,int slot) {
Sample *removed =remove_sample_from_slot(slot,track_number);
if (removed!=NULL)
  remove_sample_from_track(removed);
}


/**
 *  Removing sample from the specific track and from specific slot.
 *  @param slot - Slot which are part of this sample
 *  @param track_number - a track 
 *  @return Sample - Returns that just removed sample. Or NULL if failed.
 */ 
/*any part of the sample can be in this slot, and whole sample is removed.
if slot is already empty, nothing happens.*/
static Sample *remove_sample_from_slot(int slot, guint track_number){
    struct Track *trc=get_nth_track_from_pipeline(track_number);
    Sample *moved_sample=trc->slot_info[slot]->sample;
    if (moved_sample==NULL)
      return NULL;

    int old_value=trc->slot_info[slot]->info;
    //first number is its length, second is this parts number
    int len=old_value/10;
    int pos=old_value-(len*10);
    int j;
    //printf("len=%d, pos=%d\n",len,pos);
    for (j=0;j<len;j++)	{
      trc->slot_info[slot-(pos-1)+j]->info=0;
      trc->slot_info[slot-(pos-1)+j]->sample=NULL;
    }

return moved_sample;
}


//read more about slot_info from trac.h 
static void put_sample_to_slot(int slot,Sample *sample, guint track_number){
//printf("put_sample_to_slot called slot:%d . max=%d\n",slot,SLOT_LIMIT);

  struct Track *trc=get_nth_track_from_pipeline(track_number);
  //Count how many slots this sample takes
  guint64 length64 = sample->duration;
  //printf("duration of sample %" GST_TIME_FORMAT ")\n",GST_TIME_ARGS (length64));

  float sslot = (length64+0.0) / ( 60000000000.0 * GST_NSECOND * get_nth_track_from_pipeline(track_number)->meter /(trc->tempo+0.0));
  //printf("this sample takes exactly %f slots\n",sslot);

  //rounding might be better way
  int length=(int)sslot;

  //even sound is short, it must still take one slot.
  if (length==0) 
      length=1;

  //We can run out of boundaries without this, but failure has happened.
  if (slot+length>SLOT_LIMIT){
    printf("Not enough free slots\n");
    return;
  }

  int i;
  /*new_sample covers one or more slots, we mark it on every slot.*/
  for (i=0;i<length;i++)
    {
    int old_value=trc->slot_info[slot+i]->info;
    if (old_value!=0)	{ //there are something in way
	//remove it...
	Sample *moved_sample=remove_sample_from_slot(slot+i,track_number);
	//...and put it back.
	put_sample_to_slot(slot+length,moved_sample,track_number);
    }
    trc->slot_info[slot+i]->info=length*10+i+1;
    trc->slot_info[slot+i]->sample=sample;
  }
}


/**
 *  add_new_sample - function for adding sample to the specific track
 *  @param track_number - a track in which the sample will be added
 *  @param srcLocation - Location (path) of the sample
 *  @param startTime - Samples start time in nanoseconds on track 
 *  @return Sample - Returns that just created sample. Or NULL if failed.
 *
 * http://gstreamer.freedesktop.org/data/doc/gstreamer/head/gnonlin/html/GnlFileSource.html
 *
 */ 
Sample *add_new_sample_to_track(int track_number, const gchar srcLocation[], guint64 startTime) 
{

struct Track *trc=get_nth_track_from_pipeline(track_number);

	Sample *sample = (Sample*) malloc(sizeof(Sample));
	if(sample != NULL)
	{
		gchar audiosrc_nbr[13];		

		sample->parent = (Track*)trc;

		guint64 duration = pre_roll((gpointer)srcLocation);

		// Add audiosource to composition and set its values
		sprintf(audiosrc_nbr, "audiosrc_%d", g_list_length(trc->samples)+1);
		sample->audiosource = gst_element_factory_make("gnlfilesource", NULL);
		g_object_set(G_OBJECT(sample->audiosource), "location", srcLocation, NULL);
		g_object_set(G_OBJECT(sample->audiosource), "start", startTime, NULL);
		g_object_set(G_OBJECT(sample->audiosource), "duration", duration, NULL);
		g_object_set(G_OBJECT(sample->audiosource), "media-start", 0 * GST_SECOND, NULL);
		g_object_set(G_OBJECT(sample->audiosource), "media-duration", duration, NULL);
		g_object_set(G_OBJECT(sample->audiosource), "priority", 0, NULL);
		gst_bin_add(GST_BIN(trc->composition), sample->audiosource);		

		sample->duration = duration;
		sample->start_time = startTime;
		sample->end_time = sample->start_time + sample->duration;
		sprintf(sample->source, "%s", srcLocation);

		g_print("  Sample source: %s\n  Start time: %" GST_TIME_FORMAT "\n  End time: %" GST_TIME_FORMAT "\n",
			sample->source, 
			GST_TIME_ARGS (sample->start_time),
			GST_TIME_ARGS (sample->end_time)
			);		

		trc->samples = g_list_append(trc->samples, sample);

		update_pipelines_end_time(sample->end_time);
	}
	return sample;
}

/**
 *  remove_sample_from_track - Function for removing selected sample
 *  Note that the pipelines state should be set to NULL before removing sample.
 *  This feature isn't coded inside the function because it would cause some 
 *  unwanted side effects. So setting the pipelines state to NULL is left to the
 *  coders responsibility.
 *  @param smp - a pointer to the sample which will be removed
 */
void remove_sample_from_track(Sample *smp)
{
//gst_element_set_state (get_mainPipeline()->pipeline, GST_STATE_NULL);
	Track *trc = smp->parent;
	
	// Remove sample from the tracks composition element
	gst_bin_remove(GST_BIN(trc->composition), smp->audiosource);
	//printf("audio-source removed\n");
	// Remove sample from the tracks sample list
	trc->samples = g_list_remove(trc->samples, smp);
	// Free the memory allocated to the sample
	free_sample(smp);
}

/**
 *  free_sample - function for freeing the allocated memory of sample elements
 *  @param smp - a pointer to the sample which will be freed
 */
void free_sample(Sample *smp)
{
	g_print("  Freeing sample...\n");
	free(smp);
	smp = NULL;
	g_print("\t-> Sample freed\n");
}

/**
 *  set_samples_startTime - function for changing samples start time
 *  @param smp - a pointer to the sample which start time will be changed
 *  @param start - samples new start time (guint64)
 */ 
void set_samples_startTime(Sample *smp, guint64 startTime)
{
	if (smp==NULL)
	    return;
	// Set samples new start time
	g_object_set(G_OBJECT(smp->audiosource), "start", startTime , NULL);

	// Update samples properties
	smp->start_time = startTime ;
	smp->end_time = smp->start_time + smp->duration;
	
	update_pipelines_end_time(smp->end_time);	
}

/**
 *  update_pipelines_end_time - Function updates the pipelines end time if its
 *  current end time is smaller than the lastly added samples end time.
 *  @param sample_end - an end time of the newly added sample
 */
static void update_pipelines_end_time(guint64 sample_end)
{
	if (sample_end > get_mainPipeline()->pipelines_end_time) {
		get_mainPipeline()->pipelines_end_time = sample_end; 
		g_print("End time updated\n");
		g_print("New end time is %" GST_TIME_FORMAT "\n",
			GST_TIME_ARGS (get_mainPipeline()->pipelines_end_time));
	}
	else {
		g_print("End time not updated!\nPipelines end time is still %" GST_TIME_FORMAT "\n",
			GST_TIME_ARGS (get_mainPipeline()->pipelines_end_time));
	}
}
