/**duration.c is part of JamMo.
License: GPLv2, read more from COPYING

This handles our "preroll" -functionality when adding new samples to tracks. Gstreamer term "preroll" means that a sample is basically played/initialized to another pipeline which tells us the total length/duration of the sample. Preroll is a must functionality to us, because of the use of Gnonlin. In gnonlin all the playtimes (durations) has to be manually set and we must know how long the samples are (samples durations). When prerolling the samples we get to know those durations. We have made a request to the Gnonlin devolopers that in future Gnonlin would do this preroll automatically, but at this stage it has to be done manually.
*/

/**
From http://gstreamer.freedesktop.org/data/doc/gstreamer/head/gstreamer-libs/html/GstBaseSink.html
"[...] It will return GST_STATE_CHANGE_ASYNC from a state change to PAUSED until the first buffer arrives in this element." -> When message 'GST_MESSAGE_ASYNC_DONE' arrives, sample is ready to play and we can stop it.
*/
#include "duration.h"
/* Variables */
GMainLoop *loo;
GstElement *pip, *audi;
gint64 sample_length;
gboolean ready;

/**
* bus_callback - Basic bus callback function to catch messages from the pipeline.
*/
static gboolean bus_callback (GstBus *bu, GstMessage *ms, gpointer    dat)
{
  //g_print ("Got %s message\n", GST_MESSAGE_TYPE_NAME (ms));
  //GMainLoop
  loo = (GMainLoop *) dat;

  switch (GST_MESSAGE_TYPE (ms)) {

    case GST_MESSAGE_ASYNC_DONE:
	    ; GstFormat fmtv = GST_FORMAT_TIME;
	    gst_element_query_duration (pip, &fmtv, &sample_length);
	    //printf("*DURATION of sample counted\n");
	    ready=TRUE;
	    g_main_loop_quit (loo);
      break;
    case GST_MESSAGE_EOS:
      g_print ("\nEnd of stream\n");
      g_main_loop_quit (loo);
      break;

    case GST_MESSAGE_ERROR: {
      gchar  *debu;
      GError *erro;

      gst_message_parse_error (ms, &erro, &debu);
      g_free (debu);

      g_printerr ("Error: %s\n", erro->message);
      g_error_free (erro);

      g_main_loop_quit (loo);
      break;
    }
    default:
      break;
  }

  return TRUE;
}


/**
* newpad - Handle the linking/playing of our element.
*/
static void
newpad (GstElement *decodebi,
	   GstPad     *pa,
	   gboolean    las,
	   gpointer    dat)
{
  GstCaps *cap;
  GstStructure *st;
  GstPad *audiopa;

  /* only link once */
  audiopa = gst_element_get_pad (audi, "sink");
  if (GST_PAD_IS_LINKED (audiopa)) {
    g_object_unref (audiopa);
    return;
  }

  /* check media type */
  cap = gst_pad_get_caps (pa);
  st = gst_caps_get_structure (cap, 0);
  if (!g_strrstr (gst_structure_get_name (st), "audio")) {
    gst_caps_unref (cap);
    gst_object_unref (audiopa);
    return;
  }
  gst_caps_unref (cap);

  /* link'n'play */
  gst_pad_link (pa, audiopa);
}


/**
* Main function in this file, does all the necessary gstreamer initializations.
@param location gets filepath of the wanted sample as parameter.
*/
guint64
pre_roll (gchar *location[])
{
  //GMainLoop *loop;
  GstElement *sr, *de, *con, *sin;
  GstPad *audiopa;
  GstBus *bu;

  /* init GStreamer */
  loo = g_main_loop_new (NULL, FALSE);

  /* setup */
  pip = gst_pipeline_new ("pipe");

  bu = gst_pipeline_get_bus (GST_PIPELINE (pip));
  gst_bus_add_watch (bu, bus_callback, loo);
  gst_object_unref (bu);

  sr = gst_element_factory_make ("filesrc", "source");
  g_object_set (G_OBJECT (sr), "location", location, NULL);
  de = gst_element_factory_make ("decodebin", "decoder");
  g_signal_connect (de, "new-decoded-pad", G_CALLBACK (newpad), NULL);
  gst_bin_add_many (GST_BIN (pip), sr, de, NULL);
  gst_element_link (sr, de);

  /* create audio output */
  audi = gst_bin_new ("audiobin");
  con = gst_element_factory_make ("audioconvert", "con");
  audiopa = gst_element_get_pad (con, "sink");
  sin = gst_element_factory_make ("fakesink", "sin"); //Fakesink!
  gst_bin_add_many (GST_BIN (audi), con, sin, NULL);
  if (con==NULL)
      printf("pre_roll: converter NULL\n");
  if (sin==NULL)
      printf("pre_roll: sink NULL\n");

  gst_element_link (con, sin);
  gst_element_add_pad (audi,gst_ghost_pad_new ("sink", audiopa));
  gst_object_unref (audiopa);
  gst_bin_add (GST_BIN (pip), audi);

  ready=FALSE;
  gst_element_set_state (pip, GST_STATE_PAUSED); 
  //If machine is fast and sample is small, g_main_loop is quitted before we try start it.
  // -> we use gboolean ready

  if (!ready)
    g_main_loop_run (loo);

  /* cleanup */
  gst_element_set_state (pip, GST_STATE_NULL);
  gst_object_unref (GST_OBJECT (pip));

  return sample_length;
}
