/* This file is part of JamMo.
License:LGPL 2.1

 */
/**
 * SECTION:element-jammoslider
 *
 * JammoSlider
 */

//This is because of getline
#define _GNU_SOURCE

#define VERSION "0.10.22"
#define PACKAGE "unknown"
#define GST_PACKAGE_NAME "unknown"
#define GST_PACKAGE_ORIGIN "unknown" 

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <gst/controller/gstcontroller.h>

#include "gstjammoslider.h"

/*e.g. Nokia Internet Tablet n810 doesn't have this (maemo diablo)*/
#ifndef G_PARAM_STATIC_STRINGS
#define	G_PARAM_STATIC_STRINGS (G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB)
#endif

#ifndef M_PI
#define M_PI  3.14159265358979323846
#endif

#ifndef M_PI_2
#define M_PI_2  1.57079632679489661923
#endif

#define M_PI_M2 ( M_PI + M_PI )

GST_DEBUG_CATEGORY_STATIC (jammo_slider_debug);
#define GST_CAT_DEFAULT jammo_slider_debug

static const GstElementDetails gst_jammo_slider_details =
GST_ELEMENT_DETAILS ("JamMo Slider",
    "Source/Audio",
    "JamMo Slider",
    "Aapo Rantalainen <aapo.rantalainen@gmail.com>, Mikko Gynther <mikko.gynther@lut.fi>");

#define DEFAULT_SAMPLES_PER_BUFFER   1024
#define DEFAULT_INSTRUMENT           0
#define DEFAULT_VOLUME               0.6
#define DEFAULT_IS_LIVE              FALSE
#define DEFAULT_TIMESTAMP_OFFSET     G_GINT64_CONSTANT (0)
#define DEFAULT_CAN_ACTIVATE_PUSH    TRUE
#define DEFAULT_CAN_ACTIVATE_PULL    FALSE
#define NUMBER_OF_INSTRUMENTS	     sizeof(instruments)/sizeof(*instruments)

//This are for tuning slider
#define DEFAULT_ATTACK 5000
#define DEFAULT_DECAY 5000
#define DEFAULT_SUSTAIN 0.5
#define DEFAULT_RELEASE 5000

enum
{
  PROP_0,
  PROP_SAMPLES_PER_BUFFER,
  PROP_INSTRUMENT,
  PROP_SLIDER_FREQ,
  PROP_SLIDER_STATE,
  PROP_ATTACK,
  PROP_DECAY,
  PROP_SUSTAIN,
  PROP_RELEASE,
  PROP_VOLUME,
  PROP_IS_LIVE,
  PROP_TIMESTAMP_OFFSET,
  PROP_CAN_ACTIVATE_PUSH,
  PROP_CAN_ACTIVATE_PULL,
  PROP_LAST
};

static GstStaticPadTemplate gst_jammo_slider_src_template =
    GST_STATIC_PAD_TEMPLATE ("src",
    GST_PAD_SRC,
    GST_PAD_ALWAYS,
    GST_STATIC_CAPS ("audio/x-raw-int, "
        "endianness = (int) BYTE_ORDER, "
        "signed = (boolean) true, "
        "width = (int) 16, "
        "depth = (int) 16, "
        "rate = (int) [ 1, MAX ], "
        "channels = (int) 1; "
        "audio/x-raw-int, "
        "endianness = (int) BYTE_ORDER, "
        "signed = (boolean) true, "
        "width = (int) 32, "
        "depth = (int) 32,"
        "rate = (int) [ 1, MAX ], "
        "channels = (int) 1; "
        "audio/x-raw-float, "
        "endianness = (int) BYTE_ORDER, "
        "width = (int) { 32, 64 }, "
        "rate = (int) [ 1, MAX ], " "channels = (int) 1")
    );

GST_BOILERPLATE (GstJammoSlider, gst_jammo_slider, GstBaseSrc,
    GST_TYPE_BASE_SRC);


static void gst_jammo_slider_set_property (GObject * object,
    guint prop_id, const GValue * value, GParamSpec * pspec);
static void gst_jammo_slider_get_property (GObject * object,
    guint prop_id, GValue * value, GParamSpec * pspec);

static gboolean gst_jammo_slider_setcaps (GstBaseSrc * basesrc,
    GstCaps * caps);
static void gst_jammo_slider_src_fixate (GstPad * pad, GstCaps * caps);

static gboolean gst_jammo_slider_is_seekable (GstBaseSrc * basesrc);
static gboolean gst_jammo_slider_check_get_range (GstBaseSrc * basesrc);
static gboolean gst_jammo_slider_do_seek (GstBaseSrc * basesrc,
    GstSegment * segment);
static gboolean gst_jammo_slider_query (GstBaseSrc * basesrc,
    GstQuery * query);

static void gst_jammo_slider_get_times (GstBaseSrc * basesrc,
    GstBuffer * buffer, GstClockTime * start, GstClockTime * end);
static gboolean gst_jammo_slider_start (GstBaseSrc * basesrc);
static gboolean gst_jammo_slider_stop (GstBaseSrc * basesrc);
static GstFlowReturn gst_jammo_slider_create (GstBaseSrc * basesrc,
    guint64 offset, guint length, GstBuffer ** buffer);


static void
gst_jammo_slider_base_init (gpointer g_class)
{
  GstElementClass *element_class = GST_ELEMENT_CLASS (g_class);

  gst_element_class_add_pad_template (element_class,
      gst_static_pad_template_get (&gst_jammo_slider_src_template));
  gst_element_class_set_details (element_class, &gst_jammo_slider_details);
}


static gfloat interpolation(gfloat interpolation_point, gfloat value1, gfloat value2) {
        /* linear interpolation */
        return ((1-interpolation_point)*value1 + interpolation_point*value2);
}


/*
Some day we will have three different instrument using slider.
Organ and fm_modulation are only for testing.
*/

/* Very simple organ type sound additive synthesis algorithm. This will be changed
or at least wave should be store in a table instead of multiple sin function calls */
gfloat organ(gdouble phase){
return 0.6*(1 * sin(phase)  + 1.0/6 *sin(phase*2) + 1.0/3*sin(phase*3) + 1.0/4*sin(phase*4)+1.0/3*sin(phase*5)+1.0/6*sin(phase*6)+1.0/6*sin(phase*7)+1.0/4*sin(phase*8));
}


gfloat fm_modulation(gdouble phase){
return ( sinf(phase+0.6*sinf(phase*8.0)) );
}




#define MAXDEL  (2048)       /* MUST be a power of 2. */
#define BITMASK (MAXDEL-1)  /* Circular buffer. */
short    y[MAXDEL];
long     ap=0;
short    n=0;

short karplus_strong (short x, gfloat length) {
  long            a, b;       /* No need to remember a and b. */

  a = (long)x + interpolation(length-(int)length, y[(int)(n - length) & BITMASK],y[(gint)(n - (length+1)) & BITMASK]); /* interpolation */

  //b = ((a + ap) * 127) >> 8;            /* Coeff = +0.49609375. */
    b = ((a + ap) * 127) / (254+length*0.01); //higher coeff on lower tones
    ap = a;                               /* Remember. */
    y[n++ & BITMASK] = (short)b;          /* Write in buffer. */
    return (short)b;
}


gfloat karplus(GstJammoSlider * src){
  gfloat freq = src->slider_freq_slow;
  short inp;
  if (src->adsr_counter < src->samplerate/freq)  /* white noise BURST. */
      inp = (short)((rand() & 32767) - 16384);
  else                                           /* No more input (silence). */
      inp = 0;

return (gfloat) karplus_strong(inp, src->samplerate/freq)/32767;
}

static InstrumentFunc instruments[] = {
  (InstrumentFunc) organ,
  (InstrumentFunc) fm_modulation,
  (InstrumentFunc) karplus
};


static void
gst_jammo_slider_class_init (GstJammoSliderClass * klass)
{
  GObjectClass *gobject_class;
  GstBaseSrcClass *gstbasesrc_class;

  gobject_class = (GObjectClass *) klass;
  gstbasesrc_class = (GstBaseSrcClass *) klass;

  gobject_class->set_property = gst_jammo_slider_set_property;
  gobject_class->get_property = gst_jammo_slider_get_property;

  g_object_class_install_property (gobject_class, PROP_SAMPLES_PER_BUFFER,
      g_param_spec_int ("samplesperbuffer", "Samples per buffer",
          "Number of samples in each outgoing buffer",
          1, G_MAXINT, DEFAULT_SAMPLES_PER_BUFFER,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
  g_object_class_install_property (gobject_class, PROP_INSTRUMENT,
      g_param_spec_int ("instrument", "Virtual Instrument",
          "What instrument is used",
          0, NUMBER_OF_INSTRUMENTS-1, DEFAULT_INSTRUMENT,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
  g_object_class_install_property (gobject_class, PROP_SLIDER_FREQ,
      g_param_spec_float ("slider-freq", "Freq of slider", "Freq of slider",
          32.0,4000,440,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE | G_PARAM_STATIC_STRINGS ));
 g_object_class_install_property (gobject_class, PROP_ATTACK,
      g_param_spec_int ("attack", "attack",
          "attack",
          0, 100000, DEFAULT_ATTACK,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
 g_object_class_install_property (gobject_class, PROP_DECAY,
      g_param_spec_int ("decay", "decay",
          "decay",
          0, 100000, DEFAULT_DECAY,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
 g_object_class_install_property (gobject_class, PROP_RELEASE,
      g_param_spec_int ("release", "release",
          "release",
          0, 100000, DEFAULT_RELEASE,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
 g_object_class_install_property (gobject_class, PROP_SUSTAIN,
      g_param_spec_float ("sustain", "sustain", "sustain",
          0.0,1.0,DEFAULT_SUSTAIN,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE | G_PARAM_STATIC_STRINGS ));
 g_object_class_install_property (gobject_class, PROP_SLIDER_STATE,
      g_param_spec_boolean ("slider-state", "State of slider", "State of slider",
          FALSE,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE | G_PARAM_STATIC_STRINGS ));  
  g_object_class_install_property (gobject_class, PROP_VOLUME,
      g_param_spec_double ("volume", "Volume", "Volume of test signal", 0.0,
          1.0, DEFAULT_VOLUME,
          G_PARAM_READWRITE | GST_PARAM_CONTROLLABLE | G_PARAM_STATIC_STRINGS));
  g_object_class_install_property (gobject_class, PROP_IS_LIVE,
      g_param_spec_boolean ("is-live", "Is Live",
          "Whether to act as a live source", DEFAULT_IS_LIVE,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
  g_object_class_install_property (G_OBJECT_CLASS (klass),
      PROP_TIMESTAMP_OFFSET, g_param_spec_int64 ("timestamp-offset",
          "Timestamp offset",
          "An offset added to timestamps set on buffers (in ns)", G_MININT64,
          G_MAXINT64, DEFAULT_TIMESTAMP_OFFSET,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
  g_object_class_install_property (gobject_class, PROP_CAN_ACTIVATE_PUSH,
      g_param_spec_boolean ("can-activate-push", "Can activate push",
          "Can activate in push mode", DEFAULT_CAN_ACTIVATE_PUSH,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));
  g_object_class_install_property (gobject_class, PROP_CAN_ACTIVATE_PULL,
      g_param_spec_boolean ("can-activate-pull", "Can activate pull",
          "Can activate in pull mode", DEFAULT_CAN_ACTIVATE_PULL,
          G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS));

  gstbasesrc_class->set_caps = GST_DEBUG_FUNCPTR (gst_jammo_slider_setcaps);
  gstbasesrc_class->is_seekable =
      GST_DEBUG_FUNCPTR (gst_jammo_slider_is_seekable);
  gstbasesrc_class->check_get_range =
      GST_DEBUG_FUNCPTR (gst_jammo_slider_check_get_range);
  gstbasesrc_class->do_seek = GST_DEBUG_FUNCPTR (gst_jammo_slider_do_seek);
  gstbasesrc_class->query = GST_DEBUG_FUNCPTR (gst_jammo_slider_query);
  gstbasesrc_class->get_times =
      GST_DEBUG_FUNCPTR (gst_jammo_slider_get_times);
  gstbasesrc_class->start = GST_DEBUG_FUNCPTR (gst_jammo_slider_start);
  gstbasesrc_class->stop = GST_DEBUG_FUNCPTR (gst_jammo_slider_stop);
  gstbasesrc_class->create = GST_DEBUG_FUNCPTR (gst_jammo_slider_create);
}

static void
gst_jammo_slider_init (GstJammoSlider * src, GstJammoSliderClass * g_class)
{
  GstPad *pad = GST_BASE_SRC_PAD (src);

  gst_pad_set_fixatecaps_function (pad, gst_jammo_slider_src_fixate);

  src->samplerate = 44100;
  src->format = GST_JAMMO_SLIDER_FORMAT_NONE;

  src->volume = DEFAULT_VOLUME;
  src->slider_rounding_counter=0;
  src->slider_first_freq=TRUE;

  src->attack  = DEFAULT_ATTACK;
  src->decay   = DEFAULT_DECAY;
  src->sustain = DEFAULT_SUSTAIN;
  src->release = DEFAULT_RELEASE;

  /* we operate in time */
  gst_base_src_set_format (GST_BASE_SRC (src), GST_FORMAT_TIME);
  gst_base_src_set_live (GST_BASE_SRC (src), DEFAULT_IS_LIVE);

  src->samples_per_buffer = DEFAULT_SAMPLES_PER_BUFFER;
  src->generate_samples_per_buffer = src->samples_per_buffer;
  src->timestamp_offset = DEFAULT_TIMESTAMP_OFFSET;
  src->can_activate_pull = DEFAULT_CAN_ACTIVATE_PULL;

// gst_base_src_set_blocksize is since 0.10.22, maemo has only 0.10.13
// This works without this.
//  gst_base_src_set_blocksize (GST_BASE_SRC (src), -1)

}

static void
gst_jammo_slider_src_fixate (GstPad * pad, GstCaps * caps)
{
  GstJammoSlider *src = GST_JAMMO_SLIDER (GST_PAD_PARENT (pad));
  const gchar *name;
  GstStructure *structure;

  structure = gst_caps_get_structure (caps, 0);

  GST_DEBUG_OBJECT (src, "fixating samplerate to %d", src->samplerate);

  gst_structure_fixate_field_nearest_int (structure, "rate", src->samplerate);

  name = gst_structure_get_name (structure);
  if (strcmp (name, "audio/x-raw-int") == 0)
    gst_structure_fixate_field_nearest_int (structure, "width", 32);
  else if (strcmp (name, "audio/x-raw-float") == 0)
    gst_structure_fixate_field_nearest_int (structure, "width", 64);
}

static ProcessFunc process_funcs[];

static gboolean
gst_jammo_slider_setcaps (GstBaseSrc * basesrc, GstCaps * caps)
{
  GstJammoSlider *src = GST_JAMMO_SLIDER (basesrc);
  const GstStructure *structure;
  const gchar *name;
  gint width;
  gboolean ret;

  structure = gst_caps_get_structure (caps, 0);
  ret = gst_structure_get_int (structure, "rate", &src->samplerate);

  GST_DEBUG_OBJECT (src, "negotiated to samplerate %d", src->samplerate);

  name = gst_structure_get_name (structure);
  if (strcmp (name, "audio/x-raw-int") == 0) {
    ret &= gst_structure_get_int (structure, "width", &width);
    src->format = (width == 32) ? GST_JAMMO_SLIDER_FORMAT_S32 :
        GST_JAMMO_SLIDER_FORMAT_S16;
  } else {
    ret &= gst_structure_get_int (structure, "width", &width);
    src->format = (width == 32) ? GST_JAMMO_SLIDER_FORMAT_F32 :
        GST_JAMMO_SLIDER_FORMAT_F64;
  }

  /* allocate a new buffer suitable for this pad */
  switch (src->format) {
    case GST_JAMMO_SLIDER_FORMAT_S16:
      printf("format is now gint16\n");
      src->sample_size = sizeof (gint16);
      break;
    case GST_JAMMO_SLIDER_FORMAT_S32:
      printf("format is now gint32\n");
      src->sample_size = sizeof (gint32);
      break;
    case GST_JAMMO_SLIDER_FORMAT_F32:
      printf("format is now gfloat32\n");
      src->sample_size = sizeof (gfloat);
      break;
    case GST_JAMMO_SLIDER_FORMAT_F64:
      printf("format is now gfloat32\n");
      src->sample_size = sizeof (gdouble);
      break;
    default:
      /* can't really happen */
      ret = FALSE;
      break;
  }

  if (src->format == -1) {
    src->process = NULL;
    return FALSE;
  }
  src->process = process_funcs[src->format];

  return ret;
}

static gboolean
gst_jammo_slider_query (GstBaseSrc * basesrc, GstQuery * query)
{
  GstJammoSlider *src = GST_JAMMO_SLIDER (basesrc);
  gboolean res = FALSE;

  switch (GST_QUERY_TYPE (query)) {
    case GST_QUERY_CONVERT:
    {
      GstFormat src_fmt, dest_fmt;
      gint64 src_val, dest_val;

      gst_query_parse_convert (query, &src_fmt, &src_val, &dest_fmt, &dest_val);
      if (src_fmt == dest_fmt) {
        dest_val = src_val;
        goto done;
      }

      switch (src_fmt) {
        case GST_FORMAT_DEFAULT:
          switch (dest_fmt) {
            case GST_FORMAT_TIME:
              /* samples to time */
              dest_val =
                  gst_util_uint64_scale_int (src_val, GST_SECOND,
                  src->samplerate);
              break;
            default:
              goto error;
          }
          break;
        case GST_FORMAT_TIME:
          switch (dest_fmt) {
            case GST_FORMAT_DEFAULT:
              /* time to samples */
              dest_val =
                  gst_util_uint64_scale_int (src_val, src->samplerate,
                  GST_SECOND);
              break;
            default:
              goto error;
          }
          break;
        default:
          goto error;
      }
    done:
      gst_query_set_convert (query, src_fmt, src_val, dest_fmt, dest_val);
      res = TRUE;
      break;
    }
    default:
      res = GST_BASE_SRC_CLASS (parent_class)->query (basesrc, query);
      break;
  }

  return res;
  /* ERROR */
error:
  {
    GST_DEBUG_OBJECT (src, "query failed");
    return FALSE;
  }
}

/*This array is automatic generated with freq_util.c*/
/*It is based on facts (=supposition) 
a) there are 12 notes on one octave
b) there are octaves from 0...7
*/
static gfloat freq_of_note[] = { 32.703201293945312, 34.647899627685547, 36.708099365234375, 38.890899658203125, 41.203498840332031, 43.653598785400391, 46.249298095703125, 48.999500274658203, 51.912998199462891, 55.000000000000000, 58.270500183105469, 61.735401153564453,
 65.406402587890625, 69.295799255371094, 73.416198730468750, 77.781799316406250, 82.406997680664062, 87.307197570800781, 92.498596191406250, 97.999000549316406, 103.825996398925781, 110.000000000000000, 116.541000366210938, 123.470802307128906,
 130.812805175781250, 138.591598510742188, 146.832397460937500, 155.563598632812500, 164.813995361328125, 174.614395141601562, 184.997192382812500, 195.998001098632812, 207.651992797851562, 220.000000000000000, 233.082000732421875, 246.941604614257812,
 261.625610351562500, 277.183197021484375, 293.664794921875000, 311.127197265625000, 329.627990722656250, 349.228790283203125, 369.994384765625000, 391.996002197265625, 415.303985595703125, 440.000000000000000, 466.164001464843750, 493.883209228515625,
 523.251220703125000, 554.366394042968750, 587.329589843750000, 622.254394531250000, 659.255981445312500, 698.457580566406250, 739.988769531250000, 783.992004394531250, 830.607971191406250, 880.000000000000000, 932.328002929687500, 987.766418457031250,
 1046.502441406250000, 1108.732788085937500, 1174.659179687500000, 1244.508789062500000, 1318.511962890625000, 1396.915161132812500, 1479.977539062500000, 1567.984008789062500, 1661.215942382812500, 1760.000000000000000, 1864.656005859375000, 1975.532836914062500,
 2093.004882812500000, 2217.465576171875000, 2349.318359375000000, 2489.017578125000000, 2637.023925781250000, 2793.830322265625000, 2959.955078125000000, 3135.968017578125000, 3322.431884765625000, 3520.000000000000000, 3729.312011718750000, 3951.065673828125000,
 4186.009765625000000, 4434.931152343750000, 4698.636718750000000, 4978.035156250000000, 5274.047851562500000, 5587.660644531250000, 5919.910156250000000, 6271.936035156250000, 6644.863769531250000, 7040.000000000000000, 7458.624023437500000, 7902.131347656250000
};

/*We roll over above frequence table (sorted) and when value from table is bigger
than parameter, we check which one is nearer (which direction to round)*/
static gfloat freq_rounding(gfloat slider_freq){
 int i; //i starts from 1. All too low values are then rounded to table[0]
 for (i=1;i<NUMBER_OF_OCTAVES*NUMBER_OF_NOTES;i++){//roll over frequence table above (they are sorted)
  if (freq_of_note[i]>slider_freq){                //continue until table has bigger than parameter
    gfloat delta  =freq_of_note[i]-slider_freq;    //this is delta of parameter and bigger value from table
    gfloat delta2 =slider_freq-freq_of_note[i-1];  //this is delta of parameter and smaller value from table
    return delta<delta2?freq_of_note[i]:freq_of_note[i-1]; //return value, which is nearer to parameter
  }
 }
 return 8000; //We could also return biggest value from table.
}


gfloat pitch(gint needed_pitch){
	gfloat pitch=1.0;
		if (needed_pitch<=0)
			while (needed_pitch<0) {
				pitch*=0.9439;
				needed_pitch++;
			}

		else
			while (needed_pitch>0) {
				pitch*=1.0595;
				needed_pitch--;
			}
		
	return pitch;
}

gfloat xfader(gint16 *sampletable, gfloat *xfadecounter, gfloat *xfadeindex, gint xfade, gfloat value, gfloat scale, gint needed_pitch, gint samplerate) {

	gfloat xfademix;

	/* linear crossfade */
	/* fade in signal */
	xfademix= (0.0+1.0* (*xfadecounter)/(xfade*1.0));
	value*= xfademix;
	/* fade out signal */
	xfademix= (1.0-1.0* (*xfadecounter)/(xfade*1.0));
	value+= (xfademix)*( interpolation( *xfadeindex-(gint)*xfadeindex, sampletable[(gint)*xfadeindex], sampletable[((gint)*xfadeindex+1)] ) /(G_MAXINT16*1.0) * scale);
	/* Compensation of volume loss during crossfade */
	/*values[counter]*=1.05-fabs(0.05-0.1* (*xfadecounter)/(*xfade*1.0));*/
	*xfadecounter+=1.0;
	/* 44100 is samplerate of sample files. value could be read from wav header and stored to i.e slicedWav->samplerate. then we would be able to used samples with different sample rates */
	/* samplerate variable has the output samplerate of jammoslider */
	*xfadeindex += pitch(needed_pitch)*44100/(samplerate+0.0);

	return value;
}

#define DEFINE_PROCESS(type,scale) \
static void gst_jammo_slider_process_##type (GstJammoSlider * src, g##type * samples) {\
	/* if volume is divided with number of playing tones instrument*/\
	/*gets quieter when more simultaneous notes are played */ \
\
	/* this way overall volume can be adjusted according to number*/ \
	/*of playing notes but without lowering the overall volume too much */ \
\
	/* however it is possible to get audible clipping this way. */\
	/*one way to avoid this is not to scale samples to full volume, i.e.*/\
	/*use smaller scale value than currently. in another words leaving some*/\
	/*headroom to simultaneously playing samples */ \
\
	/* at the moment sampler works fine with flute, drum kit and ud.*/\
	/*clipping is not too bad */\
  gint i;\
  g##type value=0;\
  /*gdouble amp;*/ \
  gdouble step;\
  gint counter=0;\
\
  /*Process whole buffer*/ \
  for (i = 0; i < src->generate_samples_per_buffer; i++) \
      {\
      /* Find what tone must be processed */ \
      counter=0;\
\
	  if ( src->slider_freq_slow < src->slider_freq &&  src->slider_freq - src->slider_freq_slow > 1.0){\
	    src->slider_freq_slow+=0.02;\
	    src->slider_rounding_counter=0;\
	    }\
	  else if ( src->slider_freq_slow > src->slider_freq && src->slider_freq_slow - src->slider_freq > 1.0 ){\
	    src->slider_freq_slow-=0.02;\
	    src->slider_rounding_counter=0;\
	    }\
	  else {\
	    if (src->slider_state==TRUE)\
	      src->slider_rounding_counter++;\
	    }\
\
	  if (src->slider_rounding_counter>5000){\
	  src->slider_freq=freq_rounding(src->slider_freq);\
	/*  printf("do rounding\n");*/\
	  }\
\
/*printf("counter=%d, freq_slow=%f\n",src->slider_rounding_counter,src->slider_freq_slow);*/\
	  step = M_PI_M2 * src->slider_freq_slow / src->samplerate;\
	  src->slider_accumulator+= step;\
	  if (src->slider_accumulator >= M_PI_M2)\
			  src->slider_accumulator -= M_PI_M2;\
	  /* to avoid clipping everything is multiplied with 0.4 (less volume).*/ \
	  /*this is done to make synthesis about as loud as samples */ \
	  if(src->slider_state){\
            if (src->instrument==2)\
            value = (g##type)( (0.7+fabs(0.3*sin(src->tremolo_phase))) *0.8* scale * (karplus(src) ));\
            else\
	    value = (g##type)( (0.7+fabs(0.3*sin(src->tremolo_phase))) *0.8* scale * (instruments[src->instrument](src->slider_accumulator) ));\
}\
	  else\
	    value=0;\
	  src->tremolo_phase += (M_PI*10/src->samplerate); \
	  if (src->tremolo_phase>2*M_PI)\
		    src->tremolo_phase-=2*M_PI;\
\
	  if (src->slider_state==TRUE) {\
	    if (!src->released) {\
	      /* Attack */\
	      if (src->adsr_counter<src->attack) { \
		value*=src->adsr_counter/((gfloat)src->attack);\
		src->release_multiplier=src->adsr_counter/((gfloat)src->attack);\
		src->adsr_counter++;\
		/*printf("attack\n");*/\
	      }\
  \
	      /* Decay */ \
	      if (src->adsr_counter >= src->attack && src->adsr_counter < src->attack+src->decay) {\
		value*=(1-src->sustain)*(1-(src->adsr_counter/((gfloat)(src->attack+src->decay))))+src->sustain;\
		src->release_multiplier=(1-src->sustain)*(1-(src->adsr_counter/((gfloat)(src->attack+src->decay))))+src->sustain;\
		src->adsr_counter++; \
		/*printf("decay\n"); */\
	      } \
  \
	      /* Sustain */ \
	      if (src->adsr_counter >= src->attack + src->decay) {\
		value*=src->sustain; \
		src->release_multiplier=src->sustain;\
		/*printf("sustain\n");*/ \
	      }\
	    }\
\
	    /* Release */ \
	    if (src->released) {\
	      value*=src->release_multiplier*(1-(src->release_counter/((gfloat)src->release)));\
	      src->release_counter++;\
	      if (src->release_counter>=src->release) \
		src->slider_state=FALSE;\
	     /* printf("release\n");*/ \
	    } \
	  }\
	  samples[i]=value;\
}\
}

DEFINE_PROCESS (int16, 32767.0);
DEFINE_PROCESS (int32, 2147483647.0);
DEFINE_PROCESS (float, 1.0);
DEFINE_PROCESS (double, 1.0);


static ProcessFunc process_funcs[] = {
  (ProcessFunc) gst_jammo_slider_process_int16,
  (ProcessFunc) gst_jammo_slider_process_int32,
  (ProcessFunc) gst_jammo_slider_process_float,
  (ProcessFunc) gst_jammo_slider_process_double
};


static void
gst_jammo_slider_get_times (GstBaseSrc * basesrc, GstBuffer * buffer,
    GstClockTime * start, GstClockTime * end)
{
  /* for live sources, sync on the timestamp of the buffer */
  if (gst_base_src_is_live (basesrc)) {
    GstClockTime timestamp = GST_BUFFER_TIMESTAMP (buffer);

    if (GST_CLOCK_TIME_IS_VALID (timestamp)) {
      /* get duration to calculate end time */
      GstClockTime duration = GST_BUFFER_DURATION (buffer);

      if (GST_CLOCK_TIME_IS_VALID (duration)) {
        *end = timestamp + duration;
      }
      *start = timestamp;
    }
  } else {
    *start = -1;
    *end = -1;
  }
}

static gboolean
gst_jammo_slider_start (GstBaseSrc * basesrc)
{
  GstJammoSlider *src = GST_JAMMO_SLIDER (basesrc);
  int i;

  src->next_sample = 0;
  src->next_byte = 0;
  src->next_time = 0;
  src->check_seek_stop = FALSE;
  src->eos_reached = FALSE;
  src->tags_pushed = FALSE;
  src->tremolo_phase = 0;
  for (i=0; i<NUMBER_OF_OCTAVES*NUMBER_OF_NOTES;i++)
       src->accumulators[i] = 0;


  return TRUE;
}

static gboolean
gst_jammo_slider_stop (GstBaseSrc * basesrc)
{
  return TRUE;
}

/* seek to time, will be called when we operate in push mode. In pull mode we
 * get the requiested byte offset. */
static gboolean
gst_jammo_slider_do_seek (GstBaseSrc * basesrc, GstSegment * segment)
{
  GstJammoSlider *src = GST_JAMMO_SLIDER (basesrc);
  GstClockTime time;

  segment->time = segment->start;
  time = segment->last_stop;

  /* now move to the time indicated */
  src->next_sample =
      gst_util_uint64_scale_int (time, src->samplerate, GST_SECOND);
  src->next_byte = src->next_sample * src->sample_size;
  src->next_time =
      gst_util_uint64_scale_int (src->next_sample, GST_SECOND, src->samplerate);

  g_assert (src->next_time <= time);

  if (GST_CLOCK_TIME_IS_VALID (segment->stop)) {
    time = segment->stop;
    src->sample_stop = gst_util_uint64_scale_int (time, src->samplerate,
        GST_SECOND);
    src->check_seek_stop = TRUE;
  } else {
    src->check_seek_stop = FALSE;
  }
  src->eos_reached = FALSE;

  return TRUE;
}

static gboolean
gst_jammo_slider_is_seekable (GstBaseSrc * basesrc)
{
  /* we're seekable... */
  return TRUE;
}

static gboolean
gst_jammo_slider_check_get_range (GstBaseSrc * basesrc)
{
  GstJammoSlider *src;

  src = GST_JAMMO_SLIDER (basesrc);

  /* if we can operate in pull mode */
  return src->can_activate_pull;
}

static GstFlowReturn
gst_jammo_slider_create (GstBaseSrc * basesrc, guint64 offset,
    guint length, GstBuffer ** buffer)
{
  GstFlowReturn res;
  GstJammoSlider *src;
  GstBuffer *buf;
  GstClockTime next_time;
  gint64 next_sample, next_byte;
  guint bytes, samples;

  src = GST_JAMMO_SLIDER (basesrc);

  /* example for tagging generated data */
  if (!src->tags_pushed) {
    GstTagList *taglist;
    GstEvent *event;

    taglist = gst_tag_list_new ();

    gst_tag_list_add (taglist, GST_TAG_MERGE_APPEND,
        GST_TAG_DESCRIPTION, "jammo instrument", NULL);

    event = gst_event_new_tag (taglist);
    gst_pad_push_event (basesrc->srcpad, event);
    src->tags_pushed = TRUE;
  }

  if (src->eos_reached)
    return GST_FLOW_UNEXPECTED;

  /* if no length was given, use our default length in samples otherwise convert
   * the length in bytes to samples. */
  if (length == -1)
    samples = src->samples_per_buffer;
  else
    samples = length / src->sample_size;

  /* if no offset was given, use our next logical byte */
  if (offset == -1)
    offset = src->next_byte;

  /* now see if we are at the byteoffset we think we are */
  if (offset != src->next_byte) {
    GST_DEBUG_OBJECT (src, "seek to new offset %" G_GUINT64_FORMAT, offset);
    /* we have a discont in the expected sample offset, do a 'seek' */
    src->next_sample = src->next_byte / src->sample_size;
    src->next_time =
        gst_util_uint64_scale_int (src->next_sample, GST_SECOND,
        src->samplerate);
    src->next_byte = offset;
  }

  /* check for eos */
  if (src->check_seek_stop &&
      (src->sample_stop > src->next_sample) &&
      (src->sample_stop < src->next_sample + samples)
      ) {
    /* calculate only partial buffer */
    src->generate_samples_per_buffer = src->sample_stop - src->next_sample;
    next_sample = src->sample_stop;
    src->eos_reached = TRUE;
  } else {
    /* calculate full buffer */
    src->generate_samples_per_buffer = samples;
    next_sample = src->next_sample + samples;
  }

  bytes = src->generate_samples_per_buffer * src->sample_size;

  if ((res = gst_pad_alloc_buffer (basesrc->srcpad, src->next_sample,
              bytes, GST_PAD_CAPS (basesrc->srcpad), &buf)) != GST_FLOW_OK) {
    return res;
  }

  next_byte = src->next_byte + bytes;
  next_time = gst_util_uint64_scale_int (next_sample, GST_SECOND,
      src->samplerate);

  GST_LOG_OBJECT (src, "samplerate %d", src->samplerate);
  GST_LOG_OBJECT (src, "next_sample %" G_GINT64_FORMAT ", ts %" GST_TIME_FORMAT,
      next_sample, GST_TIME_ARGS (next_time));

  GST_BUFFER_TIMESTAMP (buf) = src->timestamp_offset + src->next_time;
  GST_BUFFER_OFFSET (buf) = src->next_sample;
  GST_BUFFER_OFFSET_END (buf) = next_sample;
  GST_BUFFER_DURATION (buf) = next_time - src->next_time;

  gst_object_sync_values (G_OBJECT (src), src->next_time);

  src->next_time = next_time;
  src->next_sample = next_sample;
  src->next_byte = next_byte;

  GST_LOG_OBJECT (src, "generating %u samples at ts %" GST_TIME_FORMAT,
      src->generate_samples_per_buffer,
      GST_TIME_ARGS (GST_BUFFER_TIMESTAMP (buf)));

  src->process (src, GST_BUFFER_DATA (buf));

  if (G_UNLIKELY (src->volume == 0.0)) {
    GST_BUFFER_FLAG_SET (buf, GST_BUFFER_FLAG_GAP);
  }

  *buffer = buf;

  return GST_FLOW_OK;
}

static void
gst_jammo_slider_set_property (GObject * object, guint prop_id,
    const GValue * value, GParamSpec * pspec)
{
  GstJammoSlider *src = GST_JAMMO_SLIDER (object);

  switch (prop_id) {
    case PROP_SAMPLES_PER_BUFFER:
      src->samples_per_buffer = g_value_get_int (value);
      break;
    case PROP_INSTRUMENT:
      src->instrument = g_value_get_int (value);
      break;
    case PROP_ATTACK:
      src->attack = g_value_get_int (value);
      break;
    case PROP_DECAY:
      src->decay = g_value_get_int (value);
      break;
    case PROP_RELEASE:
      src->release = g_value_get_int (value);
      break;
    case PROP_SUSTAIN:
      src->sustain = g_value_get_double (value);
      break;
    case PROP_VOLUME:
      src->volume = g_value_get_double (value);
      break;
    case PROP_IS_LIVE:
      gst_base_src_set_live (GST_BASE_SRC (src), g_value_get_boolean (value));
      break;
    case PROP_TIMESTAMP_OFFSET:
      src->timestamp_offset = g_value_get_int64 (value);
      break;
    case PROP_CAN_ACTIVATE_PUSH:
      GST_BASE_SRC (src)->can_activate_push = g_value_get_boolean (value);
      break;
    case PROP_CAN_ACTIVATE_PULL:
      src->can_activate_pull = g_value_get_boolean (value);
      break;

    case PROP_SLIDER_FREQ:
      src->slider_freq = g_value_get_float (value);
      if (src->slider_first_freq)
	{
	src->slider_freq_slow=src->slider_freq;
	src->slider_first_freq=FALSE;
	}
      break;
    case PROP_SLIDER_STATE:
     // src->slider_state = g_value_get_boolean (value);
      if ((g_value_get_boolean (value))==TRUE) {
	src->slider_state=TRUE;
	src->released=FALSE;
	src->release_counter=0;
	src->adsr_counter=0;
      }
      else
	src->released=TRUE;
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
gst_jammo_slider_get_property (GObject * object, guint prop_id,
    GValue * value, GParamSpec * pspec)
{
  GstJammoSlider *src = GST_JAMMO_SLIDER (object);

  switch (prop_id) {
    case PROP_SAMPLES_PER_BUFFER:
      g_value_set_int (value, src->samples_per_buffer);
      break;
    case PROP_INSTRUMENT:
      g_value_set_enum (value, src->instrument);
      break;
    case PROP_ATTACK:
      g_value_set_int (value, src->attack);
      break;
    case PROP_DECAY:
      g_value_set_int (value, src->decay);
      break;
    case PROP_RELEASE:
      g_value_set_int (value, src->release);
      break;
    case PROP_SUSTAIN:
       g_value_set_double (value, src->sustain);
      break;
    case PROP_VOLUME:
      g_value_set_double (value, src->volume);
      break;
    case PROP_IS_LIVE:
      g_value_set_boolean (value, gst_base_src_is_live (GST_BASE_SRC (src)));
      break;
    case PROP_TIMESTAMP_OFFSET:
      g_value_set_int64 (value, src->timestamp_offset);
      break;
    case PROP_CAN_ACTIVATE_PUSH:
      g_value_set_boolean (value, GST_BASE_SRC (src)->can_activate_push);
      break;
    case PROP_CAN_ACTIVATE_PULL:
      g_value_set_boolean (value, src->can_activate_pull);
      break;
    case PROP_SLIDER_FREQ:
       g_value_set_double (value, src->slider_freq);
      break;
    case PROP_SLIDER_STATE:
      g_value_set_boolean (value,src->slider_state);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;

  }
}

static gboolean
plugin_init (GstPlugin * plugin)
{

  /* initialize gst controller library */
  gst_controller_init (NULL, NULL);

  GST_DEBUG_CATEGORY_INIT (jammo_slider_debug, "jammoslider", 0,
      "Jammo Slider Instrument");
  return gst_element_register (plugin, "jammoslider",
      GST_RANK_NONE, GST_TYPE_JAMMO_SLIDER);
}

GST_PLUGIN_DEFINE (GST_VERSION_MAJOR,
    GST_VERSION_MINOR,
    "jammoslider",
    "Creates slider instrument",
    plugin_init, VERSION, "LGPL", GST_PACKAGE_NAME, GST_PACKAGE_ORIGIN);
