/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Dlg/Properties.h>

#include <list>
#include <stack>

#include <Lum/Base/L10N.h>
#include <Lum/Base/String.h>

#include <Lum/Label.h>
#include <Lum/Panel.h>
#include <Lum/ScrolledObject.h>
#include <Lum/Slider.h>
#include <Lum/String.h>
#include <Lum/Tab.h>
#include <Lum/Text.h>

#include <iostream>

namespace Lum {
  namespace Dlg {

    Properties::Visitor::~Visitor()
    {
      // no code
    }

    Properties::Properties(Def::PropGroup *props)
    : props(props),
      okAction(new Model::Action())
    {
      Observe(okAction);
      Observe(GetClosedAction());

      PushModels();

      ObserveModels();
    }

    Properties::~Properties()
    {
      PopModels();
    }

    void Properties::VisitModels(Def::PropGroup *props, Visitor& visitor)
    {
      for (std::list<Def::PropItem*>::const_iterator prop=props->GetProps().begin();
          prop!=props->GetProps().end();
          ++prop) {
        if (dynamic_cast<Def::PropGroup*>(*prop)!=NULL) {
          Def::PropGroup* group=dynamic_cast<Def::PropGroup*>(*prop);

          VisitModels(group,visitor);
        }
        else if (dynamic_cast<Def::PropBoolean*>(*prop)!=NULL) {
          Def::PropBoolean* boolean=dynamic_cast<Def::PropBoolean*>(*prop);

          visitor.Visit(boolean->GetValue().GetBoolean());
        }
        else if (dynamic_cast<Def::PropNumber*>(*prop)!=NULL) {
          Def::PropNumber* number=dynamic_cast<Def::PropNumber*>(*prop);

          visitor.Visit(number->GetValue().GetValue());
        }
        else if (dynamic_cast<Def::PropInterval*>(*prop)!=NULL) {
          Def::PropInterval* interval=dynamic_cast<Def::PropInterval*>(*prop);

          visitor.Visit(interval->GetStart().GetValue());
          visitor.Visit(interval->GetEnd().GetValue());
        }
        else if (dynamic_cast<Def::PropOneOfN*>(*prop)!=NULL) {
          Def::PropOneOfN* oneOfN=dynamic_cast<Def::PropOneOfN*>(*prop);

          visitor.Visit(oneOfN->GetValue().GetValue());
        }
        else if (dynamic_cast<Def::PropOneOfMany*>(*prop)!=NULL) {
          Def::PropOneOfMany* oneOfMany=dynamic_cast<Def::PropOneOfMany*>(*prop);

          visitor.Visit(oneOfMany->GetValue().GetValue());
        }
      }
    }

    void Properties::VisitModels(Visitor& visitor)
    {
      VisitModels(props, visitor);
    }

    void Properties::ObserveModels()
    {
      struct ObserveVisitor : Visitor
      {
        Properties* dialog;

        void Visit(Lum::Base::Model* model)
        {
          dialog->Observe(model);
        }
      };

      ObserveVisitor visitor;

      visitor.dialog=this;

      VisitModels(visitor);
    }

    void Properties::PushModels()
    {
      struct PushVisitor : Visitor
      {
        void Visit(Lum::Base::Model* model)
        {
          model->Push();
        }
      };

      PushVisitor visitor;

      VisitModels(visitor);
    }

    void Properties::SaveModels()
    {
      struct SaveVisitor : Visitor
      {
        void Visit(Lum::Base::Model* model)
        {
          model->Save();
        }
      };

      SaveVisitor visitor;

      VisitModels(visitor);
    }

    void Properties::PopModels()
    {
      struct PopVisitor : Visitor
      {
        void Visit(Lum::Base::Model* model)
        {
          model->Pop();
        }
      };

      PopVisitor visitor;

      VisitModels(visitor);
    }

    Lum::Object* Properties::GetContent()
    {
      std::list<Def::PropGroup*> groupStack;
      VPanel                     *vPanel;
      Label                      *label=NULL;
      Tab                        *tab=NULL;
      Lum::Object                *top;

      vPanel=new VPanel();
      vPanel->SetFlex(true,true);

      if (OS::display->GetSize()<OS::Display::sizeNormal) {
        top=ScrolledObject::Create(vPanel,true,true);
      }
      else {
        top=vPanel;
      }

      groupStack.push_front(props);

      while (!groupStack.empty()) {
        Def::PropGroup* props=groupStack.front();
        bool            isGroup=true;

        groupStack.erase(groupStack.begin());

        for (std::list<Def::PropItem*>::const_iterator prop=props->GetProps().begin();
            prop!=props->GetProps().end();
            ++prop) {
          if (dynamic_cast<Def::PropGroup*>(*prop)==NULL) {
            isGroup=false;
          }
        }

        if (isGroup) {
          if (OS::display->GetSize()<OS::Display::sizeNormal) {
            if (label==NULL) {
              label=new Label();
              label->SetFlex(true,true);
            }

            label->AddLabel(props->GetDesc().GetLabel(),NULL);
          }
          else {
            Tab *newTab=new Tab();
            newTab->SetFlex(true,true);

            if (tab!=NULL) {
              tab->Add(props->GetDesc().GetLabel(),newTab);
            }
            else {
              vPanel->Add(newTab);
            }

            tab=newTab;
            label=NULL;
          }

          for (std::list<Def::PropItem*>::const_iterator prop=props->GetProps().begin();
              prop!=props->GetProps().end();
              ++prop) {
            if (dynamic_cast<Def::PropGroup*>(*prop)!=NULL) {
              groupStack.push_back(dynamic_cast<Def::PropGroup*>(*prop));
            }
          }
        }
        else {
          if (label==NULL) {
            label=new Label();
            label->SetFlex(true,true);
	    vPanel->Add(label);
          }

          for (std::list<Def::PropItem*>::const_iterator prop=props->GetProps().begin();
              prop!=props->GetProps().end();
              ++prop) {
            if (dynamic_cast<Def::PropGroup*>(*prop)!=NULL) {
              Def::PropGroup *group=dynamic_cast<Def::PropGroup*>(*prop);

              label->AddLabel(group->GetDesc().GetLabel(),NULL);
            }
            else if (dynamic_cast<Def::PropBoolean*>(*prop)!=NULL) {
              Def::PropBoolean* boolean=dynamic_cast<Def::PropBoolean*>(*prop);

              label->AddLabel(boolean->GetValue().GetDesc().GetLabel(),
                              Manager::Behaviour::Instance()->GetBooleanControl(boolean->GetValue(),true,false));
            }
            else if (dynamic_cast<Def::PropNumber*>(*prop)!=NULL) {
              Def::PropNumber* number=dynamic_cast<Def::PropNumber*>(*prop);

              if (number->GetValue().GetValue()->GetMaxAsDouble()-
                  number->GetValue().GetValue()->GetMinAsDouble()<20.0) {
                label->AddLabel(number->GetValue().GetDesc().GetLabel()+L":",
                                HSlider::Create(number->GetValue().GetValue(),true,true,false));
              }
              else {
                label->AddLabel(number->GetValue().GetDesc().GetLabel()+L":",
                                String::Create(number->GetValue().GetValue(),true,false));
              }
            }
            else if (dynamic_cast<Def::PropInterval*>(*prop)!=NULL) {
              Def::PropInterval *interval=dynamic_cast<Def::PropInterval*>(*prop);
              HPanel            *hPanel;

              hPanel=new HPanel();
              hPanel->SetFlex(true,false);
              hPanel->Add(String::Create(interval->GetStart().GetValue(),true,false));
              hPanel->Add(Text::Create(L" - "));
              hPanel->Add(String::Create(interval->GetEnd().GetValue(),true,false));

              label->AddLabel(interval->GetDesc().GetLabel()+L":",hPanel);
            }
            else if (dynamic_cast<Def::PropOneOfN*>(*prop)!=NULL) {
              Def::PropOneOfN* oneOfN=dynamic_cast<Def::PropOneOfN*>(*prop);

              label->AddLabel(oneOfN->GetValue().GetDesc().GetLabel()+L":",
                              Manager::Behaviour::Instance()->GetOneOfNControl(oneOfN->GetValue(),true,false));
            }
            else if (dynamic_cast<Def::PropOneOfMany*>(*prop)!=NULL) {
              Def::PropOneOfMany* oneOfMany=dynamic_cast<Def::PropOneOfMany*>(*prop);

              label->AddLabel(oneOfMany->GetValue().GetDesc().GetLabel()+L":",
                              Manager::Behaviour::Instance()->GetOneOfManyControl(oneOfMany->GetValue(),true,false));
            }
            else {
              std::cerr << "Unsupported property" << std::endl;
            }
          }

          if (tab!=NULL) {
            tab->Add(props->GetDesc().GetLabel(),label);
            label=NULL;
          }
          else if (OS::display->GetSize()>=OS::Display::sizeNormal) {
            label=NULL;
          }
        }
      }

      return top;
    }

    void Properties::GetActions(std::vector<Lum::Dlg::ActionInfo>& actions)
    {
      Lum::Dlg::ActionDialog::CreateActionInfosOkCancel(actions,okAction,GetClosedAction());
    }

    void Properties::Resync(Lum::Base::Model* model, const Lum::Base::ResyncMsg& msg)
    {
      if (model==okAction &&
          okAction->IsFinished()) {
        SaveModels();
        Exit();
      }
      else if (model==GetClosedAction() &&
               GetClosedAction()->IsFinished()) {
        Exit();
      }

      Dialog::Resync(model,msg);
    }

    void Properties::Show(Dialog* parent, Def::PropGroup* props)
    {
      Show(parent->GetWindow(),props);
    }

    void Properties::Show(OS::Window* parent, Def::PropGroup* props)
    {
      Properties properties(props);

      properties.SetParent(parent);
      properties.SetTitle(L"Config...");

      if (properties.Open()) {
        properties.EventLoop();
        properties.Close();
      }
    }
  }
}
