#include <Lum/Label.h>

/*
  This source is part of the Illumination library
  Copyright (C) 2004  Tim Teulings

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/

#include <Lum/Base/Util.h>

#include <Lum/OS/Display.h>

#include <Lum/Text.h>
#include <Lum/TextLabel.h>

namespace Lum {

  void Label::Entry::CalcSize()
  {
    if (label!=NULL) {
      label->CalcSize();
    }

    if (object!=NULL) {
      object->CalcSize();
    }

    if (label!=NULL && object!=NULL) {
      size_t height=std::max(label->GetOHeight(),object->GetOHeight());

      label->ResizeHeight(height);
      object->ResizeHeight(height);
    }
  }

  size_t Label::Entry::GetMinHeight() const
  {
    size_t height=0;

    if (label!=NULL) {
      height=std::max(height,label->GetOMinHeight());
    }

    if (object!=NULL) {
      height=std::max(height,object->GetOMinHeight());
    }

    if (label!=NULL && object!=NULL && label->HasBaseline() && object->HasBaseline()) {
      height+=std::max(label->GetBaseline(),object->GetBaseline())-std::min(label->GetBaseline(),object->GetBaseline());
    }

    return height;
  }

  size_t Label::Entry::GetHeight() const
  {
    size_t height=0;

    if (label!=NULL) {
      height=std::max(height,label->GetOHeight());
    }

    if (object!=NULL) {
      height=std::max(height,object->GetOHeight());
    }

    if (label!=NULL && object!=NULL && label->HasBaseline() && object->HasBaseline()) {
      height+=std::max(label->GetBaseline(),object->GetBaseline())-std::min(label->GetBaseline(),object->GetBaseline());
    }

    return height;
  }

  size_t Label::Entry::CanResize(bool grow) const
  {
    if (object!=NULL && label!=NULL) {
      return label->CanResize(grow,false) || object->CanResize(grow,false);
    }
    else if (label!=NULL) {
      return label->CanResize(grow,false);
    }
    else if (object!=NULL) {
      return object->CanResize(grow,false);
    }
    else {
      return false;
    }
  }

  void Label::Entry::Resize(size_t height)
  {
    if (label!=NULL && object!=NULL && label->HasBaseline() && object->HasBaseline()) {
      height-=std::max(label->GetBaseline(),object->GetBaseline())-std::min(label->GetBaseline(),object->GetBaseline());
    }

    if (label!=NULL) {
      label->ResizeHeight(height);
    }

    if (object!=NULL) {
      object->ResizeHeight(height);
    }
  }

  bool Label::Entry::HasOffset() const
  {
    return label!=NULL && object!=NULL && label->HasBaseline() && object->HasBaseline();
  }

  size_t Label::Entry::GetLabelOffset() const
  {
    if (label!=NULL && object!=NULL && label->HasBaseline() && object->HasBaseline()) {
      if (object->GetBaseline()>label->GetBaseline()) {
        return object->GetBaseline()-label->GetBaseline();
      }
      else {
        return 0;
      }
    }
    else {
      return 0;
    }
  }

  size_t Label::Entry::GetObjectOffset() const
  {
    if (label!=NULL && object!=NULL && label->HasBaseline() && object->HasBaseline()) {
      if (label->GetBaseline()>object->GetBaseline()) {
        return label->GetBaseline()-object->GetBaseline();
      }
      else {
        return 0;
      }
    }
    else {
      return 0;
    }
  }

  Label::Label()
  : labelWidth(0),objectWidth(0),labelFlex(false)
  {
    // no code
  }

  Label::~Label()
  {
    std::list<Entry>::iterator iter;

    iter=list.begin();
    while (iter!=list.end()) {
      delete iter->label;
      delete iter->object;

      ++iter;
    }
  }

  bool Label::VisitChildren(Visitor &visitor, bool /*onlyVisible*/)
  {
    std::list<Entry>::iterator iter;

    iter=list.begin();
    while (iter!=list.end()) {
      if (iter->label!=NULL) {
        if (!visitor.Visit(iter->label)) {
          return false;
        }
      }

      if (iter->object!=NULL) {
        if (!visitor.Visit(iter->object)) {
          return false;
        }
      }

      ++iter;
    }

    return true;
  }

  /**
    Add a new entry to the label. A label entry consists of two objects.
    \p text should be a textual label, while \p object is the
    object \p text is the label for.
  */
  Label* Label::AddLabel(Object* label, Object* object)
  {
    assert(label!=NULL && object!=NULL);

    Entry entry;

    label->SetParent(this);
    object->SetParent(this);

    object->SetLabelObject(label);

    entry.indent=0;
    entry.label=label;
    entry.object=object;

    list.push_back(entry);

    return this;
  }

  Label* Label::AddLabel(const std::wstring& label, Object* object)
  {
    return AddLabel(new TextLabel(label),object);
  }

  Label* Label::AddLabel(const std::wstring& label, const std::wstring& object)
  {
    return AddLabel(new TextLabel(label),new Text(object));
  }

  void Label::SetLabelFlex(bool flex)
  {
    labelFlex=flex;
  }

  void Label::CalcSize()
  {
    std::list<Entry>::iterator iter;
    size_t                     objectMinWidth; // Minumal width of object
    /*visitor   : RealignVisitorDesc;*/

    height=0;
    minHeight=0;
    labelWidth=0;
    objectWidth=0;
    objectMinWidth=0;

    // Calculate bounds of labels and objects
    iter=list.begin();
    while (iter!=list.end()) {
      iter->CalcSize();

      labelWidth=std::max(labelWidth,iter->label->GetOWidth());
      objectWidth=std::max(objectWidth,iter->object->GetOWidth());
      objectMinWidth=std::max(objectMinWidth,iter->object->GetOMinWidth());

      minHeight+=iter->GetMinHeight();
      height+=iter->GetHeight();

      ++iter;
    }

    // Add inter line space to height
    if (list.size()>1) {
      height+=(list.size()-1)*OS::display->GetSpaceVertical(OS::Display::spaceInterObject);
      minHeight+=(list.size()-1)*OS::display->GetSpaceVertical(OS::Display::spaceInterObject);
    }

    width=labelWidth+OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject)+objectWidth;
    minWidth=labelWidth+OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject)+objectMinWidth;

    Group::CalcSize();

    if (labelFlex) {
      labelWidth=width-objectWidth-OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject);
    }
    else {
      objectWidth=width-labelWidth-OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject);
    }

    /*
    IF (l.parent#NIL) & (l.parent IS P.Panel) THEN (* if we are in a panel*)
      (*
        Now we use a visitor to walk over all children of our parent (all
        objects in the same parent panel). If CalcSize was already called for
        them, we resize them to the smae labelWidth and objectWidth values and relayout them.

        See Label.ReLayout And local visitor implementation.
      *)
      visitor.me:=l;
      IF l.parent.VisitChildren(visitor,TRUE) THEN
      END;
    END;*/
  }

  void Label::Layout()
  {
    std::list<Entry>::iterator iter;
    size_t                     curHeight;
    int                        pos;

    if (labelFlex) {
      labelWidth=width-objectWidth-OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject);
    }
    else {
      objectWidth=width-labelWidth-OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject);
    }

    /*
      Calculate current height of labels and objects.
      If label is smaller than object in same row try to make it fit.
      Also do this if object is smaller than label.
    */
    curHeight=0;
    iter=list.begin();
    while (iter!=list.end()) {
      curHeight+=iter->GetHeight();

      iter->label->ResizeWidth(labelWidth);
      iter->object->ResizeWidth(objectWidth);

      ++iter;
    }

    // Now add inter row spacing
    if (list.size()>1) {
      curHeight+=(list.size()-1)*OS::display->GetSpaceVertical(OS::Display::spaceInterObject);
    }

    /*
      If current height is not equal to exspected height, resize until equal
       or until now object can be resized anymore.
    */
    while (curHeight!=height) {
      int count=0;

      // Count number of resizable objects
      iter=list.begin();
      while (iter!=list.end()) {
        if (iter->object->CanResize(height>curHeight,false)) {
          count++;
        }

        ++iter;
      }

      // Quit if we cannot resize an object anymore.
      if (count==0) {
        break;
      }

      iter=list.begin();
      while (iter!=list.end()) {
        if (iter->CanResize(height>curHeight)) {
          int old;

          old=iter->GetHeight();
          iter->Resize(iter->GetHeight()+Base::UpDiv(height-curHeight,count));
          curHeight+=iter->GetHeight()-old;
          count--;
        }

        ++iter;
      }
    }

    /* Reposition all objects */

    pos=y;
    iter=list.begin();
    while (iter!=list.end()) {
      if (OS::display->GetDefaultTextDirection()==OS::Display::textDirectionRightToLeft) {
        iter->object->Move(x,pos+iter->GetObjectOffset());
        iter->label->Move(x+objectWidth+OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject),
                          pos+iter->GetLabelOffset());
      }
      else {
        if (iter->HasOffset()) {
          iter->label->Move(x,pos+iter->GetLabelOffset());
          iter->object->Move(x+labelWidth+OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject),
                             pos+iter->GetObjectOffset());
        }
        else if (iter->object->GetOHeight()>3*iter->label->GetOHeight()) {
          iter->label->Move(x,pos);
        }
        else {
          iter->label->Move(x,pos+(iter->GetHeight()-iter->label->GetOHeight())/2);
        }

        if (iter->object->GetOHeight()<=iter->label->GetOHeight()) {
          iter->object->Move(x+labelWidth+OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject),
          pos+(iter->label->GetOHeight()-iter->object->GetOHeight())/2);
        }
        else {
          iter->object->Move(x+labelWidth+OS::display->GetSpaceHorizontal(OS::Display::spaceLabelObject),
          pos);
        }
      }

      pos+=iter->GetHeight()+OS::display->GetSpaceVertical(OS::Display::spaceInterObject);

      ++iter;
    }

    Group::Layout();
  }

  Label* Label::Create(bool horizontalFlex, bool verticalFlex)
  {
    Label *l;

    l=new Label();
    l->SetFlex(horizontalFlex,verticalFlex);

    return l;
  }

}

