/****************************************************************************
**
** This file is part of the Hyves PhoneGap container.
** Copyright (C) 2010-2011 Hyves (Startphone Ltd.)
** http://www.hyves.nl
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#include "memorywatcher.h"

#include <e32std.h>

#include <QDebug>
#include <QSettings>


MemoryWatcher::MemoryWatcher(QObject *parent) :
    QObject(parent),
    m_timer(),
    m_warningThreshold(75),
    m_criticalThreshold(90),
    m_state(NoAlertState) {

    connect(&m_timer, SIGNAL(timeout()), this, SLOT(checkMemory()));
}

int MemoryWatcher::usage() const {

    return 100 * bytesUsed() / User::Heap().MaxLength();
}

int MemoryWatcher::bytesUsed() const {

    TInt largestBlock;
    return User::Available(largestBlock);
}

int MemoryWatcher::bytesFree() const {

    return User::Heap().MaxLength() - bytesUsed();
}

int MemoryWatcher::warningThreshold() const {

    return m_warningThreshold;
}

int MemoryWatcher::criticalThreshold() const {

    return m_criticalThreshold;
}

void MemoryWatcher::setWarningThreshold(int threshold) {

    m_warningThreshold = threshold;
}

void MemoryWatcher::setCriticalThreshold(int threshold) {

    m_criticalThreshold = threshold;
}

void MemoryWatcher::start() {

    m_timer.start(1000);
}

void MemoryWatcher::stop() {

    m_timer.stop();
}

void MemoryWatcher::checkMemory() {

    MemoryAlertState previousState = m_state;
    int use = usage();
    m_state = (use < m_warningThreshold) ? NoAlertState : (use < m_criticalThreshold) ? WarningState : CriticalState;

    if (previousState != m_state) {
        switch (m_state) {
            case CriticalState:
                qWarning() << "Memory usage critical: " << use << "%";
                emit critical(use);
                break;
            case WarningState:
                qWarning() << "Memory usage warning: " << use << "%";
                emit warning(use);
                break;
            default:
                qDebug() << "Memory usage back to normal: " << use << "%";
                break;
        }
    }
}

int MemoryWatcher::killedByGoom() const {

    QSettings settings;
    return settings.value("stats/killedByGoom", 0).toInt();
}

void MemoryWatcher::clearKilledByGoom() {

    QSettings settings;
    settings.setValue("stats/killedByGoom", 0);
    settings.sync();
}
