/*
 * Geolocation.cpp
 *
 *  Created on: Nov 24, 2009
 *      Author: nitobi-test
 */

#include "geolocation.h"

#include <QDebug>


Geolocation::Geolocation(QObject *parent) :
    QObject(parent),
    m_source(0),
#ifdef Q_WS_MAEMO_5
    m_lastKnownPosition(),
#endif
    m_watchingPosition(false),
    m_highAccuracy(false) {
}

bool Geolocation::initSource() {

    if (m_source) {
        return true;
    }

    m_source = QGeoPositionInfoSource::createDefaultSource(this);
    if (m_source) {
#ifdef Q_WS_MAEMO_5
        m_source->setPreferredPositioningMethods(QGeoPositionInfoSource::AllPositioningMethods);
#else
        m_source->setPreferredPositioningMethods(QGeoPositionInfoSource::NonSatellitePositioningMethods);
#endif
        connect(m_source, SIGNAL(positionUpdated(QGeoPositionInfo)),
                this, SLOT(onPositionUpdated(QGeoPositionInfo)));
        connect(m_source, SIGNAL(updateTimeout()),
                this, SLOT(onUpdateTimeout()));
        return true;
    }

    qWarning("No GeoPosition source available.");
    return false;
}

void Geolocation::update(int timeout) {

    if (!initSource()) {
        emitError(EPositionUnavailable, "The device does not support position location.");
        return;
    }

    m_source->requestUpdate(timeout);
}

void Geolocation::start() {

    if (!initSource()) {
        emitError(EPositionUnavailable, "The device does not support position location.");
        return;
    }

    m_source->startUpdates();
    m_watchingPosition = true;
}

void Geolocation::stop() {

    if (!initSource()) {
        emitError(EPositionUnavailable, "The device does not support position location.");
        return;
    }

    m_watchingPosition = false;
    m_source->stopUpdates();
}

void Geolocation::emitError(Geolocation::PositionError code, const QString &message) {

    QVariantMap positionError;
    positionError["code"] = code;
    positionError["message"] = message;
    emit error(positionError);
}

void Geolocation::onPositionUpdated(const QGeoPositionInfo &info) {

    qDebug() << "GeoLocation update: " << info.timestamp().toString();
#ifdef Q_WS_MAEMO_5
    m_lastKnownPosition = info;
#endif
    emit positionUpdated(positionToVariantMap(info));
}

void Geolocation::onUpdateTimeout() {

    qDebug() << "GeoLocation update timeout";
    emitError(ETimeout, "Timeout occurred");
}

bool Geolocation::highAccuracy() const {

    return m_highAccuracy;
}

void Geolocation::setHighAccuracy(bool value) {

#ifdef Q_WS_MAEMO_5
    // network-based methods seem to be broken on Maemo -> allways use all sources
    Q_UNUSED(value);
#else
    if (initSource() && m_highAccuracy != value) {
        m_source->setPreferredPositioningMethods(value ? QGeoPositionInfoSource::AllPositioningMethods :
                                                 QGeoPositionInfoSource::NonSatellitePositioningMethods);
        m_highAccuracy = value;
        if (m_watchingPosition) { // changing positioning methods seems to interrupt updates, start again...
            m_source->startUpdates();
        }
    }
#endif
}

QVariantMap Geolocation::lastPosition() {

#ifdef Q_WS_MAEMO_5
    return positionToVariantMap(m_lastKnownPosition);
#else
    if (!initSource()) {
        return QVariantMap();
    }

    return positionToVariantMap(m_source->lastKnownPosition());
#endif
}

QVariantMap Geolocation::positionToVariantMap(const QGeoPositionInfo &info) {

    QVariantMap position;
    if (info.isValid()) {
        QVariantMap coords;
        coords["latitude"] = info.coordinate().latitude();
        coords["longitude"] = info.coordinate().longitude();
        coords["altitude"] = info.coordinate().altitude();
        coords["accuracy"] = info.attribute(QGeoPositionInfo::HorizontalAccuracy);
        coords["altitudeAccurary"] = info.attribute(QGeoPositionInfo::VerticalAccuracy);
        coords["heading"] = info.attribute(QGeoPositionInfo::Direction);
        coords["speed"] = info.attribute(QGeoPositionInfo::GroundSpeed);
        position["coords"] = coords;
#ifdef Q_WS_S60
        // Symbian bug in Qt-Mobility 1.0, see http://bugreports.qt.nokia.com/browse/QTMOBILITY-398
        position["timestamp"] = QDateTime(info.timestamp().date(), info.timestamp().time(), Qt::UTC);
#else
        position["timestamp"] = info.timestamp();
#endif
    }
    return position;
}
