/****************************************************************************
**
** This file is part of the Hyves PhoneGap container.
** Copyright (C) 2010-2011 Hyves (Startphone Ltd.)
** http://www.hyves.nl
**
** This program is free software: you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation, either version 3 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program.  If not, see <http://www.gnu.org/licenses/>.
**
****************************************************************************/

#ifndef OAUTHUTIL_H
#define OAUTHUTIL_H

#include <QObject>
#include <QStringList>
#include <QVariantMap>

class OAuthUtil : public QObject {

    Q_OBJECT

    public:
        struct Parameter {
            QString key;
            QString value;

            Parameter(const QString &key, const QString &value) : key(key), value(value) { }
        };

        class ParameterMap : public QMap<QString, QString> {

            public:
                ParameterMap() :
                    QMap<QString, QString>() {
                }

                ParameterMap(const char *parameterString) :
                    QMap<QString, QString>() {

                    QStringList parameters = QString(parameterString).split('&');
                    for (QStringList::ConstIterator it = parameters.constBegin();
                        it != parameters.constEnd(); ++it) {
                        QStringList parts = (*it).split('=');
                        QString key = parts.size() > 0 ? parts[0] : "";
                        QString value = parts.size() > 1 ? parts[1] : "";
                        insert(key, value);
                    }
                }

                ParameterMap(const QString &parameterString) :
                    QMap<QString, QString>() {

                    QStringList parameters = parameterString.split('&');
                    for (QStringList::ConstIterator it = parameters.constBegin();
                        it != parameters.constEnd(); ++it) {
                        QStringList parts = (*it).split('=');
                        QString key = parts.size() > 0 ? parts[0] : "";
                        QString value = parts.size() > 1 ? parts[1] : "";
                        insert(key, value);
                    }
                }

                ParameterMap(const QVariantMap &parameterMap) :
                    QMap<QString, QString>() {

                    for (QVariantMap::ConstIterator it = parameterMap.constBegin();
                        it != parameterMap.constEnd(); ++it) {
                        insert(it.key(), it.value().toString());
                    }
                }
        };

        class ParameterList : public QList<Parameter> {

            public:
                ParameterList() :
                    QList<Parameter>() {
                }

                ParameterList(const ParameterMap &parameterMap) :
                    QList<Parameter>() {

                    for (ParameterMap::ConstIterator it = parameterMap.constBegin();
                        it != parameterMap.constEnd(); ++it) {
                        append(Parameter(it.key(), *it));
                    }
                }

                ParameterList(const char *parameterString) :
                    QList<Parameter>() {

                    QStringList parameters = QString(parameterString).split('&');
                    for (QStringList::ConstIterator it = parameters.constBegin();
                        it != parameters.constEnd(); ++it) {
                        QStringList parts = (*it).split('=');
                        QString key = parts.size() > 0 ? parts[0] : "";
                        QString value = parts.size() > 1 ? parts[1] : "";
                        append(Parameter(key, value));
                    }
                }

                ParameterList(const QString &parameterString) :
                    QList<Parameter>() {

                    QStringList parameters = parameterString.split('&');
                    for (QStringList::ConstIterator it = parameters.constBegin();
                        it != parameters.constEnd(); ++it) {
                        QStringList parts = (*it).split('=');
                        QString key = parts.size() > 0 ? parts[0] : "";
                        QString value = parts.size() > 1 ? parts[1] : "";
                        append(Parameter(key, value));
                    }
                }

                ParameterList(const QVariantMap &parameterMap) :
                    QList<Parameter>() {

                    for (QVariantMap::ConstIterator it = parameterMap.constBegin();
                        it != parameterMap.constEnd(); ++it) {
                        append(Parameter(it.key(), it.value().toString()));
                    }
                }
        };

        explicit OAuthUtil(QObject *parent = 0);

        /**
         * Normalizes a ParameterList into an encoded string.
         */
        static QString normalizeParameters(ParameterList parameterList);
        Q_INVOKABLE static QString normalizeParameters(const QVariantMap &parameterMap);

        /**
         * Encodes strings in an RFC3986 compatible encoding.
         *
         * @param string String to encode.
         * @return The encoded string.
         */
        Q_INVOKABLE static QString urlencodeRFC3986(const QString &string);

        /**
         * Decodes an RFC3986 encoded string.
         *
         * @param string String to decode.
         * @return The decoded string.
         */
        Q_INVOKABLE static QString urldecodeRFC3986(const QString &string);

        /**
         * Creates the base string needed for signing per oAuth Section 9.1.2.
         * All strings are latin1.
         *
         * @param httpMethod One of the HTTP methods (GET, POST, etc.).
         * @param uri The URI; the URL without query string.
         * @param paramaterMap Parameters.
         */
        static QString generateBaseString(const QString &httpMethod, const QString &uri,
                                          ParameterMap parameterMap);

        /**
         * Calculates the HMAC-SHA1 secret.
         *
         * @param baseString Returned by generateBaseString().
         * @param consumerSecret
         * @param tokenSecret Leave empty if no token present.
         */
        Q_INVOKABLE static QString calculateHMACSHA1Signature(const QString &baseString,
                                                              const QString &consumerSecret,
                                                              const QString &tokenSecret);

        /**
         * Calculates the OAuth signature. This method combines the calls to
         * generateBaseString() and calculateHMACSHA1Signature().
         */
        static QString calculateOAuthSignature(const QString &httpMethod, const QString &uri,
                                               const ParameterMap &parameterMap,
                                               const QString &consumerSecret, const QString &oauthTokenSecret);
        Q_INVOKABLE static QString calculateOAuthSignature(const QString &httpMethod, const QString &uri,
                                                           const QVariantMap &parameterMap,
                                                           const QString &consumerSecret, const QString &oauthTokenSecret);

};

#endif // OAUTHUTIL_H
