import os.path
import evolution
from facebook import Facebook, FacebookError
import twitter
import gnome.gconf
from contacts import ContactStore
import names

class Hermes:
  """Encapsulate the process of syncing Facebook friends' information with the
     Evolution contacts' database. This should be used as follows:
     
       * Initialise, passing in a callback (methods: need_auth(),
         block_for_auth(), use_twitter(), use_facebook()).
       * Call load_friends().
       * Call sync_contacts().
       * Retrieve information on changes effected.
       
     This requires two gconf paths to contain Facebook application keys:
         /apps/maemo/hermes/key_app
         /apps/maemo/hermes/key_secret
       
     Copyright (c) Andrew Flegg <andrew@bleb.org> 2009.
     Released under the Artistic Licence."""


  # -----------------------------------------------------------------------
  def __init__(self, callback, twitter = None, facebook = False):
    """Constructor. Passed a callback which must implement three informational
       methods:
       
         need_auth() - called to indicate a login is about to occur. The user
                       should be informed.
                       
         block_for_auth() - prompt the user to take some action once they have
                            successfully logged in to Facebook.
                          
         progress(i, j) - the application is currently processing friend 'i' of
                          'j'. Should be used to provide the user a progress bar.
                          
      Other parameters:
         twitter - a username/password tuple or None if Twitter should not be
                   used. Defaults to None.
                   
         facebook - boolean indicating if Facebook should be used. Defaults to
                    False.
                          """

    self.gc       = gnome.gconf.client_get_default()
    self.callback = callback
    self.twitter  = twitter
    self.facebook = facebook

    # -- Check the environment is going to work...
    #
    if (self.gc.get_string('/desktop/gnome/url-handlers/http/command') == 'epiphany %s'):
      raise Exception('Browser in gconf invalid (see NB#136012). Installation error.')

    # -- Get private keys for this app...
    #
    key_app    = self.gc.get_string('/apps/maemo/hermes/key_app')
    key_secret = self.gc.get_string('/apps/maemo/hermes/key_secret')
    if (key_app is None or key_secret is None):
      raise Exception('No Facebook application keys found. Installation error.')

    self.fb = Facebook(key_app, key_secret)
    self.fb.desktop = True


  # -----------------------------------------------------------------------
  def do_fb_login(self):
    """Perform authentication against Facebook and store the result in gconf
         for later use. Uses the 'need_auth' and 'block_for_auth' methods on
         the callback class. The former allows a message to warn the user
         about what is about to happen to be shown; the second is to wait
         for the user to confirm they have logged in."""
    self.fb.session_key = None
    self.fb.secret = None
    self.fb.uid = None
    
    self.callback.need_auth()
    self.fb.auth.createToken()
    self.fb.login()
    self.callback.block_for_auth()
    session = self.fb.auth.getSession()

    self.gc.set_string('/apps/maemo/hermes/session_key', session['session_key'])
    self.gc.set_string('/apps/maemo/hermes/secret_key', session['secret'])
    self.gc.set_int('/apps/maemo/hermes/uid', session['uid'])


  # -----------------------------------------------------------------------
  def load_friends(self):
    """Load information on the authenticated user's friends. If no user is
       currently authenticated, prompts for a login."""

    self.friends = {}
    self.blocked_pictures = []
    self.callback.progress(0, 0)
    
    # -- Get a user session and retrieve Facebook friends...
    #
    if self.facebook:
      if self.fb.session_key is None:
        self.fb.session_key = self.gc.get_string('/apps/maemo/hermes/session_key')
        self.fb.secret = self.gc.get_string('/apps/maemo/hermes/secret_key')
        self.fb.uid = self.gc.get_int('/apps/maemo/hermes/uid')

      # Check the available session is still valid...
      while True:
        try:
          if self.fb.users.getLoggedInUser() and self.fb.session_key:
            break
        except FacebookError:
          pass
        self.do_fb_login()

      # Get the list of friends...
      attrs = ['uid', 'name', 'pic_big', 'birthday_date', 'profile_url']
      for friend in self.fb.users.getInfo(self.fb.friends.get(), attrs):
        friend['pic'] = friend[attrs[2]]
        self.friends[friend['name']] = friend
        if not friend['pic']:
          self.blocked_pictures.append(friend)
          
    # -- Retrieve following information from Twitter...
    #
    if self.twitter is not None:
      (user, passwd) = self.twitter
      api = twitter.Api(username=user, password=passwd)
      users = api.GetFriends()
      for friend in api.GetFriends():
        self.friends[friend.name] = {'name': friend.name, 'pic': friend.profile_image_url, 'birthday_date': None, 'twitter_url': 'http://twitter.com/%s' % (friend.screen_name), 'homepage' : friend.url}
  
    # TODO What if the user has *no* contacts?

  
  # -----------------------------------------------------------------------
  def sync_contacts(self, resync = False):
    """Synchronise Facebook profiles to contact database. If resync is false,
       no existing information will be overwritten."""

    # -- Find addresses...
    #
    print "+++ Syncing contacts..."
    addresses = evolution.ebook.open_addressbook('default')
    print "+++ Addressbook opened..."
    store = ContactStore(addresses)
    print "+++ Contact store created..."
    self.updated = []
    self.unmatched = []
    self.matched = []
    contacts = addresses.get_all_contacts()
    contacts.sort(key=lambda obj: obj.get_name())
    current = 0
    maximum = len(contacts)
    for contact in contacts:
      current += 1
      self.callback.progress(current, maximum)
      found = False
      for name in names.variants(contact.get_name()):
        if name in self.friends:
          friend = self.friends[name]
          found = True
          updated = False
      
          if friend['pic'] and (resync or contact.get_property('photo') is None):
            updated = store.set_photo(contact, friend['pic']) or updated
        
          if friend['birthday_date'] and (resync or contact.get_property('birth-date') is None):
            date_str = friend['birthday_date'].split('/')
            date_str.append('0')
            updated = store.set_birthday(contact, int(date_str[1]),
                                                  int(date_str[0]),
                                                  int(date_str[2])) or updated

          if 'profile_url' in friend and friend['profile_url']:
            updated = store.add_url(contact, friend['profile_url'], unique='facebook.com') or updated
            
          if 'twitter_url' in friend and friend['twitter_url']:
            updated = store.add_url(contact, friend['twitter_url'], unique='twitter.com') or updated
            
          if 'homepage' in friend and friend['homepage']:
            updated = store.add_url(contact, friend['homepage']) or updated
    
          if updated:
            self.updated.append(contact)
            addresses.commit_contact(contact)
            print "Saved changes to [%s]" % (contact.get_name())
            
          break
      
      if found:
        self.matched.append(contact)
      else:
        self.unmatched.append(contact)

    store.close()

