#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2011 Boris Pohler <boris@pohlers-web.de>

# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:

# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.

# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
# THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
# FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
# DEALINGS IN THE SOFTWARE.
 
# Importe
from __future__ import with_statement
import sys
import dbus
import time
import os
import subprocess
import ConfigParser
from functools import partial
from PyQt4.QtGui import QApplication, QMainWindow, QDialog, QFileDialog, QInputDialog, QMessageBox
from PyQt4.QtCore import Qt, SIGNAL, SLOT, QTimer
from PyQt4.uic import loadUi
from PyQt4.phonon import Phonon 
reload(sys)
sys.setdefaultencoding('utf-8')
 
class N900_alarm(QMainWindow):
    def __init__(self):
        QMainWindow.__init__(self,parent=None)
        self.ui = loadUi("handsoff.ui")
        self.ui.setAttribute(Qt.WA_Maemo5PortraitOrientation, True)
        self.ui.showFullScreen()

        self.ui.pushButton_0.clicked.connect(self.button_pressed)
        self.ui.pushButton_1.clicked.connect(self.button_pressed)
        self.ui.pushButton_2.clicked.connect(self.button_pressed)
        self.ui.pushButton_3.clicked.connect(self.button_pressed)
        self.ui.pushButton_4.clicked.connect(self.button_pressed)
        self.ui.pushButton_5.clicked.connect(self.button_pressed)
        self.ui.pushButton_6.clicked.connect(self.button_pressed)
        self.ui.pushButton_7.clicked.connect(self.button_pressed)
        self.ui.pushButton_8.clicked.connect(self.button_pressed)
        self.ui.pushButton_9.clicked.connect(self.button_pressed)
        self.ui.pushButton_back.clicked.connect(self.button_pressed)
        self.ui.pushButton_clear.clicked.connect(self.button_pressed)
        self.ui.pushButton_alarm.clicked.connect(self.arm)
        self.ui.pushButton_settings.clicked.connect(self.show_settings)
        self.ui.pushButton_quit.clicked.connect(self.ui.close)
        self.ui.pushButton_about.clicked.connect(self.show_about)

        self.armed = False
        self.alarm_running = False
        self.sounds_alarm = ('alarm1.mp3','alarm2.mp3','alarm3.mp3','alarm4.mp3')
        self.sounds_arm = ('armed1.mp3','armed2.mp3','armed3.mp3')
    
        self.reader = AccelReader()
        self.proxy = Proximity_Sensor()

         # for playing sounds with phonon
        self.m_media = Phonon.MediaObject(self)
        audioOutput = Phonon.AudioOutput(Phonon.MusicCategory, self)
        Phonon.createPath(self.m_media, audioOutput)
        self.soundfile = ""

        # Config stuff
        self.config = ConfigParser.ConfigParser()
        if os.path.exists(os.path.expanduser('~/.config/handsoff.cfg')):
            self.readconf()
        else:
            self.defaultconf()
            self.readconf()

    def save_config(self):
        with open(os.path.expanduser('~/.config/handsoff.cfg'), 'wb') as configfile:
            self.config.write(configfile)            

    def readconf(self):
        self.config.readfp(open(os.path.expanduser('~/.config/handsoff.cfg'), 'rwb'))
        self.time_to_disarm = int(self.config.get('General', 'time_to_disarm'))
        self.time_to_arm = int(self.config.get('General', 'time_to_arm'))
        self.sound_alarm = int(self.config.get('General', 'sound_alarm'))
        self.sound_arm = int(self.config.get('General', 'sound_arm'))
        self.pin = self.config.get('General', 'pin')
        self.alarm_on_proxy = int(self.config.get('General', 'alarm_on_proxy'))
        self.alarm_on_accel = int(self.config.get('General', 'alarm_on_accel'))

    def defaultconf(self):
        self.config.add_section('General')
        self.config.set('General', 'time_to_disarm', '5')
        self.config.set('General', 'time_to_arm', '5')
        self.config.set('General', 'sound_alarm', '0')
        self.config.set('General', 'sound_arm', '0')
        self.config.set('General', 'pin', '')
        self.config.set('General', 'alarm_on_proxy','1')
        self.config.set('General', 'alarm_on_accel','0')
        self.save_config()

    def arm(self):
        self.armed = True
        self.pin = str(self.ui.lineEdit_pin.text())
        self.ui.lineEdit_pin.clear()
        if self.pin:
            self.config.set('General', 'pin', self.pin)
            self.show_notification_dialog("N900 will be armed in %s seconds \n  New PIN: %s will be safed" % (str(self.time_to_arm), self.pin))
            self.save_config()
        else:
            self.pin = self.config.get('General', 'pin')
            self.show_notification_dialog("N900 will be armed in %s seconds \n PIN: %s " % (str(self.time_to_arm), self.pin))
        self.toggle_buttons(False)
        self.timer2 = QTimer()
        self.timer2.timeout.connect(partial(self.play_sound,self.sounds_arm[self.sound_arm]))
        self.timer2.timeout.connect(self.start_check)
        self.timer2.start(self.time_to_arm*1000)

    def start_check(self):
        self.timer2.stop()
        if self.alarm_on_accel:
            self.accel_values = self.reader.values()
        self.timer = QTimer()
        self.timer.timeout.connect(self.check)
        self.timer.start(1000)

    def toggle_buttons(self,value):
        self.ui.pushButton_settings.setEnabled(value)
        self.ui.pushButton_quit.setEnabled(value)
        self.ui.pushButton_alarm.setEnabled(value)
        self.ui.pushButton_about.setEnabled(value)

    def disarm(self):
        # unlock
        systembus = dbus.SystemBus()
        tklock = systembus.get_object('com.nokia.mce','/com/nokia/mce/request')
        tklock.req_tklock_mode_change(dbus.String("unlocked"))
        # change profile to general
        sessionbus = dbus.SessionBus()
        profile = sessionbus.get_object('com.nokia.profiled', '/com/nokia/profiled')
        profile.set_profile('general', dbus_interface='com.nokia.profiled')
        # disable buttons
        subprocess.Popen(["run-standalone.sh", "pkill", "-STOP", "systemui"])
        # set Volume to Max
        os.system("dbus-send --session --type=method_call\
                  --dest=com.nokia.mafw.renderer.Mafw-Gst-Renderer-Plugin.gstrenderer\
                  /com/nokia/mafw/renderer/gstrenderer\
                  com.nokia.mafw.extension.set_extension_property\
                  string:'volume' variant:uint32:95")
        while self.armed:
            self.dialog = Disarm_Dlg(self)
            self.dialog.ui.setAttribute(Qt.WA_Maemo5PortraitOrientation, True)
            if not self.alarm_running:
                self.timer1 = QTimer()
                self.timer1.singleShot(self.time_to_disarm*1000,self.run_alarm)
            if self.dialog.ui.exec_()==1:
                if self.pin == str(self.dialog.ui.lineEdit_pin.text()):
                    self.armed = False
                    self.alarm_running = False
                    subprocess.Popen(["run-standalone.sh", "pkill", "-CONT", "systemui"])
                    self.m_media.stop()
                    self.toggle_buttons(True)
                    try:
                        self.m_media.finished.disconnect()
                    except:
                        pass
                    self.show_notification_dialog("N900 disarmed")
                    break

    def run_alarm(self):
        self.alarm_running = True
        if self.armed:
            self.play_sound(self.sounds_alarm[self.sound_alarm])
            self.m_media.finished.connect(partial(self.play_sound,self.sounds_alarm[self.sound_alarm]))
            
    def check(self):
        if self.alarm_on_proxy:
            if not self.proxy.check_status():
                self.timer.stop()
                self.disarm()
        if self.alarm_on_accel:
            abw = abs(self.reader.values()-self.accel_values)
            if abw > 100:
                self.timer.stop()
                self.disarm()

    def button_pressed(self):
        button_text = str(self.sender().text())
        if button_text in ('0123456789'):
           self.ui.lineEdit_pin.setText(self.ui.lineEdit_pin.text()+button_text)
        if button_text == '<':
           self.ui.lineEdit_pin.backspace()
        if button_text == 'C':
           self.ui.lineEdit_pin.setText("")

    def show_settings(self):
        config=(self.time_to_disarm,self.time_to_arm,
                     self.sound_alarm, self.sound_arm,
                     self.alarm_on_proxy, self.alarm_on_accel)
        self.settings_dialog = Settings_Dlg(config, self)
        self.settings_dialog.ui.setAttribute(Qt.WA_Maemo5PortraitOrientation, True)
        if self.settings_dialog.ui.exec_()==1:
            (self.time_to_disarm,self.time_to_arm, 
             self.sound_alarm, self.sound_arm,
             self.alarm_on_proxy, self.alarm_on_accel) = self.settings_dialog.get_config()
            self.config.set('General', 'time_to_disarm', self.time_to_disarm)
            self.config.set('General', 'time_to_arm', self.time_to_arm)
            self.config.set('General', 'sound_alarm', self.sound_alarm)
            self.config.set('General', 'sound_arm', self.sound_arm)
            self.config.set('General', 'alarm_on_proxy', self.alarm_on_proxy)
            self.config.set('General', 'alarm_on_accel',self.alarm_on_accel)
            self.save_config()

    def show_notification_dialog(self,message,mode="single"):
        '''
        provides notification dialog
        '''
        bus = dbus.SystemBus()
        iface = dbus.Interface(bus.get_object('org.freedesktop.Notifications',
                                        '/org/freedesktop/Notifications'),
                                        'org.freedesktop.Notifications')
        if mode == "single":
            iface.SystemNoteInfoprint(message)
        elif mode == "multiline":
            iface.SystemNoteDialog(str(message), dbus.UInt32(0), 'Ok')

    def show_about(self):
        self.msg_about = QMessageBox(self)
        self.msg_about.setAttribute(Qt.WA_Maemo5PortraitOrientation, True)
        self.msg_about.setWindowTitle("About HandsOff") 
        self.msg_about.setText(u"""
                          <p>This program turns your n900 into an alarm system.
                          Alarm can be triggerd on movement or proximity sensor or both. 
                          <p>You can change multiple options in the settings section.
                          <p>The program is still in development so please send any found bugs to:
                          <p>boris@pohlers-web.de""")
        self.msg_about.exec_()

    def play_sound(self,file=""):
        # to use volume control, app needs an applicationname (app.setApplicationName, at the end)
        if file:
           self.m_media.setCurrentSource(Phonon.MediaSource(file))
           self.m_media.play()
 
class AccelReader:
    # found at http://lesloueizeh.com/jdieter/accelreader.py
    def __init__(self):
        bus = dbus.SystemBus()
        self.accel = bus.get_object('com.nokia.mce', 
                                            '/com/nokia/mce/request', 
                                            'com.nokia.mce.request')
    def values(self):
        orientation, stand, face, x, y, z = self.accel.get_device_orientation()
        return abs(x)+abs(y)+abs(z)           

class Proximity_Sensor:
    def __init__(self):
        self.closed = False
 
    def check_status(self):
        f = open('/sys/devices/platform/gpio-switch/proximity/state','r')
	line = f.readline()
        f.close()
        if line == 'closed\n':
            self.closed = True
        else:
            self.closed = False
        return self.closed

class Disarm_Dlg(QDialog):
    def __init__(self,parent=None):
        QDialog.__init__(self)
        self.ui = loadUi("disarm_dlg.ui")
        self.ui.pushButton_0.clicked.connect(self.button_pressed_dialog)
        self.ui.pushButton_1.clicked.connect(self.button_pressed_dialog)
        self.ui.pushButton_2.clicked.connect(self.button_pressed_dialog)
        self.ui.pushButton_3.clicked.connect(self.button_pressed_dialog)
        self.ui.pushButton_4.clicked.connect(self.button_pressed_dialog)
        self.ui.pushButton_5.clicked.connect(self.button_pressed_dialog)
        self.ui.pushButton_6.clicked.connect(self.button_pressed_dialog)
        self.ui.pushButton_7.clicked.connect(self.button_pressed_dialog)
        self.ui.pushButton_8.clicked.connect(self.button_pressed_dialog)
        self.ui.pushButton_9.clicked.connect(self.button_pressed_dialog)
        self.ui.pushButton_back.clicked.connect(self.button_pressed_dialog)
        self.ui.pushButton_clear.clicked.connect(self.button_pressed_dialog)

    def button_pressed_dialog(self):
        button_text = str(self.sender().text())
        if button_text in ('0123456789'):
           self.ui.lineEdit_pin.setText(self.ui.lineEdit_pin.text()+button_text)
        if button_text == '<':
           self.ui.lineEdit_pin.backspace()
        if button_text == 'C':
           self.ui.lineEdit_pin.setText("")

class Settings_Dlg(QDialog):
    def __init__(self, config, parent=None):
        QDialog.__init__(self)
        self.time_to_disarm,self.time_to_arm, self.sound_alarm, self.sound_arm, self.alarm_on_proxy, self.alarm_on_accel = config
        self.ui = loadUi("settings.ui")

        self.sounds_alarm=('Sound 1', 'Sound 2', 'Sound 3','Sound 4')
        self.sounds_arm=('Sound 1', 'Sound 2', 'Sound 3')

        self.ui.horizontalSlider_alarm.setValue(self.time_to_disarm)
        self.ui.horizontalSlider_arm.setValue(self.time_to_arm)
        self.update_label_alarm(self.time_to_disarm)
        self.update_label_arm(self.time_to_arm)
        self.ui.label_sound_alarm.setText(self.sounds_alarm[self.sound_alarm])
        self.ui.label_sound_arm.setText(self.sounds_arm[self.sound_arm])

        self.ui.checkBox_proxy.setChecked(int(self.alarm_on_proxy))
        self.ui.checkBox_accel.setChecked(int(self.alarm_on_accel))
        self.ui.horizontalSlider_alarm.valueChanged.connect(self.update_label_alarm)
        self.ui.horizontalSlider_arm.valueChanged.connect(self.update_label_arm)
        self.ui.pushButton_alarm_fwd.clicked.connect(partial(self.change_alarm_sound,1))
        self.ui.pushButton_alarm_rwd.clicked.connect(partial(self.change_alarm_sound,-1))
        self.ui.pushButton_arm_fwd.clicked.connect(partial(self.change_arm_sound,1))
        self.ui.pushButton_arm_rwd.clicked.connect(partial(self.change_arm_sound,-1))
        
    def update_label_alarm(self, value):
        self.ui.label_time_alarm.setText(str(value)+ "s")

    def update_label_arm(self, value):
        self.ui.label_time_arm.setText(str(value)+ "s")

    def change_alarm_sound(self,step):
        self.sound_alarm = (self.sound_alarm + step) % len(self.sounds_alarm)
        self.ui.label_sound_alarm.setText(self.sounds_alarm[self.sound_alarm])
        print self.sound_alarm

    def change_arm_sound(self,step):
        self.sound_arm = (self.sound_arm + step) % len(self.sounds_arm)
        self.ui.label_sound_arm.setText(self.sounds_arm[self.sound_arm])
        print self.sound_arm

    def get_config(self):
        self.time_to_disarm = self.ui.horizontalSlider_alarm.value()
        self.time_to_arm = self.ui.horizontalSlider_arm.value()
        self.alarm_on_proxy = (1 if self.ui.checkBox_proxy.isChecked() else 0)
        self.alarm_on_accel = (1 if self.ui.checkBox_accel.isChecked() else 0)
        return (self.time_to_disarm,self.time_to_arm, self.sound_alarm, 
                self.sound_arm, self.alarm_on_proxy, self.alarm_on_accel)

if __name__ == "__main__":
    app = QApplication(sys.argv)
    #app.setApplicationName('handsoff') # for Phonon
    n900_alarm = N900_alarm()
    sys.exit(app.exec_())
