#include "filedetailsdialog.h"

FileDetailsDialog::FileDetailsDialog(QWidget *parent, QFileInfo info) :
    QDialog(parent),
    ui(new Ui::FileDetailsDialog)
{
    ui->setupUi(this);

    this->info = info;

    user = getenv("USER");

    // Change color of descriptive labels
    QPalette labelPalette = ui->nameLabel->palette();
    labelPalette.setBrush(QPalette::Active, QPalette::WindowText, QBrush(QMaemo5Style::standardColor("SecondaryTextColor")));
    ui->nameLabel->setPalette(labelPalette);
    ui->targetLabel->setPalette(labelPalette);
    ui->sizeLabel->setPalette(labelPalette);
    ui->modificationLabel->setPalette(labelPalette);
    ui->ownerLabel->setPalette(labelPalette);
    ui->permissionsLabel->setPalette(labelPalette);

    // Name
    ui->nameEdit->setText(info.fileName());
    ui->nameEdit->setValidator(new QRegExpValidator(QRegExp("[^/]+"), this));

    // Symlink target
    if (info.isSymLink()) {
        char buffer[4096];
        QString linkTarget = QString::fromUtf8(buffer, readlink(info.absoluteFilePath().toUtf8(), buffer, 4096));
        ui->targetInfo->setText(linkTarget);
    } else {
        ui->targetLabel->hide();
        ui->targetInfo->hide();
    }

    // Size
    qint64 size = info.size();
    ui->sizeInfo->setText(FileSystemModel::sizeString(size) + QString(" (%1 B)").arg(size));

    // Ownership
    ui->ownerInfo->setText(QString("%1:%2").arg(info.owner(), info.group()));

    // Modification date
    ui->modificationInfo->setText(info.lastModified().toString());

    // Permissions
    QFile::Permissions permissions = info.permissions();
    if (permissions & QFile::ReadOwner) ui->permUserReadBox->setChecked(true);
    if (permissions & QFile::WriteOwner) ui->permUserWriteBox->setChecked(true);
    if (permissions & QFile::ExeOwner) ui->permUserExeBox->setChecked(true);
    if (permissions & QFile::ReadGroup) ui->permGroupReadBox->setChecked(true);
    if (permissions & QFile::WriteGroup) ui->permGroupWriteBox->setChecked(true);
    if (permissions & QFile::ExeGroup) ui->permGroupExeBox->setChecked(true);
    if (permissions & QFile::ReadOther) ui->permOtherReadBox->setChecked(true);
    if (permissions & QFile::WriteOther) ui->permOtherWriteBox->setChecked(true);
    if (permissions & QFile::ExeOther) ui->permOtherExeBox->setChecked(true);

    if (!(QFileInfo(info.absolutePath()).permissions() & QFile::WriteUser))
        ui->nameEdit->setEnabled(false);

    if (!(info.owner() == user || user == "root"))
        foreach (QCheckBox *box, ui->permissionsWidget->findChildren<QCheckBox*>())
            box->setEnabled(false);

    ui->nameEdit->setFocus();
    this->adjustSize();

    connect(ui->saveButton, SIGNAL(clicked()), this, SLOT(save()));
}

FileDetailsDialog::~FileDetailsDialog()
{
    delete ui;
}

void FileDetailsDialog::save()
{

    if (ui->nameEdit->text().isEmpty()) {
        QMaemo5InformationBox::information(this, tr("Filename is empty"));
        return;
    }

    QFile file(info.absoluteFilePath());

    if (ui->permUserReadBox->isEnabled()) {
        QFile::Permissions permissions;
        if (ui->permUserReadBox->isChecked()) permissions |= QFile::ReadOwner;
        if (ui->permUserWriteBox->isChecked()) permissions |= QFile::WriteOwner;
        if (ui->permUserExeBox->isChecked()) permissions |= QFile::ExeOwner;
        if (ui->permGroupReadBox->isChecked()) permissions |= QFile::ReadGroup;
        if (ui->permGroupWriteBox->isChecked()) permissions |= QFile::WriteGroup;
        if (ui->permGroupExeBox->isChecked()) permissions |= QFile::ExeGroup;
        if (ui->permOtherReadBox->isChecked()) permissions |= QFile::ReadOther;
        if (ui->permOtherWriteBox->isChecked()) permissions |= QFile::WriteOther;
        if (ui->permOtherExeBox->isChecked()) permissions |= QFile::ExeOther;

        file.setPermissions(permissions);
        qDebug() << "Saving permissions" << file.error();
    }

    if (ui->nameEdit->isEnabled()) {
        file.rename(info.absolutePath() + '/' + ui->nameEdit->text());
        qDebug() << "Renaming" << file.error();
    }

    this->close();
}
