#include <QMenuBar>
#include <QDebug>
#include <QPainter>
#include <QHBoxLayout>
#include <QMaemo5InformationBox>
#include <QMessageBox>
#include "feedswindow.h"
#include "entrieswindow.h"

#define APPNAME "grr"
#define APPVERSION "0.0.5"

FeedsWindow::FeedsWindow(QWidget *) : QMainWindow(0) {
	setAttribute(Qt::WA_Maemo5StackedWindow);
	
	setWindowIcon(QIcon("/usr/share/icons/hicolor/64x64/apps/grr.png"));
	
	settings = new QSettings("dufo.be", "grr"); 

	menuBar()->addAction(tr("Settings"), this, SLOT(showSettings()));
	menuBar()->addAction(tr("Sync now"), this, SLOT(sync()));
	menuBar()->addAction(tr("About"), this, SLOT(about()));

	QActionGroup *filter_group = new QActionGroup(this);
	show_all = new QAction(tr("Show All"), filter_group);
	show_all->setCheckable(true);
	show_updated= new QAction(tr("Show Updated"), filter_group);
	show_updated->setCheckable(true);

	if(settings->value("show_updated", false).toBool())
		show_updated->setChecked(true);
	else
		show_all->setChecked(true);

	menuBar()->addActions(filter_group->actions());

	setWindowTitle(APPNAME);

	list = new QListView();
	setCentralWidget(list);
	list->setItemDelegate(new FeedListDelegate(this));
	
	connect(list, SIGNAL(activated(const QModelIndex &)),
		SLOT(feedSelected(const QModelIndex &)));

	reader = new GoogleReader();

	connect(reader, SIGNAL(updateUnreadComplete()), SLOT(feedsUpdated()));
	connect(reader, SIGNAL(allReadChanged()), SLOT(refreshModel()));
	connect(reader, SIGNAL(loginFailed(QString)), SLOT(loginFailed(QString)));
	
	connect(show_updated, SIGNAL(toggled(bool)), SLOT(showUpdated(bool)));
	
	if((settings->value("login", "").toString() == "") || 
		(settings->value("passwd", "").toString() == "")) {
		emit loginFailed("Incomplete username or password");
	}
	else {
		reader->setLogin(settings->value("login", "").toString());
		reader->setPasswd(settings->value("passwd", "").toString());
		sync();
	}
}

FeedsWindow::~FeedsWindow() {
}

void FeedsWindow::showSettings() {
	SettingsDialog settingsDialog(this, settings);
	if(settingsDialog.exec() == QDialog::Accepted) {
		reader->logOut();
		reader->setLogin(settings->value("login", "").toString());
		reader->setPasswd(settings->value("passwd", "").toString());
		sync();
	}
}

void FeedsWindow::sync() {
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator, true);
	reader->updateSubscriptions();
	reader->updateUnread();
}

void FeedsWindow::refreshModel() {
	QList<Feed *>feeds = reader->getFeeds();
	QAbstractItemModel *old_model = list->model();
	FeedListModel *new_model = new FeedListModel(this, feeds, show_updated->isChecked());
	list->setModel(new_model);
	connect(show_updated, SIGNAL(toggled(bool)), new_model, SLOT(showUpdated(bool)));
	delete(old_model);
}

void FeedsWindow::feedsUpdated() {
	refreshModel();
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
}

void FeedsWindow::feedSelected(const QModelIndex &index) {
	Feed *f = qVariantValue<Feed *>(index.data());
	EntriesWindow *w = new EntriesWindow(this, f);
	w->show();
}

void FeedsWindow::showUpdated(bool updated) {
	settings->setValue("show_updated", updated);
}

void FeedsWindow::loginFailed(QString message) {
	setAttribute(Qt::WA_Maemo5ShowProgressIndicator, false);
	QMaemo5InformationBox::information(this, message, QMaemo5InformationBox::DefaultTimeout);
	showSettings();
}

void FeedsWindow::about() {
	QMessageBox::about(this, APPNAME " " APPVERSION, "");
}

int FeedListModel::rowCount(const QModelIndex &) const {
	int size = 0;
	
	if(show_updated) {
		for(int i = 0; i < feed_list.size(); i++) {
			if(feed_list.at(i)->unread || feed_list.at(i)->special)
				size++;
		}
	}
	else
		size = feed_list.size();

	return size;;
}

QVariant FeedListModel::data(const QModelIndex &index, int role) const {
	if(!index.isValid())
		return QVariant();

	if(index.row() >= rowCount() || index.row() < 0) {
		return QVariant();
	}

	if(role == Qt::DisplayRole) {
		if(show_updated) {
			int i, j;
			for(i = 0, j = 0; i < feed_list.size(); i++) {
				if(feed_list.at(i)->unread || feed_list.at(i)->special) j++;
				if(j > index.row())
					return qVariantFromValue(feed_list.at(i));
			}
		}
		else
			return qVariantFromValue(feed_list.at(index.row()));
	}

	return QVariant();
}

void FeedListModel::showUpdated(bool updated) {
	beginResetModel();
	show_updated = updated;
	endResetModel();
}

void FeedListDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option,
	const QModelIndex &index) const {
   
	QStyledItemDelegate::paint(painter, option, index);

	Feed *f = qVariantValue<Feed *>(index.data());

	QRect rect = option.rect;
	rect.adjust(20, 8, -20, -8);
	QFont font = option.font;
	int flags = Qt::AlignVCenter;

	painter->save();

	if(f->cat_label != "")
		flags = Qt::AlignTop;

	if(f->unread) {
		if(!(option.state & QStyle::State_Selected))
			painter->setPen(option.palette.highlight().color());

		painter->drawText(rect, flags | Qt::AlignRight, QString("%1").arg(f->unread)); 
	}
	
	painter->drawText(rect, flags | Qt::AlignLeft, f->title); 

	if(f->cat_label != "") {
		painter->setPen(option.palette.mid().color());
		font.setPointSizeF(font.pointSizeF() * 0.70);
		painter->setFont(font);
		painter->drawText(rect, Qt::AlignBottom | Qt::AlignLeft, f->cat_label);
	}
	
	painter->restore();
}

SettingsDialog::SettingsDialog(QWidget *parent, QSettings *s) : QDialog(parent) {
	settings = s;

	loginLabel = new QLabel(tr("Login:"));
	loginEdit = new QLineEdit(settings->value("login", "").toString());
	loginLabel->setBuddy(loginEdit);

	passwdLabel = new QLabel(tr("Password:"));
	passwdEdit = new QLineEdit(settings->value("passwd", "").toString());
	passwdEdit->setEchoMode(QLineEdit::PasswordEchoOnEdit);
	passwdLabel->setBuddy(passwdEdit);

	saveButton = new QPushButton(tr("Save"));
	saveButton->setDefault(true);
	cancelButton = new QPushButton(tr("Cancel"));
	cancelButton->setDefault(false);

	buttonBox = new QDialogButtonBox(Qt::Vertical);
	buttonBox->addButton(cancelButton, QDialogButtonBox::ActionRole);
	buttonBox->addButton(saveButton, QDialogButtonBox::ActionRole);

	QHBoxLayout *loginLayout = new QHBoxLayout;
	loginLayout->addWidget(loginLabel);
	loginLayout->addWidget(loginEdit);

	QHBoxLayout *passwdLayout = new QHBoxLayout;
	passwdLayout->addWidget(passwdLabel);
	passwdLayout->addWidget(passwdEdit);

	QVBoxLayout *leftLayout = new QVBoxLayout;
	leftLayout->addLayout(loginLayout);
	leftLayout->addLayout(passwdLayout);
	leftLayout->addStretch(1);

	QGridLayout *mainLayout = new QGridLayout;
	mainLayout->setSizeConstraint(QLayout::SetFixedSize);
	mainLayout->addLayout(leftLayout, 0, 0);
	mainLayout->addWidget(buttonBox, 0, 1);
	setLayout(mainLayout);

	setWindowTitle(tr("Settings"));

	connect(saveButton, SIGNAL(clicked()), SLOT(accept()));
	connect(cancelButton, SIGNAL(clicked()), SLOT(reject()));
}
		
void SettingsDialog::accept() {
	settings->setValue("login", loginEdit->text());
	settings->setValue("passwd", passwdEdit->text());
	QDialog::accept();
}
