/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <math.h>
#include <string.h>
#include <ctype.h>

#include <glib.h>
#include <glib/gstdio.h>

#include "gpxview.h"

char strlastchr(char *str) {
  return str[strlen(str)]-1;
}

/* make sure the entire path "dir" exists and create it if not */
int checkdir(char *dir) {
  struct stat filestat;
  char *p = dir, tmp;

  /* don't try to create root dir */
  if(p[0] == '/') p++;

  do {
    while(*p && *p != '/') p++;
    tmp = *p;
    *p = 0;

    int err = stat(dir, &filestat);
    if(err) {
      if(mkdir(dir, S_IRWXU) != 0) {
	perror("mkdir()");
	*p++ = tmp;
	return -1;
      }
    } else {
      if(!filestat.st_mode & S_IFDIR) {
	printf("File %s exists and is _no_ directory\n", dir);
	*p++ = tmp;
	return -1;
      }
    }
    
    *p++ = tmp;
  } while(tmp && strchr(p, '/'));

  return 0;
}

void pos_lat_str(char *str, int len, float latitude) {
  char *c = _("N");
  float integral, fractional;

  if(isnan(latitude)) 
    str[0] = 0;
  else {
    if(latitude < 0) { latitude = fabs(latitude); c = _("S"); }
    fractional = modff(latitude, &integral);

    snprintf(str, len, "%s %02d° %06.3f'", c, (int)integral, fractional*60.0);
  }
}

GtkWidget *pos_lat(float latitude, int size, int strikethrough) {
  char str[32];

  pos_lat_str(str, sizeof(str), latitude);
  return gtk_label_attrib(str, size, strikethrough);
}

void pos_lon_str(char *str, int len, float longitude) {
  char *c = _("E");
  float integral, fractional;

  if(isnan(longitude)) 
    str[0] = 0;
  else {
    if(longitude < 0) { longitude = fabs(longitude); c = _("W"); }
    fractional = modff(longitude, &integral);

    snprintf(str, len, "%s %03d° %06.3f'", c, (int)integral, fractional*60.0);
  }
}

GtkWidget *pos_lon(float longitude, int size, int strikethrough) {
  char str[32];

  pos_lon_str(str, sizeof(str), longitude);
  return gtk_label_attrib(str, size, strikethrough);
}

float pos_parse_lat(char *str) {
  int integral_int;
  float fractional;
  char c;

  if(sscanf(str, "%c %d° %f'", &c, &integral_int, &fractional) == 3) {
    c = toupper(c);
    
    if(c != 'S' && c != 'N')
      return NAN;
    
    /* prevent -0.0 */
    if(!integral_int && (fractional == 0.0))
      return 0.0;

    return ((c == 'S')?-1:+1) * (integral_int + fractional/60.0);
  }
  
  return NAN;
}

float pos_parse_lon(char *str) {
  int integral_int;
  float fractional;
  char c;

  if(sscanf(str, "%c %d° %f'", &c, &integral_int, &fractional) == 3) {
    c = toupper(c);

    /* O is german "Ost" for "East" */
    if(c != 'E' && c != 'W' && c != 'O')
      return NAN;

    /* prevent -0.0 */
    if(!integral_int && (fractional == 0.0))
      return 0.0;

    return ((c == 'W')?-1:+1) * (integral_int + fractional/60.0);
  }

  return NAN;
}

const char *pos_get_bearing_str(pos_t from, pos_t to) {
  static const char *bear_str[]={ 
    "N", "NE", "E", "SE", "S", "SW", "W", "NW" };
  int idx = (gpx_pos_get_bearing(from, to)+22.5)/45.0;
  /* make sure we stay in icon bounds */
  while(idx < 0) idx += 8;
  while(idx > 7) idx -= 8;
  return _(bear_str[idx]);
}

/* the maemo font size is quite huge, so we adjust some fonts */
/* differently on maemo and non-maemo. Basically "BIG" does nothing */
/* on maemo and "SMALL" only does something on maemo */
#ifdef USE_MAEMO
#define MARKUP_SMALL "<span size='small'>%s</span>"
GtkWidget *gtk_label_small(char *str) {
  GtkWidget *label = gtk_label_new("");
  char *markup = g_markup_printf_escaped(MARKUP_SMALL, str);
  gtk_label_set_markup(GTK_LABEL(label), markup);
  g_free(markup);
  return label;
}
#else
#define MARKUP_BIG "<span size='x-large'>%s</span>"
GtkWidget *gtk_label_big(char *str) {
  GtkWidget *label = gtk_label_new("");
  char *markup = g_markup_printf_escaped(MARKUP_BIG, str);
  gtk_label_set_markup(GTK_LABEL(label), markup);
  g_free(markup);
  return label;
}
#endif

void gtk_label_attrib_set(GtkWidget *label, 
			  char *str, int size, int strikethrough) {
  char format[80];

  snprintf(format, sizeof(format), "<span%s%s%s>%%s</span>", 
#ifdef USE_MAEMO
   (size==SIZE_SMALL)?" size='small'":"",
#else
   (size==SIZE_BIG)?" size='x-large'":"",
#endif
   strikethrough?" strikethrough='yes'":"",
   (strikethrough==STRIKETHROUGH_RED)?" strikethrough_color='red'":"");

  char *markup = g_markup_printf_escaped(format, str);
  //  printf("markup = %s\n", markup);
  gtk_label_set_markup(GTK_LABEL(label), markup);
  g_free(markup);
}

GtkWidget *gtk_label_attrib(char *str, int size, int strikethrough) {
  GtkWidget *label = gtk_label_new("");
  gtk_label_attrib_set(label, str, size, strikethrough);
  return label;
}

GtkWidget *gtk_button_attrib(char *str, int size, int strikethrough) {
  GtkWidget *button = gtk_button_new_with_label("");
  gtk_label_attrib_set(gtk_bin_get_child(GTK_BIN(button)), 
		       str, size, strikethrough);
  return button;
}

void textbox_disable(GtkWidget *widget) {
  gtk_editable_set_editable(GTK_EDITABLE(widget), FALSE);
  gtk_widget_set_sensitive(widget, FALSE);
}

void textbox_enable(GtkWidget *widget) {
  gtk_widget_set_sensitive(widget, TRUE);
  gtk_editable_set_editable(GTK_EDITABLE(widget), TRUE);
}

pos_t *get_pos(appdata_t *appdata) {
  pos_t *pos = &appdata->home;

  if(appdata->active_location) {
    int i = appdata->active_location-1;
    location_t *loc = appdata->location;
    while(i--) loc = loc->next;
    pos = &loc->pos;
  }

  if(appdata->use_gps) {
    pos = gps_get_pos(appdata);

    if(!pos) pos = &appdata->gps;   /* use saved position */
    else     appdata->gps = *pos;   /* save position */
  }
  return pos;
}

void distance_str(char *str, int len, float dist, gboolean imperial) {
  if(isnan(dist))
    snprintf(str, len, "---");
  else if(imperial) {
    /* 1 mil = 1760 yd = 5280 ft ... */
    if(dist<0.018)      snprintf(str, len, "%.1f ft", dist*5280.0);
    else if(dist<0.055) snprintf(str, len, "%.1f yd", dist*1760.0);
    else if(dist<0.55)  snprintf(str, len, "%.0f yd", dist*1760.0);
    else if(dist<10.0)  snprintf(str, len, "%.2f mi", dist);
    else if(dist<100.0) snprintf(str, len, "%.1f mi", dist);
    else                snprintf(str, len, "%.0f mi", dist);
  } else {
    if(dist<0.01)       snprintf(str, len, "%.2f m",  dist*1000.0);
    else if(dist<0.1)   snprintf(str, len, "%.1f m",  dist*1000.0);
    else if(dist<1.0)   snprintf(str, len, "%.0f m",  dist*1000.0);
    else if(dist<100.0) snprintf(str, len, "%.1f km", dist);
    else                snprintf(str, len, "%.0f km", dist);
  }
}

/* return distance in miles or kilometers */
float distance_parse(char *str, gboolean imperial) {
  char unit[4];
  float val = NAN;

  if(sscanf(str, "%f %3s", &val, unit) == 2) {
    gboolean fimp = FALSE;

    if(strcasecmp(unit, "ft") == 0)      { fimp = TRUE;  val /= 5280.0; }
    else if(strcasecmp(unit, "yd") == 0) { fimp = TRUE;  val /= 1760.0; }
    else if(strcasecmp(unit, "mi") == 0) { fimp = TRUE;  }
    else if(strcasecmp(unit, "m")  == 0) { fimp = FALSE; val /= 1000.0; }
    else if(strcasecmp(unit, "km") == 0) { fimp = FALSE; }
    else val = NAN;

    /* found imperial and metric requested? convert miles into kilometers */
    if(fimp & !imperial) val *= 1.609344;

    /* found metric and imperial requested? convert kilometers into miles */
    if(!fimp & imperial) val /= 1.609344;
  }
  return val;
}

static gboolean mark(GtkWidget *widget, gboolean valid) {
  gtk_widget_set_state(widget, valid?GTK_STATE_NORMAL:TAG_STATE);
  return valid;
}

static void callback_modified_lat(GtkWidget *widget, gpointer data ) {
  float i = pos_parse_lat((char*)gtk_entry_get_text(GTK_ENTRY(widget)));
  mark(widget, !isnan(i));
}

/* a entry that is colored red when being "active" */
GtkWidget *lat_entry_new(float lat) {
  GdkColor color;
  GtkWidget *widget = gtk_entry_new();
  gdk_color_parse("#ff0000", &color);
  gtk_widget_modify_text(widget, TAG_STATE, &color);

  char str[32];
  pos_lat_str(str, sizeof(str), lat);
  gtk_entry_set_text(GTK_ENTRY(widget), str);

  g_signal_connect(G_OBJECT(widget), "changed",
		   G_CALLBACK(callback_modified_lat), NULL);
 
  return widget;
}

static void callback_modified_lon(GtkWidget *widget, gpointer data ) {
  float i = pos_parse_lon((char*)gtk_entry_get_text(GTK_ENTRY(widget)));
  mark(widget, !isnan(i));
}

/* a entry that is colored red when filled with invalid coordinate */
GtkWidget *lon_entry_new(float lon) {
  GdkColor color;
  GtkWidget *widget = gtk_entry_new();
  gdk_color_parse("#ff0000", &color);
  gtk_widget_modify_text(widget, TAG_STATE, &color);

  char str[32];
  pos_lon_str(str, sizeof(str), lon);
  gtk_entry_set_text(GTK_ENTRY(widget), str);

  g_signal_connect(G_OBJECT(widget), "changed",
		   G_CALLBACK(callback_modified_lon), NULL);
 
  return widget;
}


float lat_get(GtkWidget *widget) {
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
  return pos_parse_lat(p);
}

float lon_get(GtkWidget *widget) {
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
  return pos_parse_lon(p);
}

static void callback_modified_dist(GtkWidget *widget, gpointer data ) {
  /* don't care for metric/imperial since we only want to know if this */
  /* is parseable at all */
  float i = distance_parse((char*)gtk_entry_get_text(GTK_ENTRY(widget)), FALSE);
  mark(widget, !isnan(i));
}

/* a entry that is colored red when filled with invalid distance */
GtkWidget *dist_entry_new(float dist, gboolean mil) {
  GdkColor color;
  GtkWidget *widget = gtk_entry_new();
  gdk_color_parse("#ff0000", &color);
  gtk_widget_modify_text(widget, TAG_STATE, &color);

  char str[32];
  distance_str(str, sizeof(str), dist, mil);
  gtk_entry_set_text(GTK_ENTRY(widget), str);

  g_signal_connect(G_OBJECT(widget), "changed",
		   G_CALLBACK(callback_modified_dist), NULL);
 
  return widget;
}

float dist_get(GtkWidget *widget, gboolean mil) {
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
  return distance_parse(p, mil);
}

#ifndef USE_MAEMO
#ifdef ENABLE_BROWSER_INTERFACE
#include <libgnome/gnome-url.h>

int browser_url(appdata_t *appdata, char *url) {
  /* taken from gnome-open, part of libgnome */
  GError *err = NULL;
  gnome_url_show(url, &err);
  return 0;
}
#endif
#endif

/* recursively remove an entire file system */
void rmdir_recursive(char *path) {
  GDir *dir = g_dir_open(path, 0, NULL);
  if(dir) {
    const char *name = g_dir_read_name(dir);
    while(name) {
      char *fullname = g_strdup_printf("%s/%s", path, name);
      //      printf("deleting %s\n", fullname);
      
      if(g_file_test(fullname, G_FILE_TEST_IS_DIR)) 
	rmdir_recursive(fullname);
      else if(g_file_test(fullname, G_FILE_TEST_IS_REGULAR)) 
	g_remove(fullname);

      g_free(fullname);
      name = g_dir_read_name(dir);
    }

    g_dir_close(dir);
  }
  g_rmdir(path);
}

#ifdef ENABLE_BROWSER_INTERFACE
static void on_link_clicked(GtkButton *button, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  char *url = g_object_get_data(G_OBJECT(button), "url"); 
  if(url) browser_url(appdata, url);
}
#endif

/* a button containing a weblink */
GtkWidget *link_button_attrib(appdata_t *appdata, char *str, char *url, 
		       int size, int strikethrough) {

#ifdef ENABLE_BROWSER_INTERFACE
  if(url) {
    GtkWidget *button = gtk_button_attrib(str, size, strikethrough);
    g_object_set_data(G_OBJECT(button), "url", url); 
    gtk_signal_connect(GTK_OBJECT(button), "clicked",
		       (GtkSignalFunc)on_link_clicked, appdata);

    return button;
  }
#endif
  return gtk_label_attrib(str, size, strikethrough);
}

#ifdef ENABLE_BROWSER_INTERFACE
static void on_link_id_clicked(GtkButton *button, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  unsigned int id = (unsigned int)g_object_get_data(G_OBJECT(button), "id"); 
  char *type = g_object_get_data(G_OBJECT(button), "type"); 

  char *url = g_strdup_printf("http://www.geocaching.com/%s?id=%u", 
			      type, id);

  if(url) {
    browser_url(appdata, url);
    g_free(url);
  }
}
#endif

GtkWidget *link_button_by_id(appdata_t *appdata, char *str, 
			     const char *type, int id) {

#ifdef ENABLE_BROWSER_INTERFACE
  if(id) {
    GtkWidget *ref = gtk_button_new_with_label(str);
#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR == 5)
    //    hildon_gtk_widget_set_theme_size(ref, 
    //	       (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif  
    g_object_set_data(G_OBJECT(ref), "id", (gpointer)id);
    g_object_set_data(G_OBJECT(ref), "type", (gpointer)type);
    gtk_signal_connect(GTK_OBJECT(ref), "clicked",
		       GTK_SIGNAL_FUNC(on_link_id_clicked), appdata);
    
    return ref;
  }
#endif
  return gtk_label_new(str);
}


GtkWidget *link_icon_button_by_id(appdata_t *appdata, GtkWidget *icon, 
			     const char *type, int id) {

#ifdef ENABLE_BROWSER_INTERFACE
  if(id) {
    GtkWidget *ref = gtk_button_new();
    gtk_button_set_image(GTK_BUTTON(ref), icon);

#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR == 5)
    //    hildon_gtk_widget_set_theme_size(ref, 
    //	       (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif  
    g_object_set_data(G_OBJECT(ref), "id", (gpointer)id);
    g_object_set_data(G_OBJECT(ref), "type", (gpointer)type);
    gtk_signal_connect(GTK_OBJECT(ref), "clicked",
		       GTK_SIGNAL_FUNC(on_link_id_clicked), appdata);
    
    return ref;
  }
#endif
  return icon;
}

/* left aligned, word wrapped multiline widget */
GtkWidget *simple_text_widget(char *text) {
  GtkWidget *label = gtk_label_new(text);
  
  gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
  gtk_label_set_line_wrap_mode(GTK_LABEL(label), PANGO_WRAP_WORD);
  gtk_misc_set_alignment(GTK_MISC(label), 0.0, 0.5);

  return label;
}
