/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include <math.h>

#define STR_NAN  "----"

typedef struct {
  appdata_t *appdata;

  GtkWidget *lon1, *lat1, *dist1, *dir1;
  GtkWidget *lon2, *lat2;

  GtkWidget *distance, *proj_lat, *proj_lon;
  GtkWidget *middle_lat, *middle_lon;
} math_dialog_state_t;

static gboolean mark(GtkWidget *widget, gboolean valid) {
  gtk_widget_set_state(widget, valid?GTK_STATE_NORMAL:TAG_STATE);
  return valid;
}

/* a entry that is colored red when being "active" */
static GtkWidget *gtk_red_entry_new(void) {
  GdkColor color;
  GtkWidget *widget = gtk_entry_new();
  gdk_color_parse("#ff0000", &color);
  gtk_widget_modify_text(widget, TAG_STATE, &color);
  return widget;
}

/* a comboboxentry that is colored red when being "active" */
static GtkWidget *gtk_red_combo_box_entry_new_text(void) {
  GdkColor color;
  GtkWidget *widget = gtk_combo_box_entry_new_text();
  gdk_color_parse("#ff0000", &color);
  gtk_widget_modify_text(GTK_BIN(widget)->child, TAG_STATE, &color);
  return widget;
}

/* a label that is left aligned */
static GtkWidget *gtk_left_label_new(char *str) {
  GtkWidget *widget = gtk_label_new(str);
  gtk_misc_set_alignment(GTK_MISC(widget), 0.0f, 0.5f);
  return widget;
}

/* a label that is colored red when being "active" */
static GtkWidget *gtk_red_label_new(char *str) {
  GdkColor color;
  GtkWidget *widget = gtk_left_label_new(str);
  gdk_color_parse("#ff0000", &color);
  gtk_widget_modify_fg(widget, TAG_STATE, &color);
  return widget;
}

static GtkWidget *gtk_dir_entry_new(float val) {
  char str[16];
  GtkWidget *w = gtk_red_entry_new();
  
  snprintf(str, sizeof(str), _("%.1f°"), val);
  gtk_entry_set_text(GTK_ENTRY(w), str);

  return w;
}

float pos_lat_eval_combo(GtkWidget *widget) {
  char *p = (char*)gtk_combo_box_get_active_text(GTK_COMBO_BOX(widget));
  float val = pos_parse_lat(p);
  mark(widget, !isnan(val));

  return val;
}

static float distance_eval(GtkWidget *widget, math_dialog_state_t *state) {
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
  float val = distance_parse(p, state->appdata->imperial);
  mark(widget, !isnan(val));

  return val;
}

float direction_eval(GtkWidget *widget) {
  char *p = (char*)gtk_entry_get_text(GTK_ENTRY(widget));
  float val;
  if(sscanf(p, _("%f°"), &val) != 1)
    val = NAN;

  mark(widget, !isnan(val));

  return val;
}

static void on_calc_clicked(GtkButton *button, gpointer user_data) {
  math_dialog_state_t *state = (math_dialog_state_t*)user_data;
  pos_t pos1, pos2;
  gboolean pos1_ok = FALSE, pos2_ok = FALSE;
  float dist1;
  gboolean dist1_ok = FALSE;
  float dir1;
  gboolean dir1_ok = FALSE;

  /* parse input */
  pos1.lat = lat_get(state->lat1);
  pos1.lon = lon_get(state->lon1);
  if(!isnan(pos1.lat) && !isnan(pos1.lon)) pos1_ok = TRUE;

  pos2.lat = lat_get(state->lat2);
  pos2.lon = lon_get(state->lon2);
  if(!isnan(pos2.lat) && !isnan(pos2.lon)) pos2_ok = TRUE;

  dist1 = distance_eval(state->dist1, state);
  if(!isnan(dist1)) dist1_ok = TRUE;

  dir1 = direction_eval(state->dir1);
  if(!isnan(dir1)) dir1_ok = TRUE;

  /* ------------------- do all calculations ------------------- */


  /* ------------------- distance of coo1 and coo2  ------------------- */
  if(mark(state->distance, pos1_ok && pos2_ok)) {
    char str[32];
    float dist = gpx_pos_get_distance(pos1, pos2, state->appdata->imperial);  
    distance_str(str, sizeof(str), dist, state->appdata->imperial);

    gtk_label_set_text(GTK_LABEL(state->distance), str);
  } else
    gtk_label_set_text(GTK_LABEL(state->distance), STR_NAN);

  // N 53° 09.033'  W 001° 50.666' 100km / 30° = N 53° 55.616, W001° 04.850
  /* ------------------- coordinate projection ---------------- */
  mark(state->proj_lat, pos1_ok && dist1_ok && dir1_ok);
  if(mark(state->proj_lon, pos1_ok && dist1_ok && dir1_ok)) {
    pos_t pro;

    /* get great circle radius in miles/kilometers */
    float gcrad = state->appdata->imperial?3959.0:6371.0;

    // from: http://www.movable-type.co.uk/scripts/latlong.html
    pro.lat = asin(sin(pos1.lat/180*M_PI) * cos(dist1/gcrad) + 
		   cos(pos1.lat/180*M_PI) * sin(dist1/gcrad) * 
		   cos(dir1/180*M_PI) )/M_PI*180;
    pro.lon = pos1.lon + atan2(sin(dir1/180*M_PI)*sin(dist1/gcrad)*
			       cos(pos1.lat/180*M_PI), 
			       cos(dist1/gcrad)-sin(pos1.lat/180*M_PI)*
			       sin(pro.lat/180*M_PI))/M_PI*180;
    pro.lon = fmodf(pro.lon+180,360) - 180;  // normalise to -180...+180

    char str[16];
    pos_lat_str(str, sizeof(str), pro.lat);
    gtk_label_set_text(GTK_LABEL(state->proj_lat), str);
    pos_lon_str(str, sizeof(str), pro.lon);
    gtk_label_set_text(GTK_LABEL(state->proj_lon), str);

    if(!isnan(pro.lat) && !isnan(pro.lon)) 
      state->appdata->geomath = pro;
  } else {
    gtk_label_set_text(GTK_LABEL(state->proj_lat), STR_NAN);
    gtk_label_set_text(GTK_LABEL(state->proj_lon), STR_NAN);
  }

#if 0
  /* ------------ middle between both points ------------- */
  mark(state->middle_lat, pos1_ok && pos2_ok);
  if(mark(state->middle_lon, pos1_ok && pos2_ok)) {
    pos_t res;

    float dlon = fabs(pos2.lon - pos1.lon)/180*M_PI;

    /* http://mathforum.org/library/drmath/view/51822.html */
    res.lon = 180/M_PI * atan(cos(pos2.lat/180*M_PI)*sin(pos2.lon/180*M_PI)/
     (cos(pos1.lat/180*M_PI)+cos(pos2.lat/180*M_PI)*cos(pos2.lon/180*M_PI)));
    res.lat = 180/M_PI * atan((sin(pos1.lat/180*M_PI)+sin(pos2.lat/180*M_PI))/
        sqrt(pow(cos(pos2.lat/180*M_PI)+cos(pos2.lat/180*M_PI)*cos(dlon),2)+
        pow(cos(pos2.lat/180*M_PI)*sin(dlon),2)));

    char str1[32], str2[32];
    pos_lat_str(str1,sizeof(str1),res.lat);
    pos_lon_str(str2,sizeof(str2),res.lon);
    printf("pos = %s/%s\n", str1, str2);
  } else {
    gtk_label_set_text(GTK_LABEL(state->middle_lat), STR_NAN);
    gtk_label_set_text(GTK_LABEL(state->middle_lon), STR_NAN);
  }
#endif
}

static gint waypoint_changed_event(GtkWidget *widget, gpointer data ) {
  math_dialog_state_t *state = (math_dialog_state_t*)data;
  int wpt_idx = gtk_combo_box_get_active(GTK_COMBO_BOX(widget));
  pos_t *pos = NULL;

  if(wpt_idx < 0)
    return FALSE;

  if(wpt_idx == 0) 
    pos = gps_get_pos(state->appdata);
#ifdef USE_MAEMO
  else if(wpt_idx == 1) {
    pos_t cache_pos = gpx_cache_pos(state->appdata->cur_cache);
    pos = &cache_pos;
  } else {
    wpt_t *wpt = state->appdata->cur_cache->wpt;
    while(wpt_idx > 2) {
      g_assert(wpt != NULL);
      wpt = wpt->next;
      wpt_idx--;
    }
    pos = &wpt->pos;
  }
#endif

  if(pos) {
    char str[32];
    pos_lat_str(str,sizeof(str),pos->lat);
    gtk_entry_set_text(GTK_ENTRY(state->lat1), str);
    pos_lon_str(str,sizeof(str),pos->lon);
    gtk_entry_set_text(GTK_ENTRY(state->lon1), str);
  } else {
    gtk_entry_set_text(GTK_ENTRY(state->lat1), STR_NAN);
    gtk_entry_set_text(GTK_ENTRY(state->lon1), STR_NAN);
  }

  mark(state->lat1, pos != NULL);
  mark(state->lon1, pos != NULL);

  return FALSE;
}

void geomath_dialog(appdata_t *appdata) {
  static pos_t pos1 = { 0.0, 0.0 }, pos2 = { 0.0, 0.0 };
  static float dist1 = 0.0;
  static float dir1 = 0.0;
  static gboolean is_imperial = FALSE;

  math_dialog_state_t state;
  char str[32];

  /* this is quite ugly. It would be nice to run the entire system on */
  /* one specific system (e.g. metric) and only convert for in- and output */
  if(!appdata->imperial && is_imperial)
    dist1 *= 6371.0/3959.0;   /* we just switched to metric */
  if(appdata->imperial && !is_imperial)
    dist1 *= 3959.0/6371.0;   /* we just switched to imperial */
  is_imperial = appdata->imperial;

  state.appdata = appdata;

#ifdef USE_MAEMO
  if(appdata->cur_cache) 
    printf("current cache is %s\n", appdata->cur_cache->id);
  else
    printf("no current cache\n");
#endif

  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("Geomath"),
                          GTK_WINDOW(appdata->window),
			  //      GTK_DIALOG_NO_SEPARATOR | 
			  GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                          GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
                          NULL);

#if defined(USE_MAEMO) && defined(HILDON_HELP)
  hildon_help_dialog_help_enable(GTK_DIALOG(dialog), HELP_ID_GEOMATH, 
				 appdata->osso_context);
#endif

  gtk_window_set_default_size(GTK_WINDOW(dialog), DIALOG_WIDTH, DIALOG_HEIGHT);

  /* do geomath dialog */
  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);

  /* ------------------------- input area ------------------------- */
  GtkWidget *table = gtk_table_new(5, 3, FALSE);
  gtk_table_set_col_spacing(GTK_TABLE(table), 1, 20);

  gtk_table_attach_defaults(GTK_TABLE(table), gtk_label_new(_("Coordinate 1")),                    1, 2, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), gtk_label_new(_("Coordinate 2")),                    2, 3, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), gtk_left_label_new(_("Latitude:")),                  0, 1, 1, 2);

  GtkWidget *cbox = gtk_red_combo_box_entry_new_text();
  gtk_combo_box_append_text(GTK_COMBO_BOX(cbox), _("GPS"));
  
#ifdef USE_MAEMO
  if(appdata->cur_cache) {
    gtk_combo_box_append_text(GTK_COMBO_BOX(cbox), appdata->cur_cache->id);
    wpt_t *wpt = appdata->cur_cache->wpt;
    while(wpt) {
      gtk_combo_box_append_text(GTK_COMBO_BOX(cbox), wpt->id);
      wpt = wpt->next;
    }
  }
#endif
  
  state.lat1 = GTK_BIN(cbox)->child;
  pos_lat_str(str, sizeof(str), pos1.lat);
  gtk_entry_set_text(GTK_ENTRY(state.lat1), str);
  
  gtk_signal_connect(GTK_OBJECT(cbox), "changed",
		     (GtkSignalFunc)waypoint_changed_event, &state);
  gtk_table_attach_defaults(GTK_TABLE(table), cbox, 1,2,1,2);


  gtk_table_attach_defaults(GTK_TABLE(table), state.lat2 = lat_entry_new(pos2.lat),         2, 3, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), gtk_left_label_new(_("Longitude:")),             0, 1, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), state.lon1 = lon_entry_new(pos1.lon),         1, 2, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), state.lon2 = lon_entry_new(pos2.lon),         2, 3, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), gtk_left_label_new(_("Distance:")),              0, 1, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(table), state.dist1 = dist_entry_new(dist1, appdata->imperial), 1, 2, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(table), gtk_left_label_new(_("Direction:")),             0, 1, 4, 5);
  gtk_table_attach_defaults(GTK_TABLE(table), state.dir1 = gtk_dir_entry_new(dir1),         1, 2, 4, 5);

  gtk_box_pack_start(GTK_BOX(hbox), table, TRUE, TRUE, 0);

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, TRUE, 0);

  /* ------------------------- do-it-button ------------------------- */

  hbox = gtk_hbox_new(FALSE, 0);
  GtkWidget *button = gtk_button_new_with_label(_("Calculate!"));
#if defined(USE_MAEMO) && (MAEMO_VERSION_MAJOR == 5)
  hildon_gtk_widget_set_theme_size(button, 
		   (HILDON_SIZE_FINGER_HEIGHT | HILDON_SIZE_AUTO_WIDTH));
#endif  
  gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox, TRUE, FALSE, 0);
  g_signal_connect(button, "clicked", (GCallback)on_calc_clicked, &state);

  /* ------------------------- output area ------------------------- */
  
  table = gtk_table_new(3, 3, FALSE);

  gtk_table_attach_defaults(GTK_TABLE(table), gtk_left_label_new(_("Distance = ")),
			    0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), state.distance = gtk_red_label_new(STR_NAN),
			    1, 3, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), gtk_left_label_new(_("Projection = ")),
			    0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), state.proj_lat = gtk_red_label_new(STR_NAN),
			    1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), state.proj_lon = gtk_red_label_new(STR_NAN),
			    2, 3, 1, 2);
#if 0
  gtk_table_attach_defaults(GTK_TABLE(table), gtk_left_label_new(_("Middle = ")),
			    0, 1, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), state.middle_lat = gtk_red_label_new(STR_NAN),
			    1, 2, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), state.middle_lon = gtk_red_label_new(STR_NAN),
			    2, 3, 2, 3);
#endif

  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), table, TRUE, TRUE, 0);
  

  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_CLOSE);

  gtk_widget_show_all(dialog);
  gtk_dialog_run(GTK_DIALOG(dialog));

  /* copy values back to local static variables so they re-appear if */
  /* the dialog is re-opened, convert illegal values (NAN) to 0 */

  pos1.lat = lat_get(state.lat1); if(isnan(pos1.lat)) pos1.lat=0; 
  pos1.lon = lon_get(state.lon1); if(isnan(pos1.lon)) pos1.lon=0; 
  pos2.lat = lat_get(state.lat2); if(isnan(pos2.lat)) pos2.lat=0; 
  pos2.lon = lon_get(state.lon2); if(isnan(pos2.lon)) pos2.lon=0; 
  dist1 = distance_eval(state.dist1, &state); if(isnan(dist1)) dist1=0; 
  dir1 = direction_eval(state.dir1); if(isnan(dir1)) dir1=0; 

  gtk_widget_destroy(dialog);
}
