/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"
#include <math.h>    // for isnan

#ifdef ENABLE_OSM_GPS_MAP
#include "osm-gps-map.h"
#endif

typedef struct {
  appdata_t *appdata;
  GtkWidget *widget;
  GtkWidget *zoomin, *zoomout, *gps;
  gint handler_id;
} map_context_t;

#define PROXY_KEY  "/system/http_proxy/"

static const char *get_proxy_uri(appdata_t *appdata) {
  static char proxy_buffer[64] = "";
  
  /* use environment settings if preset */
  const char *proxy = g_getenv("http_proxy");
  if(proxy) {
    printf("http_proxy: %s\n", proxy);
    return proxy;
  }

  /* ------------- get proxy settings -------------------- */
  if(gconf_client_get_bool(appdata->gconf_client, 
			   PROXY_KEY "use_http_proxy", NULL)) {

    /* we can savely ignore things like "ignore_hosts" since we */
    /* are pretty sure not inside the net of one of our map renderers */
    /* (unless the user works at google :-) */
      
    /* get basic settings */
    char *host = 
      gconf_client_get_string(appdata->gconf_client, PROXY_KEY "host", NULL);
    if(host) {
      int port =
	gconf_client_get_int(appdata->gconf_client, PROXY_KEY "port", NULL);

      snprintf(proxy_buffer, sizeof(proxy_buffer),
	       "http://%s:%u", host, port);

      g_free(host);
    }
    return proxy_buffer;
  }

  return NULL;
}

static void map_zoom(map_context_t *context, int step) {
  int zoom;
  OsmGpsMap *map = OSM_GPS_MAP(context->widget);
  g_object_get(map, "zoom", &zoom, NULL);
  zoom = osm_gps_map_set_zoom(map, zoom+step);

  /* enable/disable zoom buttons as required */
  gtk_widget_set_sensitive(context->zoomin, zoom<17);
  gtk_widget_set_sensitive(context->zoomout, zoom>1);
}

static gboolean
cb_map_zoomin(GtkButton *button, map_context_t *context) {
  map_zoom(context, +1);
  return FALSE;
}

static gboolean
cb_map_zoomout(GtkButton *button, map_context_t *context) {
  map_zoom(context, -1);
  return FALSE;
}

static gboolean
cb_map_gps(GtkButton *button, map_context_t *context) {
  pos_t *refpos = get_pos(context->appdata);
  if(refpos && !isnan(refpos->lat) && !isnan(refpos->lon)) {
    osm_gps_map_set_mapcenter(OSM_GPS_MAP(context->widget),
			      refpos->lat, refpos->lon, 14);     
  } else {
    /* no coordinates given: display the entire world */
    osm_gps_map_set_mapcenter(OSM_GPS_MAP(context->widget), 
			      0.0, 0.0, 1);
  }

  return FALSE;
}

static GtkWidget 
*map_add_button(const gchar *icon, GCallback cb, gpointer data, 
		char *tooltip) {
  GtkWidget *button = gtk_button_new();
  gtk_button_set_image(GTK_BUTTON(button), 
		       gtk_image_new_from_stock(icon, GTK_ICON_SIZE_MENU));
  g_signal_connect(button, "clicked", cb, data);
#ifndef USE_MAEMO
  gtk_widget_set_tooltip_text(button, tooltip);
#endif
  return button;
}

static gboolean map_gps_update(gpointer data) {
  map_context_t *context = (map_context_t*)data;

  pos_t *refpos = get_pos(context->appdata);
  gboolean ok = (refpos!= NULL) && !isnan(refpos->lat) && !isnan(refpos->lon);

  /* get reference position and go there */
  gtk_widget_set_sensitive(context->gps, ok);

  return TRUE;
}

static gboolean on_map_configure(GtkWidget *widget,
				 GdkEventConfigure *event,
				 map_context_t *context) {

  cb_map_gps(NULL, context);

  return FALSE;
}

void map(appdata_t *appdata) {
  map_context_t context;
  context.appdata = appdata;

  GtkWidget *dialog = gtk_dialog_new_with_buttons(_("Map"),
			  GTK_WINDOW(appdata->window),
			  GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
                          GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
                          NULL);

#ifndef USE_MAEMO
  gtk_window_set_default_size(GTK_WINDOW(dialog), 400, 350);
#else
  gtk_window_set_default_size(GTK_WINDOW(dialog), 800, 480);
#endif

  GtkWidget *hbox = gtk_hbox_new(FALSE, 0);

  char *path = g_strdup_printf("%s/map/", appdata->image_path);
  const char *proxy = get_proxy_uri(appdata);

  context.widget = g_object_new(OSM_TYPE_GPS_MAP,
                "repo-uri", MAP_SOURCE_OPENSTREETMAP,
                "tile-cache", path,
		proxy?"proxy-uri":NULL, proxy,
                 NULL);

  g_free(path);

  g_signal_connect(G_OBJECT(context.widget), "configure-event",
		   G_CALLBACK(on_map_configure), &context);
#if 0
  g_signal_connect(G_OBJECT(context.widget), "button-release-event",
                   G_CALLBACK(on_map_button_release_event), &context);
#endif

  gtk_box_pack_start_defaults(GTK_BOX(hbox), context.widget);
  /* zoom button box */
  GtkWidget *vbox = gtk_vbox_new(FALSE,0);

  context.zoomin = 
    map_add_button(GTK_STOCK_ZOOM_IN, G_CALLBACK(cb_map_zoomin),
                   &context, _("Zoom in"));
  gtk_box_pack_start(GTK_BOX(vbox), context.zoomin, FALSE, FALSE, 0);

  context.zoomout = 
    map_add_button(GTK_STOCK_ZOOM_OUT, G_CALLBACK(cb_map_zoomout),
                   &context, _("Zoom out"));
  gtk_box_pack_start(GTK_BOX(vbox), context.zoomout, FALSE, FALSE, 0);

  context.gps = 
    map_add_button(GTK_STOCK_HOME, G_CALLBACK(cb_map_gps),
		   &context, _("Jump to GPS position"));
  gtk_widget_set_sensitive(context.gps, FALSE);
  /* install handler for timed updates of the gps button */
  context.handler_id = gtk_timeout_add(1000, map_gps_update, &context);
  gtk_box_pack_start(GTK_BOX(vbox), context.gps, FALSE, FALSE, 0);

  gtk_box_pack_start(GTK_BOX(hbox), vbox, FALSE, FALSE, 0);

  gtk_box_pack_start_defaults(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox);

  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_CLOSE);

  gtk_widget_show_all(dialog);

  gtk_dialog_run(GTK_DIALOG(dialog));
  
  gtk_timeout_remove(context.handler_id);

  gtk_widget_destroy(dialog);
}
