/*
 * Copyright (C) 2008 Till Harbaum <till@harbaum.org>.
 *
 * This file is part of GPXView.
 *
 * GPXView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * GPXView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with GPXView.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "gpxview.h"

typedef struct load_context {
  int active;
  GMutex *mutex;
  GtkWidget *view;
  char *url, *path;
  GtkHTMLStream *stream;
  struct load_context *next;
} load_context_t;

typedef struct {
  appdata_t *appdata;
  cache_t *cache;
  GtkWidget *view;
  load_context_t *load_context;
} http_context_t;

static unsigned long name_hash(const char *name) {
  unsigned long val = 0;

  while(*name) {
    val = (val<<8) ^ (val >> 24) ^ (*name & 0xff);
    name++;
  }

  return val;
}

void release_load_context(GThread *self, load_context_t *context) {
  /* this must be atomar, so acquire a lock */

  printf("%p: freeing context at %p\n", self, context);

  g_mutex_lock(context->mutex);

  /* don't do much if the other thread still uses this */
  if(context->active) {
    printf("  still active -> just close link\n");

    /* close link to view as the thread is either done or */
    /* main wants to close the view */
    gtk_html_end(GTK_HTML(context->view), context->stream, 
		 GTK_HTML_STREAM_OK);

    context->active = FALSE;
    g_mutex_unlock(context->mutex);
    return;
  }

  /* ok, other thread is also gone -> we can destroy everything */
  printf("  not active -> destroy\n");

  /* just free everything */
  g_mutex_unlock(context->mutex);
  free(context->url);
  free(context->path);
  free(context);
}

gpointer loader_thread(gpointer data) {
  GThread *self = g_thread_self();

  GnomeVFSResult result;
  GnomeVFSHandle *handle;
  char buffer[4096];
  GnomeVFSFileSize bytes_read;

  load_context_t *context = (load_context_t*)data;

  printf("%p: loader thread for %s running\n", self, context->url);

  result = gnome_vfs_open(&handle, context->url, GNOME_VFS_OPEN_READ);
  if(result != GNOME_VFS_OK) {
    g_print("%p: open error: %s\n", self, gnome_vfs_result_to_string(result));

    release_load_context(self, context);
    return NULL;
  }
  
  /* try to open file for writing */
  FILE *f = fopen(context->path, "wb");
  int running = TRUE;
  
  do {
    result = gnome_vfs_read(handle, buffer, sizeof(buffer)-1, &bytes_read);
    if((result == GNOME_VFS_OK) && (bytes_read > 0)) {

      /* update local "running" variable from shared variable */
      if(running) {
	g_mutex_lock(context->mutex);
	running = context->active;
	g_mutex_unlock(context->mutex);
      }

      if(running) 
	gtk_html_write(GTK_HTML(context->view), 
		       context->stream, buffer, bytes_read);
      
      /* and also write local file */
      if(f) fwrite(buffer, 1l, bytes_read, f);
    }
  } while((result == GNOME_VFS_OK) && (bytes_read > 0) && running);
  
  if(f) fclose(f);

  gnome_vfs_close(handle);

  /* this file is likely incomplete, remove it */
  if(!running) {
    printf("%p: thread killed, removing imcomplete cached image\n", self);
    remove(context->path);
    release_load_context(self, context);
    return NULL;
  }

  printf("%p: loader thread successfully finished\n", self);
  release_load_context(self, context);
  return NULL;
}

static void on_request_url(GtkHTML *html, const gchar *url, 
			   GtkHTMLStream *stream, gpointer data) {
  char buffer[4096];
  GnomeVFSFileSize bytes_read;
  
  http_context_t *context = (http_context_t*)data;

  if(context->cache) {
    /* try to build local path */
    char *path = malloc(strlen(context->appdata->image_path)+
			strlen(context->cache->id)+
			14);  /* strlen("/xxxxxxxx.xxx\0") == 14 */

    strcpy(path, context->appdata->image_path);
    strcat(path, context->cache->id);
    sprintf(path+strlen(path), "/%lx", name_hash(url));

    /* only append extension if there's a useful one up to three chars ... */
    char *dot = strrchr(url, '.');
    if(dot) 
      if(strlen(dot) <= 4)
	strcat(path, dot);
    
    printf("cache name = %s\n", path);
    if(g_file_test(path, G_FILE_TEST_EXISTS)) {
      printf("image file exists!\n");
      
      FILE *f = fopen(path, "rb");
      
      while ((bytes_read = fread(buffer, 1, sizeof(buffer), f)) != 0) {

	gtk_html_write(GTK_HTML(context->view), 
		       stream, buffer, bytes_read);

	while (gtk_events_pending ())
	  gtk_main_iteration ();
      }
      fclose(f);
      
    } else {
      if(context->appdata->load_images) {
	printf("image file doesn't exist, starting extra thread!\n");
	
	checkdir(path); 

	/* walk to end of list */
	load_context_t **load_context = &(context->load_context);
	while(*load_context) 
	  load_context = &(*load_context)->next;

	*load_context = g_new0(load_context_t, 1);

	(*load_context)->url = strdup(url);
	(*load_context)->path = strdup(path);
	(*load_context)->view = context->view;
	(*load_context)->stream = stream;
	(*load_context)->next = NULL;
	(*load_context)->active = TRUE;
	(*load_context)->mutex = g_mutex_new();
	
	g_thread_create(loader_thread, *load_context, TRUE, NULL);
	return;
      } else
	g_print("Image loading disabled\n");
    }
  } else {
    /* not a cache, maybe help, so load images from icon directory */

    /* try to build local path */
    char *path = malloc(strlen(ICONPATH)+strlen(url)+1);

    strcpy(path, ICONPATH);
    strcat(path, url);
    if(!g_file_test(path, G_FILE_TEST_EXISTS)) {
      strcpy(path, "./icons/");
      strcat(path, url);
    }

    if(g_file_test(path, G_FILE_TEST_EXISTS)) {
      FILE *f = fopen(path, "rb");
      
      while ((bytes_read = fread(buffer, 1, sizeof(buffer), f)) != 0) {
	gtk_html_write(GTK_HTML(context->view), 
		       stream, buffer, bytes_read);
	
	while (gtk_events_pending ())
	  gtk_main_iteration ();
      }
      fclose(f);
    }
  }

  gtk_html_end(GTK_HTML(context->view), stream, GTK_HTML_STREAM_OK);
}

/* notify all open html views of zoom event */
void html_zoom(appdata_t *appdata, gboolean in) {
  struct html_view *html_view = appdata->html_view;

  while(html_view) {
    printf("zoom notify view %p\n", html_view->view);

    if(in) gtk_html_zoom_in(GTK_HTML(html_view->view));
    else   gtk_html_zoom_out(GTK_HTML(html_view->view));

    html_view = html_view->next;
  }
}

static void on_destroy_textview(GtkWidget *widget, gpointer data) {
  appdata_t *appdata = (appdata_t*)data;
  int destroy_active = FALSE;

  /* only do this if main windows hasn't already been destroyed */
  if(!appdata->window) {
    printf("detroy_textview: main window is gone\n");
    return;
  }

  if(!appdata->active_buffer)
    printf("Destroy, but there was no active buffer!\n");
  else {
    if(GTK_WIDGET_TYPE(widget) == GTK_TYPE_TEXT_VIEW) {
      printf("destroying textview\n");
      
      if(appdata->active_buffer == 
	 gtk_text_view_get_buffer(GTK_TEXT_VIEW(widget))) {
	printf("This was the active buffer\n");
	destroy_active = TRUE;
      }
    } else {
      if(widget == (GtkWidget*)appdata->active_buffer) {
	printf("This was the active html view\n");
	destroy_active = TRUE;
      }
    }
  }

  if(destroy_active) {
    appdata->active_buffer = NULL;

    gtk_widget_set_sensitive(appdata->menu_cut, FALSE);
    gtk_widget_set_sensitive(appdata->menu_copy, FALSE);
    gtk_widget_set_sensitive(appdata->menu_paste, FALSE);
  }
}

static void on_destroy_htmlview(GtkWidget *widget, gpointer data) {
  http_context_t *context = (http_context_t*)data;

  printf("destroying html context\n");
  //  printf("associated view = %p\n", context->view);

  struct html_view **h = &(context->appdata->html_view);

  while(*h && (*h)->view != context->view) 
    h = &((*h)->next);

  g_assert(h);

  /* remove entry from chain */
  void *tmp = *h;
  *h = (*h)->next;
  free(tmp);

  load_context_t *load_context = context->load_context;
  while(load_context) {
    printf("found an associated thread\n");

    load_context_t *tmp_context = load_context->next;
    release_load_context(NULL, load_context);
    load_context = tmp_context;
  }

  on_destroy_textview(widget, context->appdata);

  /* destroy context */
  free(data);
}

static gboolean focus_in(GtkWidget *widget, GdkEventFocus *event,
			 gpointer data) {
  appdata_t *appdata = (appdata_t*)data;

  printf("focus in!\n");
 
  /* these buffers are read-only, thus only "copy" is enabled */
  gtk_widget_set_sensitive(appdata->menu_cut, FALSE);
  gtk_widget_set_sensitive(appdata->menu_copy, TRUE);
  gtk_widget_set_sensitive(appdata->menu_paste, FALSE);

  if(GTK_WIDGET_TYPE(widget) == GTK_TYPE_TEXT_VIEW) {
    appdata->active_buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(widget));
  } else if(GTK_WIDGET_TYPE(widget) == gtk_html_get_type()) {
    appdata->active_buffer = (GtkTextBuffer*)widget;
  } else {
    printf("not a text nor a html view\n");
  }

  return FALSE;
}

void html_copy_to_clipboard(appdata_t *appdata) {
  gtk_html_copy(GTK_HTML(appdata->active_buffer));
}

/* panning a gtkhtml view currently doesn't work well */
#undef PANNABLE_HTML

#ifdef PANNABLE_HTML
/* eat the button events */
static gboolean on_button_press(GtkWidget *widget, GdkEventButton *event,
				gpointer user_data) {
  return TRUE;
}
#endif

/* the cache descriptions are not valid html and need some surrounding stuff */
static const char *html_start = "<html><head>"
  "<meta http-equiv=content-type content=\"text/html; charset=UTF-8\">"
  "</head></body>";
static const char *html_end = "</body></html>";

GtkWidget *html_view(appdata_t *appdata, char *text, 
		     gboolean is_html, gboolean scrollwin,
		     cache_t *cache, char *anchor) {
  GtkWidget *view;

  if(is_html) {
    http_context_t *context = g_new0(http_context_t, 1);
    context->appdata = appdata;
    context->cache = cache;

    context->view = view = gtk_html_new();

    /* create a callback to load images only if a cache has been given */
    /* so that images can be cached/stored appropriately */
    g_signal_connect(G_OBJECT(view), "url_requested", 
		     G_CALLBACK(on_request_url), context);

    GtkHTMLStream *stream = gtk_html_begin(GTK_HTML(view));

    gtk_html_write(GTK_HTML(view), stream, html_start, strlen(html_start));
    gtk_html_write(GTK_HTML(view), stream, text, strlen(text));
    gtk_html_write(GTK_HTML(view), stream, html_end, strlen(html_end));
    gtk_html_end(GTK_HTML(view), stream, GTK_HTML_STREAM_OK);

    if(anchor) {
      while(gtk_events_pending())
	gtk_main_iteration ();

      gtk_html_jump_to_anchor(GTK_HTML(context->view), anchor);
    }

    /* register this html view */
    struct html_view **h = &(appdata->html_view);
    while(*h) h = &((*h)->next);
    *h = g_new0(struct html_view, 1);
    (*h)->view = view;

#ifdef PANNABLE_HTML
    /* this causes finger scrolling to work nicely but also prevents */
    /* copy'n paste from working correctly */
    gtk_widget_set_sensitive(GTK_WIDGET(view), FALSE);

    g_signal_connect(G_OBJECT(view), "button-press-event", 
		     G_CALLBACK(on_button_press), NULL);
#endif

    g_signal_connect(G_OBJECT(view), "destroy", 
		     G_CALLBACK(on_destroy_htmlview), context);
  } else {
    GtkTextBuffer *buffer = gtk_text_buffer_new(NULL);
    gtk_text_buffer_set_text(buffer, text, strlen(text));

#ifndef USE_MAEMO    
    view = gtk_text_view_new_with_buffer(buffer);
#else
    view = hildon_text_view_new();
    hildon_text_view_set_buffer(HILDON_TEXT_VIEW(view), buffer);
#endif

    gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(view), GTK_WRAP_WORD);
    gtk_text_view_set_editable(GTK_TEXT_VIEW(view), FALSE);
    gtk_text_view_set_cursor_visible(GTK_TEXT_VIEW(view), FALSE);

    g_signal_connect(G_OBJECT(view), "destroy", 
		     G_CALLBACK(on_destroy_textview), appdata);
  } 
  
  g_signal_connect(G_OBJECT(view), "focus-in-event", 
		   G_CALLBACK(focus_in), appdata);

  if(scrollwin) {
#ifndef USE_PANNABLE_AREA
    GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (scrolled_window), 
		     GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    gtk_container_add(GTK_CONTAINER(scrolled_window), view);

#if 1
    return scrolled_window;
#else
    GtkWidget *fixed = gtk_fixed_new();
    gtk_fixed_put(GTK_FIXED(fixed), scrolled_window, 0, 0);
    GtkWidget *tbutton = gtk_toggle_button_new_with_label("sel");
    gtk_fixed_put(GTK_FIXED(fixed), tbutton, 0, 0);
    return fixed;
#endif
#else
#ifndef PANNABLE_HTML
  if(is_html) { 
      GtkWidget *scrolled_window = gtk_scrolled_window_new(NULL, NULL);
      gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW (scrolled_window), 
		     GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
      gtk_container_add(GTK_CONTAINER(scrolled_window), view);
      return scrolled_window;
    } else
#endif
    {
      GtkWidget *pannable_area = hildon_pannable_area_new();
      gtk_container_add(GTK_CONTAINER(pannable_area), view);
      return pannable_area;    
    }
#endif
  }

  return view;
}

