#include "keysconfig.h"

const char* KeysConfig::name[] = {
    "Up",
    "Down",
    "Left",
    "Right",
    "Start",
    "Select",
    "A",
    "B",
    "L",
    "R",
    "LeftUp",
    "RightUp",
    "LeftDown",
    "RightDown",
    "LoadState",
    "SaveState",
    "FastForward",
    "ShowFPS",
    "Quit",
    "Pause"
};

const char* KeysConfig::displayName[] = {
    "Up",
    "Down",
    "Left",
    "Right",
    "Start",
    "Select",
    "A",
    "B",
    "L",
    "R",
    "Left + Up",
    "Right + Up",
    "Left + Down",
    "Right + Down",
    "Load State",
    "Save State",
    "Fast Forward",
    "Show FPS",
    "Quit",
    "Pause"
};

// Construct and bind to a file
KeysConfig::KeysConfig(QString file) : AbstractConfig(file)
{
    // Fill with default values
    for (int i = 0; i < KEY_COUNT; i++)
        code[i] = 0;

    // Load from the bound file
    if (!load(file))
        // Fall back to the global configuration
        if (!load(Path::cfgFile()))
            // Fall back to gpspm's user configuration
            if (!read(Path::emuDir() + "keys"))
                // Fall back to gpspm's stock configuration
                read(Path::emuRoot() + "keys");
}

// Load from a native file
bool KeysConfig::load(QString file)
{
    QSettings settings(file, QSettings::IniFormat);

    // Abort if the contents are not satisfactory
    if (!exists(settings)) return false;

    settings.beginGroup(group());

    for (int i = 0; i < KEY_COUNT; i++)
        code[i] = settings.value(name[i]).toInt();

    return true;
}

// Save to the bound file
void KeysConfig::save()
{
    QSettings settings(file, QSettings::IniFormat);

    settings.beginGroup(group());

    for (int i = 0; i < KEY_COUNT; i++)
        settings.setValue(name[i], code[i]);
}

// Import from a gpspm-formatted file
bool KeysConfig::read(QString file)
{
    QFile inFile(file);
    if (!inFile.open(QIODevice::ReadOnly | QIODevice::Text))
        return false;

    QTextStream in(&inFile);
    for (int i = 0; i < KEY_COUNT && !in.atEnd(); i++)
        in >> code[i];

    return true;
}

// Export to a gpspm-formatted file
void KeysConfig::write(QString file)
{
    QFile outFile(file);
    outFile.open(QIODevice::WriteOnly | QIODevice::Text | QIODevice::Truncate);

    QTextStream out(&outFile);
    for (int i = 0; i < KEY_COUNT; i++)
        out << code[i] << "\n";
}
