/* -*- Mode: C; tab-width: 2; indent-tabs-mode: t; c-basic-offset: 2 -*- */
/* vi: set ts=2 sw=2: */
/* gjiten.c

   GJITEN : A GTK+/GNOME BASED JAPANESE DICTIONARY

   Copyright (C) 1999 - 2005 Botond Botyanszki <boti@rocketmail.com>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published  by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdlib.h>
#include <string.h>
/*#include <libgnome/libgnome.h>
#include <libgnomeui/libgnomeui.h>*/
#include <gtk/gtk.h>
#include <glib/gi18n.h>
#include <popt.h>
#include <libosso.h>
#include <hildon/hildon-program.h>
#include <gconf/gconf-client.h>
#ifdef HAVE_INTTYPES_H
#include <inttypes.h>
#endif
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif

#include "error.h"
#include "constants.h"
#include "conf.h"
#include "dicfile.h"
#include "worddic.h"
#include "kanjidic.h"
#include "gjiten.h"
#include "dicutil.h"

GjitenApp *gjitenApp = NULL;

static void parse_an_arg(poptContext state,
												 enum poptCallbackReason reason,
												 const struct poptOption *opt,
												 const char *arg, void *data);

/***************** VARIABLES ***********************/

gchar *clipboard_text = NULL;

/* FIXME: GConfEnumPair also in kanjidic.h */
gchar *kanjidicstrg[] = { "kanji", "radical", "strokes", "reading", "korean", 
													"pinyin", "english", "bushu", "classic", "freq", "jouyou",
													"deroo", "skip", "fourc", "hindex", "nindex", "vindex",
													"iindex", "mnindex", "mpindex", "eindex", "kindex", 
													"lindex", "oindex", "cref", "missc", "unicode", "jisascii" };


enum {
  KANJIDIC_KEY        = -1,
  WORD_LOOKUP_KEY     = -2,
  KANJI_LOOKUP_KEY    = -3,
  CLIP_KANJI_KEY      = -4,
  CLIP_WORD_KEY       = -5,
  VERSION_KEY         = -6
};

/* Command line arguments via popt */
static struct poptOption arg_options [] = {
  { NULL, '\0', POPT_ARG_CALLBACK, (gpointer)parse_an_arg, 0,
    NULL, NULL },

  { "kanjidic", 'k', POPT_ARG_NONE, NULL, KANJIDIC_KEY,
    N_("Start up Kanjidic instead of Word dictionary"), NULL },

  { "word-lookup", 'w', POPT_ARG_STRING, NULL, WORD_LOOKUP_KEY,
    N_("Look up WORD in first dictionary"), N_("WORD") },

  { "kanji-lookup", 'l', POPT_ARG_STRING, NULL, KANJI_LOOKUP_KEY,
    N_("Look up KANJI in kanji dictionary"), N_("KANJI") },

  { "clip-kanji", 'c', POPT_ARG_NONE, NULL, CLIP_KANJI_KEY,
    N_("Look up kanji from clipboard"), NULL },

  { "clip-word", 'v', POPT_ARG_NONE, NULL, CLIP_WORD_KEY,
    N_("Look up word from clipboard"), NULL },

  { "version", '\0', POPT_ARG_NONE, NULL, VERSION_KEY,
    N_("Print version number and exit"), NULL },

  { NULL, '\0', 0, NULL, 0, NULL, NULL }

};

static void ignore_an_arg(poptContext state, enum poptCallbackReason reason,
 const struct poptOption *opt, const char *arg, void *data){
  /*Actual option parsing is handled by gtk_init().
    This option table is just here to generate the help message properly.*/
}

/*Unfortunately, translations for this table are in libbonobo (for dependency
   reasons, which we don't have).*/
static struct poptOption gtk_options [] = {

  { NULL, '\0', POPT_ARG_CALLBACK, &ignore_an_arg, 0,
    NULL, NULL },

  { "gdk-debug", '\0', POPT_ARG_STRING, NULL, 0,
    N_("Gdk debugging flags to set"), N_("FLAGS") },

  { "gdk-no-debug", '\0', POPT_ARG_STRING, NULL, 0,
    N_("Gdk debugging flags to unset"), N_("FLAGS") },

  { "display", '\0', POPT_ARG_STRING, NULL, 0,
    N_("X display to use"), N_("DISPLAY") },

  { "screen", '\0', POPT_ARG_INT, NULL, 0,
    N_("X screen to use"), N_("SCREEN") },

  { "sync", '\0', POPT_ARG_NONE, NULL, 0,
    N_("Make X calls synchronous"), NULL },

  { "name", '\0', POPT_ARG_STRING, NULL, 0,
    N_("Program name as used by the window manager"), N_("NAME") },

  { "class", '\0', POPT_ARG_STRING, NULL, 0,
    N_("Program class as used by the window manager"), N_("CLASS") },

  { "gtk-debug", '\0', POPT_ARG_STRING, NULL, 0,
    N_("Gtk+ debugging flags to unset"), N_("FLAGS") },

  { "g-fatal-warnings", '\0', POPT_ARG_NONE, NULL, 0,
    N_("Make all warnings fatal"), NULL },

  { "gtk-module", '\0', POPT_ARG_STRING, NULL, 0,
    N_("Load an additional Gtk module"), N_("MODULE") },

  { NULL, '\0', 0, NULL, 0, NULL, NULL }

};

static struct poptOption prog_options [] = {
  { NULL, '\0', POPT_ARG_INCLUDE_TABLE, poptHelpOptions, 0,
    N_("Help options"), NULL},

  { NULL, '\0', POPT_ARG_INCLUDE_TABLE, arg_options, 0,
    N_("Application options"), NULL},

  { NULL, '\0', POPT_ARG_INCLUDE_TABLE, gtk_options, 0,
    N_("GTK+"), NULL},

  { NULL, '\0', 0, NULL, 0, NULL, NULL }

};



/*================ Functions ===============================================*/

static void parse_an_arg(poptContext state,
												 enum poptCallbackReason reason,
												 const struct poptOption *opt,
												 const char *arg, void *data) {

  
  switch (opt->val) {
  case KANJIDIC_KEY:
    gjitenApp->conf->startkanjidic = TRUE;
    break;
  case WORD_LOOKUP_KEY:
    gjitenApp->conf->word_to_lookup = (gchar *)arg;
    break;
  case KANJI_LOOKUP_KEY:
    gjitenApp->conf->kanji_to_lookup = (gchar *)arg;
    break;
  case CLIP_KANJI_KEY:
    gjitenApp->conf->clip_kanji_lookup = TRUE;
    gjitenApp->conf->clip_word_lookup = FALSE;
    break;
  case CLIP_WORD_KEY:
    gjitenApp->conf->clip_word_lookup = TRUE;
    gjitenApp->conf->clip_kanji_lookup = FALSE;
    break;
  case VERSION_KEY:
    g_print ("gjiten %s\n", VERSION);
    exit(0);
    break;
  default:
    break;
  }
}

void gjiten_clear_entry_box(gpointer entrybox) {
  gtk_entry_set_text(GTK_ENTRY(entrybox), "");
}

void gjiten_exit() {
	if ((gjitenApp->worddic == NULL) && (gjitenApp->kanjidic == NULL)) {
		GJITEN_DEBUG("gjiten_exit()\n");
		conf_save_options(gjitenApp->conf);
		dicutil_unload_dic();
		dicfile_list_free(gjitenApp->conf->dicfile_list);
		conf_close_handler();
		gtk_main_quit();
	}
}

void gjiten_start_kanjipad() {
  FILE *kanjipad_binary;
  char *kpad_cmd;
	int32_t len;

  kanjipad_binary = fopen(gjitenApp->conf->kanjipad, "r");
  if (kanjipad_binary == NULL) {
    gjiten_print_error(_("Couldn't find the KanjiPad executable!\n"
												 "Please make sure you have it installed on your system \n"
												 "and set the correct path to it in the Preferences.\n"
												 "See the Documentation for more details about KanjiPad."));
  }
  else {
		len = strlen(gjitenApp->conf->kanjipad) + 2;
    fclose(kanjipad_binary);
    kpad_cmd = g_malloc(len);
    strncpy(kpad_cmd, gjitenApp->conf->kanjipad, len);
    strncat(kpad_cmd, "&", 1);
    system(kpad_cmd); /* FIXME */
    g_free(kpad_cmd);
  }
}

/*void gjiten_display_manual(GtkWidget *widget, void *data) {
  GtkWidget *window = data;
  GError *err = NULL;
  gboolean retval = FALSE;

  retval = gnome_help_display("gjiten.xml", NULL, &err);
  
  if (retval == FALSE) {
    GtkWidget *dialog;
    dialog = gtk_message_dialog_new(GTK_WINDOW(window),
																		GTK_DIALOG_DESTROY_WITH_PARENT,       
																		GTK_MESSAGE_ERROR,
																		GTK_BUTTONS_CLOSE,
																		_("Could not display help: %s"),
																		err->message);
    
    g_signal_connect(G_OBJECT(dialog), "response",
										 G_CALLBACK(gtk_widget_destroy),
										 NULL);
    
    gtk_window_set_resizable(GTK_WINDOW(dialog), FALSE);
    
    gtk_widget_show(dialog);
    
    g_error_free(err);
  }
}*/

static void action_fullscreen(GtkToggleAction *_action,
 GtkWindow *_window){
  if(gtk_toggle_action_get_active(_action))gtk_window_fullscreen(_window);
  else gtk_window_unfullscreen(_window);
}

static const gchar *fullscreen_ui_desc=
 "<ui>\
  <menubar name='MainMenu'>\
  <separator/>\
  <menuitem action='Fullscreen'/>\
  </menubar>\
  <toolbar name='Toolbar'>\
  <toolitem action='Fullscreen'/>\
  </toolbar>\
  </ui>";

static GtkToggleActionEntry fullscreen_toggle_actions[]={
  {"Fullscreen","general_fullsize",N_("Fullscreen"),"F6",NULL,
   G_CALLBACK(action_fullscreen),FALSE},
};

void gjiten_add_fullscreen_action(GtkUIManager *_manager,GtkWindow *_window){
  GtkActionGroup *actions;
  actions=gtk_action_group_new("fullscreen");
  gtk_action_group_set_translation_domain(actions,GETTEXT_PACKAGE);
  gtk_action_group_add_toggle_actions(actions,fullscreen_toggle_actions,
   G_N_ELEMENTS(fullscreen_toggle_actions),_window);
  gtk_ui_manager_add_ui_from_string(_manager,fullscreen_ui_desc,-1,NULL);
  gtk_ui_manager_insert_action_group(_manager,actions,0);
  if(!gdk_net_wm_supports(gdk_atom_intern("_NET_WM_STATE_FULLSCREEN",FALSE))){
    GtkAction *action;
    action=gtk_action_group_get_action(actions,"Fullscreen");
    gtk_action_set_sensitive(action,FALSE);
  }
}

static void about_response(GtkDialog *_about,gint _response,
 GtkWidget *_credits){
  switch(_response){
    case 1:gtk_window_present(GTK_WINDOW(_credits));break;
    default:gtk_widget_destroy(GTK_WIDGET(_about));break;
  }
}

static GtkWidget *about_create_label(void){
  GtkWidget *ret;
  ret=gtk_label_new(NULL);
  gtk_label_set_selectable(GTK_LABEL(ret),TRUE);
  gtk_misc_set_alignment(GTK_MISC(ret),0.0,0.0);
  gtk_misc_set_padding(GTK_MISC(ret),8,8);
  gtk_label_set_justify(GTK_LABEL(ret),GTK_JUSTIFY_LEFT);
  return ret;
}

static void about_add_author_pane(GtkWidget *_notebook,const gchar **_authors,
 const gchar *_label){
  GtkWidget *label;
  GtkWidget *sw;
  if(_authors!=NULL&&_authors[0]!=NULL){
    GString *string;
    int      i;
    label=about_create_label();
    sw=gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
     GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(sw),label);
    gtk_viewport_set_shadow_type(
     GTK_VIEWPORT(GTK_BIN(sw)->child),GTK_SHADOW_NONE);
    gtk_notebook_append_page(GTK_NOTEBOOK(_notebook),sw,
     gtk_label_new(_label));
    string=g_string_new(NULL);
    for(i=0;_authors[i]!=NULL;i++){
      gchar *tmp;
      tmp=g_markup_escape_text(_authors[i],-1);
      g_string_append(string,tmp);
      if(_authors[i+1]!=NULL)g_string_append_c(string,'\n');
      g_free(tmp);
    }
    gtk_label_set_markup(GTK_LABEL(label),string->str);
    g_string_free(string,TRUE);
  }
}

static GtkWidget *about_new(const gchar *_name,const gchar *_version,
 const gchar *_copyright,const gchar *_comments,const gchar **_authors,
 const gchar **_documenters,const gchar *_translator_credits,
 GdkPixbuf *_logo_pixbuf){
  GtkWidget *ret;
  GtkWidget *vbox;
  GtkWidget *logo_image;
  GtkWidget *name_label;
  GtkWidget *comments_label;
  GtkWidget *copyright_label;
  GtkWidget *image;
  GtkWidget *label;
  GtkWidget *hbox;
  GtkWidget *alignment;
  GtkWidget *button;
  GtkWidget *credits;
  GtkWidget *notebook;
  gchar     *tmp;
  gchar     *tmpf;
  ret=gtk_dialog_new();
  if(_name==NULL)_name="";
  tmp=g_strdup_printf(_("About %s"),_name);
  gtk_window_set_title(GTK_WINDOW(ret),tmp);
  g_free(tmp);
  gtk_dialog_set_has_separator(GTK_DIALOG(ret),FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(ret),5);
  gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(ret)->vbox),5);
  vbox=gtk_vbox_new(FALSE,8);
  gtk_container_set_border_width(GTK_CONTAINER(vbox),5);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(ret)->vbox),vbox,TRUE,TRUE,0);
  logo_image=gtk_image_new();
  gtk_box_pack_start(GTK_BOX(vbox),logo_image,FALSE,FALSE,0);
  if(_logo_pixbuf!=NULL){
    gtk_image_set_from_pixbuf(GTK_IMAGE(logo_image),_logo_pixbuf);
  }
  /*else use gnome_about_logo.png*/
  name_label=gtk_label_new(NULL);
  gtk_label_set_selectable(GTK_LABEL(name_label),TRUE);
  gtk_label_set_justify(GTK_LABEL(name_label),GTK_JUSTIFY_CENTER);
  gtk_box_pack_start(GTK_BOX(vbox),name_label,FALSE,FALSE,0);
  tmp=g_markup_escape_text(_name,-1);
  if(_version!=NULL){
    gchar *tmpv;
    tmpv=g_markup_escape_text(_version,-1);
    tmpf=g_strdup_printf(
     "<span size=\"xx-large\" weight=\"bold\">%s %s</span>",tmp,tmpv);
    g_free(tmpv);
  }
  else{
    tmpf=g_strdup_printf(
     "<span size=\"xx-large\" weight=\"bold\">%s</span>",tmp);
  }
  g_free(tmp);
  gtk_label_set_markup(GTK_LABEL(name_label),tmpf);
  g_free(tmpf);
  comments_label=gtk_label_new(_comments);
  gtk_label_set_selectable(GTK_LABEL(comments_label),TRUE);
  gtk_label_set_justify(GTK_LABEL(comments_label),GTK_JUSTIFY_CENTER);
  gtk_label_set_line_wrap(GTK_LABEL(comments_label),TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),comments_label,FALSE,FALSE,0);
  copyright_label=gtk_label_new(NULL);
  gtk_label_set_selectable(GTK_LABEL(copyright_label),TRUE);
  gtk_label_set_justify(GTK_LABEL(copyright_label),GTK_JUSTIFY_CENTER);
  gtk_box_pack_start(GTK_BOX(vbox),copyright_label,FALSE,FALSE,0);
  if(_copyright!=NULL){
    tmp=g_markup_escape_text(_copyright,-1);
    tmpf=g_strdup_printf("<span size=\"small\">%s</span>",tmp);
    g_free(tmp);
    gtk_label_set_markup(GTK_LABEL(copyright_label),tmpf);
    g_free(tmpf);
  }
  gtk_widget_show_all(vbox);
  gtk_dialog_add_button(GTK_DIALOG(ret),GTK_STOCK_CLOSE,GTK_RESPONSE_CLOSE);
  gtk_dialog_set_default_response(GTK_DIALOG(ret),GTK_RESPONSE_CLOSE);
  image=gtk_image_new_from_stock(GTK_STOCK_ABOUT,GTK_ICON_SIZE_BUTTON);
  label=gtk_label_new_with_mnemonic(_("C_redits"));
  hbox=gtk_hbox_new(FALSE,2);
  gtk_box_pack_start(GTK_BOX(hbox),image,FALSE,FALSE,0);
  gtk_box_pack_end(GTK_BOX(hbox),label,FALSE,FALSE,0);
  alignment=gtk_alignment_new(0.5,0.5,0,0);
  gtk_container_add(GTK_CONTAINER(alignment),hbox);
  button=gtk_button_new();
  gtk_container_add(GTK_CONTAINER(button),alignment);
  gtk_widget_show_all(button);
  gtk_dialog_add_action_widget(GTK_DIALOG(ret),button,1);
  gtk_button_box_set_child_secondary(
   GTK_BUTTON_BOX(GTK_DIALOG(ret)->action_area),button,TRUE);
  gtk_window_set_resizable(GTK_WINDOW(ret),FALSE);
  credits=gtk_dialog_new_with_buttons(_("Credits"),
   GTK_WINDOW(ret),GTK_DIALOG_DESTROY_WITH_PARENT,
   GTK_STOCK_CLOSE,GTK_RESPONSE_CLOSE,NULL);
  gtk_window_set_default_size(GTK_WINDOW(credits),360,260);
  gtk_dialog_set_default_response(GTK_DIALOG(credits),GTK_RESPONSE_CLOSE);
  gtk_dialog_set_has_separator(GTK_DIALOG(credits),FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(credits),5);
  gtk_box_set_spacing(GTK_BOX(GTK_DIALOG(credits)->vbox),2);
  notebook=gtk_notebook_new();
  gtk_container_set_border_width(GTK_CONTAINER(notebook),5);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(credits)->vbox),notebook,TRUE,TRUE,0);
  about_add_author_pane(notebook,_authors,_("Written by"));
  about_add_author_pane(notebook,_documenters,_("Documented by"));
  {
    const gchar *translators[2];
    translators[0]=_translator_credits;
    translators[1]=NULL;
    about_add_author_pane(notebook,translators,_("Translated by"));
  }
  gtk_widget_show_all(notebook);
  g_signal_connect(credits,"response",G_CALLBACK(gtk_widget_hide),NULL);
  gtk_window_set_destroy_with_parent(GTK_WINDOW(credits),TRUE);
  g_signal_connect(ret,"response",G_CALLBACK(about_response),credits);
  return ret;
}


void gjiten_create_about() {
  const gchar *authors[] = { "Botond Botyanszki", "  <boti@rocketmail.com>",
   "Ported to Maemo by:",
   "Timothy B. Terriberry", "  <tterribe@users.sourceforge.net>", NULL };
  const gchar *documenters[] = { NULL };
  const gchar *translator = _("TRANSLATORS! PUT YOUR NAME HERE");
  static GtkWidget *about = NULL;
  GdkPixbuf *pixbuf = NULL;

  if (about != NULL) {
    gdk_window_show(about->window);
    gdk_window_raise(about->window);
    return;
  }

  if (pixbuf != NULL) {
    GdkPixbuf* temp_pixbuf = NULL;
    
    temp_pixbuf = gdk_pixbuf_scale_simple(pixbuf, 
					   gdk_pixbuf_get_width(pixbuf) / 2,
					   gdk_pixbuf_get_height(pixbuf) / 2,
					   GDK_INTERP_HYPER);
    g_object_unref(pixbuf);
    pixbuf = temp_pixbuf;
  }

  pixbuf = gdk_pixbuf_new_from_file(PIXMAPDIR"/gjiten/gjiten-logo.png", NULL);

  if (strcmp(translator, "translated_by") == 0) translator = NULL;

  /*
    _("Released under the terms of the GNU GPL.\n"
    "Check out http://gjiten.sourceforge.net for updates"), 
  */
  about = about_new("gjiten", VERSION, "Copyright \xc2\xa9 1999-2005 Botond Botyanszki",
			  _("gjiten is a Japanese dictionary for Gnome"),
			  (const char **)authors,
			  (const char **)documenters,
			  (const char *)translator,
			  pixbuf);

  gtk_window_set_destroy_with_parent(GTK_WINDOW(about), TRUE);
  if (pixbuf != NULL)  g_object_unref (pixbuf);

  g_signal_connect(G_OBJECT(about), "destroy", G_CALLBACK(gtk_widget_destroyed), &about);
  gtk_widget_show(about);

}



/*********************** MAIN ***********************************/
int main (int argc, char **argv) {
  char *icon_path = PIXMAPDIR"/jiten.png";
  osso_context_t *osso_ctx;
 
	gjitenApp = g_new0(GjitenApp, 1);
  conf_init_handler();
	gjitenApp->conf = conf_load();

	if (gjitenApp->conf->envvar_override == TRUE) {
		if (gjitenApp->conf->gdk_use_xft == TRUE) putenv("GDK_USE_XFT=1");
		else putenv("GDK_USE_XFT=0");
		/* if (gjitenApp->conf->force_ja_JP == TRUE) putenv("LC_CTYPE=ja_JP"); */
		if (gjitenApp->conf->force_ja_JP == TRUE) putenv("LC_ALL=ja_JP");
		if (gjitenApp->conf->force_language_c == TRUE) putenv("LANGUAGE=C");
	}

  gtk_set_locale();  

#ifdef ENABLE_NLS
  bind_textdomain_codeset(PACKAGE, "UTF-8");
  bindtextdomain(PACKAGE, GJITEN_LOCALE_DIR);
  textdomain(PACKAGE);
#endif

  g_set_prgname("gjiten");
  {
    poptContext argctx;
    int         nextopt;
    argctx=poptGetContext("gjiten",argc,(const char **)argv,prog_options,0);
    while((nextopt=poptGetNextOpt(argctx))>0||nextopt==POPT_ERROR_BADOPT);
    if(nextopt!=-1){
      g_print("Error on option %s: %s.\n"
       "Run '%s --help' to see a full list "
       "of available command line options.\n",
       poptBadOption(argctx,0),poptStrerror(nextopt),argv[0]);
      exit(1);
    }
    poptFreeContext(argctx);
  }
  gtk_init(&argc,&argv);
  /*gnome_program_init("gjiten", VERSION, LIBGNOMEUI_MODULE, argc, argv, 
										 GNOME_PARAM_POPT_TABLE, arg_options, 
										 GNOME_PARAM_HUMAN_READABLE_NAME, _("gjiten"), 
										 GNOME_PARAM_APP_DATADIR, GNOMEDATADIR,
										 NULL);*/
  osso_ctx=osso_initialize("gjiten",VERSION,FALSE,NULL);
  g_set_application_name(_("Gjiten"));
  gjitenApp->program = hildon_program_get_instance();

  if (! g_file_test (icon_path, G_FILE_TEST_EXISTS)) {
    g_warning ("Could not find %s", icon_path);
  }
  else {
    gtk_window_set_default_icon_from_file(icon_path,NULL);
  }
  /*GtkActionEntry's must get their icons from a stock-id.
    Note that this is broken in the original gjiten.*/
  {
    GtkIconFactory *factory;
    factory=gtk_icon_factory_new();
    gtk_icon_factory_add(factory,"jiten",
     gtk_icon_set_new_from_pixbuf(
     gdk_pixbuf_new_from_file(PIXMAPDIR"/jiten.png",NULL)));
    gtk_icon_factory_add(factory,"gjiten-kanjidic",
     gtk_icon_set_new_from_pixbuf(
     gdk_pixbuf_new_from_file(PIXMAPDIR"/kanjidic.png",NULL)));
    gtk_icon_factory_add(factory,"gjiten-kanjipad",
     gtk_icon_set_new_from_pixbuf(
     gdk_pixbuf_new_from_file(PIXMAPDIR"/kanjipad.png",NULL)));
    gtk_icon_factory_add_default(factory);
  }

  /* the following is for clipboard lookup. */
  if ((gjitenApp->conf->clip_kanji_lookup == TRUE) || (gjitenApp->conf->clip_word_lookup == TRUE)) {
    if (gjitenApp->conf->clip_word_lookup) {
      gjitenApp->worddic = worddic_create();
      worddic_paste();
      on_text_entered();
    }
    else {
      if (gjitenApp->conf->clip_kanji_lookup){
				clipboard_text = gtk_clipboard_wait_for_text(gtk_clipboard_get(GDK_SELECTION_PRIMARY));
				/* validate	*/
				/* FIXME: try to convert EUC-JP to UTF8 if it's non-utf8 */
				if (g_utf8_validate(clipboard_text, -1, NULL) == FALSE) {
					gjiten_print_error(_("Unable to look up kanji: NON-UTF8 string received from clipboard!\n"));
					exit(0);
				}
				else if (isKanjiChar(g_utf8_get_char(clipboard_text)) == FALSE) {
					gjiten_print_error(_("Non-kanji string received from clipboard: %s\n"), clipboard_text);
					exit(0);
				}
				else {
					if (gjitenApp->kanjidic == NULL) kanjidic_create();
					print_kanjinfo(g_utf8_get_char(clipboard_text));
					/* if (clipboard_text !	= NULL) gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo_entry)->entry), clipboard_text); */
				}
      }
    }
  }

  if (argc > 1) {
    if (gjitenApp->conf->startkanjidic) {
      kanjidic_create();
    }
    else  
      if (gjitenApp->conf->word_to_lookup) {
				gjitenApp->worddic = worddic_create();
				gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(gjitenApp->worddic->combo_entry)->entry), gjitenApp->conf->word_to_lookup);
				on_text_entered();
      }
      else if (gjitenApp->conf->kanji_to_lookup != NULL) {
				if (g_utf8_validate(gjitenApp->conf->kanji_to_lookup, -1, NULL) == FALSE) {
					gjiten_print_error(_("Unable to look up kanji: NON-UTF8 string received from clipboard!\n"));
					exit(0); /* FIXME */
				}
				else if (isKanjiChar(g_utf8_get_char(gjitenApp->conf->kanji_to_lookup)) == FALSE) {
					gjiten_print_error(_("Non-kanji string received from clipboard: %s\n"), gjitenApp->conf->kanji_to_lookup);
					exit(0); /* FIXME */
				}
				else {
					if (gjitenApp->kanjidic == NULL) kanjidic_create();
					print_kanjinfo(g_utf8_get_char(gjitenApp->conf->kanji_to_lookup));
				}
			}
			else if (!gjitenApp->conf->clip_kanji_lookup && !gjitenApp->conf->clip_word_lookup) 
				gjitenApp->worddic = worddic_create();
  }
  else {
    gjitenApp->worddic = worddic_create();
  }
	gjiten_flush_errors();
  gtk_main();
  osso_deinitialize(osso_ctx);
  return 0;
}
